/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "$(appName).h"

typedef struct pagedata {
	Evas_Object *scroller;
	Evas_Object *page[IMAGE_MAX];
	int current_page;
	Ecore_Timer *slide_timer;
	Eina_Bool slide_right;
} pagedata_s;

typedef struct appdata {
	Evas_Object *win;
	Evas_Object *conform;
	Evas_Object *layout;
	Evas_Object *nf;
	pagedata_s *pd;
} appdata_s;

typedef struct itemdata {
	int index;
	const char *path;
	appdata_s *ad;
} itemdata_s;

static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static Eina_Bool
nf_it_pop_cb(void *data, Elm_Object_Item *it)
{
	appdata_s *ad = data;
	/* Don't pop the last view but hide window */
	elm_win_lower(ad->win);
	return EINA_FALSE;
}

static void
layout_resize_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	pagedata_s *pd = data;
	Evas_Coord w, h;
	int i;

	evas_object_geometry_get(obj, NULL, NULL, &w, &h);

	for(i = 0; i < IMAGE_MAX; i++) {
		evas_object_size_hint_min_set(pd->page[i], w, h);
	}

	elm_scroller_page_size_set(pd->scroller, w, h);
	elm_scroller_page_show(pd->scroller, pd->current_page, 0);
}

static void
layout_del_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	pagedata_s *pd = data;

	if (pd) {
		if (pd->slide_timer) {
			ecore_timer_del(pd->slide_timer);
			pd->slide_timer = NULL;
		}

		free(pd);
	}
}

static void
scroll_cb(void *data, Evas_Object *scroller, void *event_info)
{
	pagedata_s *pd = data;
	int page_no;

	elm_scroller_current_page_get(pd->scroller, &page_no, NULL);

	if(pd->current_page != page_no)
		pd->current_page = page_no;
}

static Evas_Object*
create_page(Evas_Object *parent, int page_num, appdata_s *ad)
{
	Evas_Object *layout, *box, *page_layout, *img;
	char buf[PATH_MAX];
	int i;

	pagedata_s *pd = calloc(1, sizeof(pagedata_s));
	pd->slide_right = EINA_TRUE;
	ad->pd = pd;

	/* Create Layout */
	layout = elm_layout_add(parent);
	elm_layout_theme_set(layout, "layout", "application", "default");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_show(layout);

	evas_object_event_callback_add(layout, EVAS_CALLBACK_RESIZE, layout_resize_cb, pd);
	evas_object_event_callback_add(layout, EVAS_CALLBACK_DEL, layout_del_cb, pd);

	/* Create Scroller */
	pd->scroller = elm_scroller_add(layout);
	elm_scroller_loop_set(pd->scroller, EINA_FALSE, EINA_FALSE);
	evas_object_size_hint_weight_set(pd->scroller, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(pd->scroller, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_scroller_policy_set(pd->scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_OFF);
	elm_scroller_page_scroll_limit_set(pd->scroller, 1, 0);
	elm_object_scroll_lock_y_set(pd->scroller, EINA_TRUE);
	elm_object_part_content_set(layout, "elm.swallow.content", pd->scroller);

	evas_object_smart_callback_add(pd->scroller, "scroll", scroll_cb, pd);

	/* Create Box */
	box = elm_box_add(pd->scroller);
	elm_box_horizontal_set(box, EINA_TRUE);
	elm_object_content_set(pd->scroller, box);
	evas_object_show(box);

	/* Create Pages */
	for(i = 0; i < IMAGE_MAX; i++) {
		page_layout = elm_layout_add(box);
		elm_layout_theme_set(page_layout, "layout", "application", "default");
		evas_object_size_hint_weight_set(page_layout, 0, 0);
		evas_object_size_hint_align_set(page_layout, 0, EVAS_HINT_FILL);
		evas_object_show(page_layout);

		img = elm_image_add(page_layout);
		snprintf(buf, sizeof(buf), "%s/%d.jpg", ICON_DIR, i);
		elm_image_file_set(img, buf, NULL);
		evas_object_show(img);
		pd->page[i] = img;

		elm_object_part_content_set(page_layout, "elm.swallow.content", img);

		elm_box_pack_end(box, page_layout);
	}
	pd->current_page = page_num;
	elm_scroller_page_show(pd->scroller, pd->current_page, 0);
	return layout;
}

static Eina_Bool
slide_cb(void *data)
{
	pagedata_s *pd = data;
	int page_no;

	elm_scroller_current_page_get(pd->scroller, &page_no, NULL);

	if (page_no == 0)
		pd->slide_right = EINA_TRUE;

	if (page_no == IMAGE_MAX -1)
		pd->slide_right = EINA_FALSE;

	if (pd->slide_right)
		elm_scroller_page_bring_in(pd->scroller, page_no + 1, 0);
	else
		elm_scroller_page_bring_in(pd->scroller, page_no - 1, 0);

	return ECORE_CALLBACK_RENEW;
}

static void
btn_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	pagedata_s *pd = ad->pd;
	const char *btn_text;

	btn_text = elm_object_text_get(obj);

	if (pd->slide_timer) {
		ecore_timer_del(pd->slide_timer);
		pd->slide_timer = NULL;
		elm_object_text_set(obj, "Slide Start");
	}
	else {
		pd->slide_timer = ecore_timer_add(SLIDE_INTERVAL, slide_cb, pd);
		elm_object_text_set(obj, "Slide Stop");
	}
}

static void
gengrid_it_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *layout, *btn;
	itemdata_s *id = data;
	appdata_s *ad = id->ad;
	Evas_Object *nf = ad->nf;

	layout = create_page(nf, id->index, ad);
	elm_naviframe_item_push(nf, "Image Viewer", NULL, NULL, layout, NULL);

	btn = elm_button_add(nf);
	elm_object_text_set(btn, "Slide Start");
	elm_object_part_content_set(nf, "toolbar", btn);
	evas_object_show(btn);

	evas_object_smart_callback_add(btn, "clicked", btn_clicked_cb, ad);
}

static Evas_Object*
gengrid_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	itemdata_s *id = data;

	if (!strcmp(part, "elm.swallow.icon")) {
		Evas_Object *img = elm_image_add(obj);

		elm_image_file_set(img, id->path, NULL);
		elm_image_aspect_fixed_set(img, EINA_FALSE);
		elm_image_preload_disabled_set(img, EINA_FALSE);
		evas_object_show(img);

		return img;
	}

	return NULL;
}

static Evas_Object*
create_gengrid(appdata_s *ad)
{
	Elm_Gengrid_Item_Class *gic;
	Evas_Object *gengrid;
	char buf[PATH_MAX];
	int i;

	gengrid = elm_gengrid_add(ad->nf);
	evas_object_size_hint_weight_set(gengrid, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(gengrid, EVAS_HINT_FILL, EVAS_HINT_EXPAND);
	elm_gengrid_item_size_set(gengrid, ELM_SCALE_SIZE(110), ELM_SCALE_SIZE(110));
	elm_gengrid_align_set(gengrid, 0.5, 0.1);

	gic = elm_gengrid_item_class_new();
	gic->item_style = "default";
	gic->func.text_get = NULL;
	gic->func.content_get = gengrid_content_get_cb;
	gic->func.state_get = NULL;
	gic->func.del = NULL;

	for (i = 0; i < IMAGE_MAX; i++) {
		itemdata_s *id = calloc(sizeof(itemdata_s), 1);
		snprintf(buf, sizeof(buf), "%s/%d.jpg", ICON_DIR, i);
		id->index = i;
		id->path = eina_stringshare_add(buf);
		id->ad = ad;
		elm_gengrid_item_append(gengrid, gic, id, gengrid_it_cb, id);
	}
	evas_object_show(gengrid);

	return gengrid;
}

static Evas_Object*
create_main_view(appdata_s *ad)
{
	Elm_Object_Item *nf_it;
	Evas_Object *gengrid;

	ad->nf = elm_naviframe_add(ad->conform);
	eext_object_event_callback_add(ad->nf, EEXT_CALLBACK_BACK, eext_naviframe_back_cb, ad);

	gengrid = create_gengrid(ad);
	nf_it = elm_naviframe_item_push(ad->nf, "Gallery", NULL, NULL, gengrid, NULL);
	elm_naviframe_item_pop_cb_set(nf_it, nf_it_pop_cb, ad);

	return ad->nf;
}

static void
create_base_gui(appdata_s *ad)
{
	Evas_Object *bg;

	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_conformant_set(ad->win, EINA_TRUE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	if (elm_win_wm_rotation_supported_get(ad->win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);

	/* Conformant */
	ad->conform = elm_conformant_add(ad->win);
	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_OPAQUE);
	evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(ad->win, ad->conform);
	evas_object_show(ad->conform);

	/* Indicator BG */
	bg = elm_bg_add(ad->conform);
	elm_object_style_set(bg, "indicator/headerbg");
	elm_object_part_content_set(ad->conform, "elm.swallow.indicator_bg", bg);
	evas_object_show(bg);

	/* Naviframe */
	ad->nf = create_main_view(ad);
	elm_object_content_set(ad->conform, ad->nf);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
	   Initialize UI resources and application's data
	   If this function returns true, the main loop of application starts
	   If this function returns false, the application is terminated */
	appdata_s *ad = data;

	elm_app_base_scale_set(1.8);
	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control , void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}
