/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global $, App, localStorage, console */

(function strict() {
    'use strict';

    /**
     * Constructs NFCCard object.
     * @param {NFCControl} nfc
     */
    App.NFCCard = function nfc_NFCCard(nfc) {
        this.nfc = nfc;
    };

    App.NFCCard.prototype = {

        /**
         * Performs error action when NFC tag is attached to NFC target
         * for reading data.
         * @param {error} e
         */
        readMessageErr: function nfc_card_readMessageErr(e) {
            console.error('Read error! ' + e.message, e);
            this.nfc.timeExpired('Read error! ' + e.message);
        },

        /**
         * Performs success action when NFC tag is attached to NFC target
         * for reading data.
         * @param {NFCTag} tag
         */
        sucTagReadAttach: function nfc_card_sucTagReadAttach(tag) {
            try {
                if (!tag.isSupportedNDEF) {
                    throw {
                        message: 'This tag doesn\'t support NDEF'
                    };
                }
                tag.readNDEF(
                    this.nfc.readMessage.bind(this.nfc),
                    this.readMessageErr.bind(this)
                );
            } catch (e) {
                this.readMessageErr(e);
            }
        },

        /**
         * Registers a callback function to invoke when an NFC tag is detected.
         */
        setTagDetectRead: function nfc_card_setTagDetectRead() {
            try {
                this.nfc.nfcAdapter.unsetTagListener();
                this.nfc.nfcAdapter.setTagListener({
                    onattach: this.sucTagReadAttach.bind(this),
                    ondetach: this.nfc.sucDetach.bind(this.nfc)
                });
            } catch (error) {
                this.readMessageErr(error);
            }
        },

        /**
         * Performs action when all data has been saved to NFC tag.
         */
        sucSend: function nfc_card_sucSend() {
            this.nfc.timeExpired('Send success!');
        },

        /**
         * Performs action when data has not been saved to NFC tag.
         * @param {event} e
         */
        errSend: function nfc_card_errSend(e) {
            console.warn('errSend', e);
            this.nfc.timeExpired('Write error! ' + e.message);
        },

        /**
         * Performs success action when NFC tag is attached for write.
         * @param {NFCTag} tag
         */
        sucTagWriteAttach: function nfc_card_sucTagWriteAttach(tag) {
            var newMessage = null,
                fullContact = '';

            try {
                fullContact = this.nfc.prepareForNFC(localStorage);
                newMessage = this.nfc.phoneNumber2NDEF(fullContact);
                if (!tag.isSupportedNDEF) {
                    throw {
                        message: 'This tag doesn\'t support NDEF'
                    };
                }
                tag.writeNDEF(
                    newMessage,
                    this.sucSend.bind(this),
                    this.errSend.bind(this)
                );
            } catch (e) {
                this.errSend(e);
            }
        },

        /**
         * Registers a callback function to invoke when an NFC tag is detected.
         */
        setTagDetectWrite: function nfc_card_setTagDetectWrite() {
            var suc = {
                onattach: this.sucTagWriteAttach.bind(this),
                ondetach: this.nfc.sucDetach.bind(this.nfc)
            };

            try {
                this.nfc.nfcAdapter.unsetTagListener();
                this.nfc.nfcAdapter.setTagListener(suc);
            } catch (error) {
                console.error(error);
            }
        }

    };

}());
