/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global App, console, localStorage */

(function strict() {
    'use strict';

    /**
     * Constructs NFCPeer object.
     * @param {NFCControl} nfc
     */
    App.NFCPeer = function NFCPeer(nfc) {
        this.nfc = nfc;
    };

    App.NFCPeer.prototype = {

        /**
         * Performs success action when NFC message has been sent.
         */
        sucSendToTarget: function nfc_peer_sucSendToTarget() {
            return;
        },

        /**
         * Performs error action when NFC message has not been sent.
         * @param {event} e
         */
        errSendToTarget: function nfc_peer_errSendToTarget(e) {
            this.nfc.timeExpired('Send problem! ' + e.message);
        },

        /**
         * Registers a callback function to be invoked
         * when an NDEF message is received
         * from the connected NFC peer-to-peer target.
         */
        setReceiveFromTarget: function nfc_peer_setReceiveFromTarget() {
            try {
                if (!this.nfc.nfcTarget) {
                    console.warn('app.nfc.nfcTarget not set');
                    return;
                }
                this.nfc.nfcTarget.unsetReceiveNDEFListener();
                this.nfc.nfcTarget.setReceiveNDEFListener(
                    this.nfc.readMessage.bind(this.nfc)
                );
            } catch (error) {
                console.error('setReceiveFromTarget error: ' + error.message);
            }
        },

        /**
         * Performs success action when NFC target is attached
         * to another NFC target.
         * @param {object} target
         */
        sucTargetAttach: function nfc_peer_sucTargetAttach(target) {
            var newMessage = null,
                fullContact = this.nfc.prepareForNFC(localStorage);
            this.nfc.nfcTarget = target;
            this.setReceiveFromTarget();

            try {
                newMessage = this.nfc.phoneNumber2NDEF(fullContact);
                target.sendNDEF(
                    newMessage,
                    this.sucSendToTarget.bind(this),
                    this.errSendToTarget.bind(this)
                );
            } catch (e) {
                console.error('NDEFMessage problem: ' + e.message);
            }
        },

        /**
         * Registers a callback function to be invoked
         * when NFC peer-to-peer target is detected.
         */
        setTargetDetect: function nfc_peer_setTargetDetect() {
            var successCallbacks = {
                onattach: this.sucTargetAttach.bind(this),
                ondetach: this.nfc.sucDetach.bind(this.nfc)
            };

            try {
                this.nfc.nfcAdapter.unsetPeerListener();
                this.nfc.nfcAdapter.setPeerListener(successCallbacks);
            } catch (error) {
                console.error(error.message);
            }
        }
    };

}());
