/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global define, tizen*/

/**
 * Module duration.
 */

define({
    name: 'helpers/duration',
    def: function helpersDuration() {
        'use strict';

        /**
         * How many minutes are in one day.
         * @type {number}
         */
        var dayMinutes = 1440,
            /**
             * How many minutes are in one hour.
             * @type {number}
             */
            hourMinutes = 60,
            /**
             * Days duration unit.
             * @type {string}
             */
            dayUnit = 'DAYS',
            /**
             * Hours duration unit.
             * @type {string}
             */
            hourUnit = 'HOURS',
            /**
             * Minutes duration unit.
             * @type {string}
             */
            minuteUnit = 'MINS',
            /**
             * Label for day duration.
             * @type {string}
             */
            dayLabel = 'day before',
            /**
             * Label for days duration.
             * @type {string}
             */
            daysLabel = 'days before',
            /**
             * Label for hour duration.
             * @type {string}
             */
            hourLabel = 'hour before',
            /**
             * Label for hours duration.
             * @type {string}
             */
            hoursLabel = 'hours before',
            /**
             * Label for minute duration.
             * @type {string}
             */
            minuteLabel = 'minute before',
            /**
             * Label for minutes duration.
             * @type {string}
             */
            minutesLabel = 'minutes before';

        /**
         * Converts time duration to time in minutes.
         * @param {tizen.TimeDuration} duration Time duration
         * @returns {number}
         */
        function durationToMinutes(duration) {
            var minutes = duration.length;

            if (duration.unit === dayUnit) {
                minutes = minutes * dayMinutes;
            } else if (duration.unit === hourUnit) {
                minutes = minutes * hourMinutes;
            }
            return minutes;
        }

        /**
         * Converts time in minutes to time duration.
         * @param {number} minutes Time in minutes
         * @returns {tizen.TimeDuration}
         */
        function minutesToDuration(minutes) {
            if (minutes % dayMinutes === 0) {
                return new tizen.TimeDuration(minutes / dayMinutes, dayUnit);
            }
            if (minutes % hourMinutes === 0) {
                return new tizen.TimeDuration(minutes / hourMinutes, hourUnit);
            }
            return new tizen.TimeDuration(minutes, minuteUnit);
        }

        /**
         * Returns printable string from time in minutes.
         * @param {number} minutes Time in minutes
         * @returns {string}
         */
        function minutesToString(minutes) {
            var value = 0,
                label = '';

            if (minutes === -1) {
                label = 'Off';
            } else if (minutes === 0) {
                label = 'On time';
            } else {
                if (minutes % dayMinutes === 0) {
                    value = minutes / dayMinutes;
                    if (value === 1) {
                        label = dayLabel;
                    } else {
                        label = daysLabel;
                    }
                } else if (minutes % hourMinutes === 0) {
                    value = minutes / hourMinutes;
                    if (value === 1) {
                        label = hourLabel;
                    } else {
                        label = hoursLabel;
                    }
                } else {
                    value = minutes;
                    if (value === 1) {
                        label = minuteLabel;
                    } else {
                        label = minutesLabel;
                    }
                }
                return value + ' ' + label;
            }
            return label;
        }

        /**
         * Returns printable string from time duration.
         * @param {tizen.TimeDuration} duration Time duration
         * @returns {string}
         */
        function durationToString(duration) {
            return minutesToString(durationToMinutes(duration));
        }

        return {
            durationToMinutes: durationToMinutes,
            minutesToDuration: minutesToDuration,
            minutesToString: minutesToString,
            durationToString: durationToString
        };
    }
});
