/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global define, tau, tizen, console, setTimeout*/

/**
 * Module filter.
 */

define({
    name: 'models/filter',
    requires: [
        'views/popup'
    ],
    def: function modelsFilter(popup) {
        'use strict';

        /**
         * Handles error callback.
         * @param {error} error Error object
         */
        function errorCallback(error) {
            // Needed because error is invoked before page change
            setTimeout(function onTimeout() {
                var pageId = tau.engine.getRouter().activePage.id;
                popup.show(pageId, error.name + ': ' + error.message);
            }, 100);
            console.error('Calendar error: ', error);
        }

        /**
         * Creates an intersection of filters.
         * @param {AbstractFilter[]} arr
         * @returns {tizen.CompositeFilter}
         */
        function filtersIntersection(arr) {
            try {
                return new tizen.CompositeFilter('INTERSECTION', arr);
            } catch (error) {
                errorCallback(error);
            }
        }

        /**
         * Creates a filter for selecting normal events on the given day.
         * @param {Date} date
         * @return {tizen.CompositeFilter}
         */
        function getNormalEventsDateFilter(date) {
            try {
                var beginOfToday = new tizen.TZDate(date.getFullYear(),
                        date.getMonth(),
                        date.getDate(),
                        0, 0, 0, 0
                    ),
                    endOfToday = new tizen.TZDate(date.getFullYear(),
                        date.getMonth(),
                        date.getDate(),
                        23, 59, 59, 999
                    ),
                    startsBeforeEndOfToday = new tizen.AttributeRangeFilter(
                        'startDate',
                        null,
                        endOfToday
                    ),
                    endsAfterBeginOfToday = new tizen.AttributeRangeFilter(
                        'endDate',
                        beginOfToday,
                        null
                    );

                return filtersIntersection([
                    startsBeforeEndOfToday,
                    endsAfterBeginOfToday
                ]);
            } catch (error) {
                errorCallback(error);
            }
        }

        /**
         * Creates a filter for selecting all day events on the given day.
         * @param {Date} date
         * @return {tizen.CompositeFilter}
         */
        function getAllDayEventsDateFilter(date) {
            try {
                var beginOfToday = new tizen.TZDate(date.getFullYear(),
                        date.getMonth(),
                        date.getDate(),
                        0, 0, 0, 0, 'UTC'
                    ),
                    endOfToday = new tizen.TZDate(date.getFullYear(),
                        date.getMonth(),
                        date.getDate(),
                        23, 59, 59, 999, 'UTC'
                    ),
                    startsBeforeEndOfToday = new tizen.AttributeRangeFilter(
                        'startDate',
                        null,
                        endOfToday
                    ),
                    endsAfterBeginOfToday = new tizen.AttributeRangeFilter(
                        'endDate',
                        beginOfToday,
                        null
                    );

                return filtersIntersection([
                    startsBeforeEndOfToday,
                    endsAfterBeginOfToday
                ]);
            } catch (error) {
                errorCallback(error);
            }
        }

        return {
            getNormalEventsDateFilter: getNormalEventsDateFilter,
            getAllDayEventsDateFilter: getAllDayEventsDateFilter
        };
    }
});
