/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global define, document, history, tau*/

/**
 * Module alarm.
 */

define({
    name: 'views/alarm',
    requires: [
        'core/event'
    ],
    def: function viewsAlarm(e) {
        'use strict';

        /**
         * List of checkboxes on alarm page.
         * @type {NodeList}
         */
        var checkboxes = document.getElementsByName('alarm-choice'),
            /**
             * Number of checkboxes on alarm page.
             * @type {number}
             */
            checkboxesLen = checkboxes.length,
            /**
             * Custom alarm checkbox.
             * @type {HTMLElement}
             */
            customCheckbox = document.getElementById('alarm-custom'),
            /**
             * Custom alarm input.
             * @type {HTMLElement}
             */
            customDuration = document.getElementById('alarm-custom-duration');

        /**
         * Changes state of custom input depending on checked radio.
         */
        function onAlarmChanged() {
            if (customCheckbox.checked) {
                customDuration.disabled = false;
                customDuration.classList.remove('ui-disabled');
                customDuration.focus();
            } else {
                customDuration.disabled = true;
                customDuration.classList.add('ui-disabled');
            }
        }

        /**
         * Shows alarm page and selects proper checkbox.
         * @param {string} duration Alarm time before event in minutes
         */
        function show(duration) {
            var checkbox = null,
                isCustom = true,
                i = 0;

            tau.changePage('#alarm-page');

            for (i = 0; i < checkboxesLen; i += 1) {
                checkbox = checkboxes[i];
                if (checkbox.value === duration) {
                    checkbox.checked = true;
                    isCustom = false;
                } else {
                    checkbox.checked = false;
                }
                tau.widget.Checkboxradio(checkbox).refresh();
            }
            if (isCustom === true) {
                customCheckbox.checked = true;
                customDuration.value = duration;
                tau.widget.Checkboxradio(customCheckbox).refresh();
            } else {
                customDuration.value = '';
            }
            onAlarmChanged();
        }

        /**
         * Sends event with alarm duration value and changes page.
         */
        function onSaveClicked() {
            var i = 0,
                checkbox = null;

            for (i = 0; i < checkboxesLen; i += 1) {
                checkbox = checkboxes[i];
                if (checkbox.checked) {
                    if (checkbox.value === 'custom') {
                        e.fire('selected', {duration: customDuration.value});
                    } else {
                        e.fire('selected', {duration: checkbox.value});
                    }
                }
            }

            history.back();
        }

        /**
         * Shows previous page
         */
        function onCancelClicked() {
            history.back();
        }

        /**
         * Binds events.
         */
        function bindEvents() {
            var i = 0,
                saveBtn = document.getElementById('alarm-save'),
                cancelBtn = document.getElementById('alarm-cancel');

            saveBtn.addEventListener('click', onSaveClicked);
            cancelBtn.addEventListener('click', onCancelClicked);

            for (i = 0; i < checkboxesLen; i += 1) {
                checkboxes[i].addEventListener('change', onAlarmChanged);
            }
        }

        /**
         * Initializes module.
         */
        function init() {
            bindEvents();
        }

        return {
            init: init,
            show: show
        };
    }
});
