/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*jslint devel:true*/
/*global Config, Model, Ui, app, tizen */

var App = null;

(function strict() { // strict mode wrapper
    'use strict';

    /**
     * Creates a new application object.
     *
     * @class Application
     */
    App = function App() {
        return false;
    };

    App.prototype = {

        /**
         * Array with application dependencies.
         * @type {Array}
         */
        requires: ['js/app.config.js', 'js/app.model.js', 'js/app.alarm.js',
            'js/app.ui.js', 'js/app.ui.templateManager.js',
            'js/app.ui.templateManager.modifiers.js'],

        /**
         * Config object.
         * @type {Config}
         */
        config: null,

        /**
         * Model object.
         * @type {Model}
         */
        model: null,

        /**
         * Ui object.
         * @type {Ui}
         */
        ui: null,

        /**
         * Application control data key.
         * @type {string}
         */
        APP_CONTROL_DATA_KEY: 'http://tizen.org/appcontrol/data/alarm_id',

        /**
         * Application control operation URI.
         * @type {string}
         */
        APP_CONTROL_OPERATION_URI:
            'http://tizen.org/appcontrol/operation/exercise',

        /**
         * Initialisation function.
         */
        init: function init() {
            // instantiate the libs
            this.config = new Config();
            this.model = new Model();
            this.ui = new Ui();

            // initialise the modules
            this.model.init(this);
            this.ui.init(this, this.getRequestedAppControlData());

            return this;
        },

        /**
         * Returns this application id.
         * @return {number} application id
         */
        getId: function getId() {
            return tizen.application.getCurrentApplication().appInfo.id;
        },

        /**
         * Returns connected exercise id.
         * @return {string} exercise id or undefined
         */
        getRequestedAppControlData: function getRequestedAppControlData() {
            var reqAppControl = tizen
                .application
                .getCurrentApplication()
                .getRequestedAppControl(),
                data,
                len,
                exerciseId;

            if (reqAppControl) {
                data = reqAppControl.appControl.data;
                len = data.length - 1;

                while (len >= 0) {
                    if (data[len].key === this.APP_CONTROL_DATA_KEY) {
                        exerciseId = data[len].value[0];
                        break;
                    }
                    len -= 1;
                }

                return exerciseId;
            }
        },

        /**
         * Closes application.
         */
        exit: function exit() {
            tizen.application.getCurrentApplication().exit();
        },

        /**
         * Adds exercise to storage.
         * @param {Object} exercise
         * @param {Function} success callback
         * @param {Function} failure callback
         */
        addExercise: function addExercise(exercise, success, failure) {
            // if add was successful
            if (this.model.add(exercise)) {
                if (success instanceof Function) {
                    success();
                }
            } else { // if add fail
                console.log('problem with adding');
                if (failure instanceof Function) {
                    failure();
                }
            }
        },

        /**
         * Returns all stored exercises.
         * @return {Array} list of exercises
         */
        getAllExercises: function getAllExercises() {
            return this.model.getAll();
        },

        /**
         * Returns single exercise which match value in corresponding key.
         * @param {string} attr name
         * @param {*} value
         * @return {Object|undefined} exercise object
         */
        getExercise: function getExercise(attr, value) {
            return this.model.find(attr, value)[0];
        },

        /**
         * Removes exercise from storage.
         * @param {string} exerciseId
         * @param {Function} success callback
         * @param {Function} failure callback
         */
        removeExercise: function removeExercise(exerciseId, success, failure) {
            // if removed was successfully completed
            if (this.model.remove(exerciseId)) {
                if (success instanceof Function) {
                    success();
                }
            } else { // if there was problem with removing exercise
                console.log('problem with removing');
                if (failure instanceof Function) {
                    failure();
                }
            }
        }

    };
}());
