/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*jslint devel:true*/
/*global window, Alarm, app */

/**
 * @class Model
 */
var Model = function Model() {
    'use strict';
    return false;
};

(function strict() { // strict mode wrapper
    'use strict';
    Model.prototype = {

        /**
         * Initialization function.
         * @param {Object} app object
         */
        init: function Model_init(app) {
            var exercises = window.localStorage.getItem('exercises');

            this.app = app;
            this.exercises = exercises ? JSON.parse(exercises) : [];
            this.alarmHelper = new Alarm();
        },

        /**
         * Adds new exercise, saves it in local storage and set new alarm.
         * @param {Object} exercise Exercise object
         * @return {Object|undefined} exercise object
         */
        add: function Model_saveAlarm(exercise) {
            var alarmId = this.alarmHelper.add(exercise);

            if (alarmId) {
                exercise.id = alarmId;

                // add to session storage
                this.exercises.push(exercise);

                // add to localStorage
                this.updateStorage();
                return exercise;
            }
        },

        /**
         * Removes exercise with a given id.
         * @param {String} exerciseId
         * @return {Boolean} result of remove, true if removed otherwise false
         */
        remove: function Model_remove(exerciseId) {
            // find exercise to remove
            var exercise = this.find('id', exerciseId)[0], index;
            // if exercise remove connected alarm
            if (exercise) {
                if (this.alarmHelper.remove(exercise)) {
                    // if alarm removed update session store
                    index = this.exercises.indexOf(exercise);
                    this.exercises.splice(index, 1);
                    //update localStorage
                    this.updateStorage();
                    return true;
                }
            }
            // error removing
            return false;
        },

        /**
         * Finds list of exercises matching values with given attribute name.
         * @param {String} attr name
         * @param {*} value of attr
         * @return {Array} list of exercises
         */
        find: function Model_find(attr, value) {
            var result = this.exercises.filter(
                function (el) {
                    return el[attr] === value.toString();
                }
            );
            return result;
        },

        /**
         * Saves exercises in local storage.
         */
        updateStorage: function () {
            try {
                window.localStorage.setItem(
                    'exercises',
                    JSON.stringify(this.exercises)
                );
            } catch (e) {
                if (e.code === 22) {
                    //QuotaExceededError
                    app.ui.popup(
                        'Not enough memory to save data.' +
                            ' Please remove unnecessary files.'
                    );
                }
            }
        },

        /**
         * Returns array of all currently stored exercises.
         * @return {Array} list of exercises
         */
        getAll: function Model_getAll() {
            return this.exercises;
        }

    };
}());
