/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global tizen, console, App*/

(function strict() {
    'use strict';

    /**
     * App.Model constructor.
     * @constructor
     */
    App.Model = function Model() {
        this.lastFolderItems = [];
        this.lastId = -1;
    };

    App.Model.prototype = {

        /**
         * Returns directories.
         * @param {function} successCallback
         * @param {function} errorCallback
         */
        getFolders: function getFolders(successCallback, errorCallback) {
            try {
                tizen.content.getDirectories(successCallback, errorCallback);
            } catch (exc) {
                console.warn('getLocalMediaSource exc:' + exc.message);
            }
        },

        /**
         * Returns folders items.
         * @param {number} folderId
         * @param {string} mediaType
         * @param {Content[]} folders
         * @param {function} findItemsSuccess
         * @param {function} findItemsError
         */
        getFolderItems: function getFolderItems(folderId, mediaType, folders,
                findItemsSuccess, findItemsError) {
            var self = this,
                folder = null,
                filter = null;

            if (folderId !== null) {
                this.lastId = folderId;
            }
            if (this.lastId !== null) {
                folder = folders[this.lastId].id;
            }
            if (mediaType !== 'ALL') {
                filter = new tizen.AttributeFilter('type', 'EXACTLY',
                    mediaType);
            }

            function success(items) {
                self.lastFolderItems = items;
                findItemsSuccess(items);
            }
            try {
                tizen.content.find(success, findItemsError,
                    folder,
                    filter, new tizen.SortMode('title', 'ASC'));
            } catch (exc) {
                console.warn('tizen.content.find exception: ' + exc.message);
            }
        },

        /**
         * Returns details about item by specific id.
         * @param {number} id
         * @return {Array}
         */
        getDetailsItemById: function getDetailsItemById(id) {
            return this.lastFolderItems[id];
        },

        /**
         * Adds storage change listener.
         * @param {function} changeListener
         */
        addStorageChangeListener: function addChangeListener(changeListener) {
            try {
                tizen.systeminfo.addPropertyValueChangeListener('STORAGE',
                    changeListener);
            } catch (error) {
                console.error(
                    'Add storage change listener error: ' + error.message
                );
            }
        },

        /**
         * Adds content change listener.
         * @param {function} changeListener
         */
        addContentChangeListener: function addContentChangeListener(
            changeListener
        ) {
            try {
                tizen.content.setChangeListener({
                    oncontentadded: changeListener,
                    oncontentremoved: changeListener
                });
            } catch (error) {
                console.error(
                    'Add content change listener error: ' + error.message
                );
            }
        }
    };
}());
