/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global define, tizen, tau, document, setInterval, clearInterval*/

/**
 * Main module.
 */

define({
    name: 'views/main',
    requires: [
        'core/template',
        'helpers/capabilities',
        'helpers/parser',
        'models/systemInfo'
    ],
    def: function viewsMain(req) {
        'use strict';

        /**
         * Template module object.
         * @type {Object}
         */
        var template = req.core.template,

            /**
             * Capabilities module object.
             * @type {Object}
             */
            capabilities = req.helpers.capabilities,

            /**
             * Parser module object.
             * @type {Object}
             */
            parser = req.helpers.parser,

            /**
             * System info module object.
             * @type {Object}
             */
            systemInfo = req.models.systemInfo,

            /**
             * Property change listener id.
             * @type {number}
             */
            plId = 0;

        /**
         * Fills page with data and displays it.
         * @param {string} pageId Id of the page to display
         * @param {string} title Title of the page
         * @param {array} data Data to be displayed
         */
        function displayPage(pageId, title, data) {
            var pageTitle = document.getElementById(pageId + '-title'),
                pageList = document.getElementById(pageId + '-list'),
                content = null;

            if (pageId === 'packages') {
                content = template.get('dataListAnchor', {details: data});
            } else {
                content = template.get('dataList', {details: data});
            }

            pageTitle.innerHTML = title;
            pageList.innerHTML = content;

            tau.widget.Listview(pageList).refresh();
            if (document.querySelector('.ui-page-active').id !== pageId) {
                tau.changePage('#' + pageId);
            }
        }

        /**
         * Displays details list with battery information.
         * @param {Object} battery
         */
        function onBatteryInfo(battery) {
            var data = parser.parseBatteryInfo(battery);

            displayPage('details', 'Battery', data);
        }

        /**
         * Displays details list with cpu information.
         * @param {Object} cpu
         */
        function onCpuInfo(cpu) {
            var data = parser.parseCpuInfo(cpu);

            displayPage('details', 'CPU', data);
        }

        /**
         * Displays details list with orientation information.
         * @param {Object} orientation
         */
        function onOrientationInfo(orientation) {
            var data = parser.parseOrientationInfo(orientation);

            displayPage('details', 'Device orientation', data);
        }

        /**
         * Displays details list with display information.
         * @param {Object} display
         */
        function onDisplayInfo(display) {
            var data = parser.parseDisplayInfo(display);

            displayPage('details', 'Display', data);
        }

        /**
         * Displays details list with storage information.
         * @param {Object} storage
         */
        function onStorageInfo(storage) {
            var data = parser.parseStorageInfo(storage);

            displayPage('details', 'Storage', data);
        }

        /**
         * Displays device capabilities.
         * @param {array} capabilities
         */
        function onCapabilities(capabilities) {
            displayPage('details', 'Capabilities', capabilities);
        }

        /**
         * Displays installed packages list.
         * @param {array} packages
         */
        function onPackagesInfo(packages) {
            var data = parser.parsePackagesInfo(packages),
                title = 'Package list (' + packages.length + ')';

            displayPage('packages', title, data);
        }

        /**
         * Displays details list with selected package details.
         * @param {Object} details
         */
        function onPackageDetails(details) {
            var data = parser.parsePackageDetails(details);

            displayPage('details', details.name, data);
        }

        /**
         * Binds events.
         */
        function bindEvents() {
            var detailsPage = document.getElementById('details'),
                batteryButton = document.getElementById('battery-info'),
                cpuButton = document.getElementById('cpu-info'),
                orientationButton = document.getElementById('orientation-info'),
                displayButton = document.getElementById('display-info'),
                storageButton = document.getElementById('storage-info'),
                capabilityButton = document.getElementById('capability-info'),
                packageButton = document.getElementById('package-info'),
                packageList = document.getElementById('packages-list');

            detailsPage.addEventListener('pagehide', function onDetailsHide() {
                if (plId) {
                    systemInfo.removePropertyListener(plId);
                    plId = 0;
                }
            });

            batteryButton.addEventListener('click', function onBatteryButton() {
                systemInfo.getSystemProperty('BATTERY', onBatteryInfo);
                plId = systemInfo.addPropertyListener('BATTERY', onBatteryInfo);
            });

            cpuButton.addEventListener('click', function onCpuButton() {
                systemInfo.getSystemProperty('CPU', onCpuInfo);
                plId = systemInfo.addPropertyListener('CPU', onCpuInfo);
            });

            orientationButton.addEventListener('click',
                function onOrientationButton() {
                    systemInfo.getSystemProperty('DEVICE_ORIENTATION',
                        onOrientationInfo);
                    plId = systemInfo.addPropertyListener('DEVICE_ORIENTATION',
                        onOrientationInfo);
                }
            );

            displayButton.addEventListener('click', function onDisplayButton() {
                systemInfo.getSystemProperty('DISPLAY', onDisplayInfo);
                plId = systemInfo.addPropertyListener('DISPLAY', onDisplayInfo);
            });

            storageButton.addEventListener('click', function onStorageButton() {
                systemInfo.getSystemProperty('STORAGE', onStorageInfo);
                plId = systemInfo.addPropertyListener('STORAGE', onStorageInfo);
            });

            capabilityButton.addEventListener('click',
                function onCapabilityButton() {
                    capabilities.getCapabilities(onCapabilities);
                }
            );

            packageButton.addEventListener('click', function onPackageButton() {
                systemInfo.getPackagesInfo(onPackagesInfo);
            });

            packageList.addEventListener('click',
                function onPackageClick(event) {
                    var id = event.target.dataset.attribute;

                    systemInfo.getPackageDetails(id, onPackageDetails);
                }
            );
        }

        /**
         * Initializes module.
         */
        function init() {
            bindEvents();
        }

        return {
            init: init
        };
    }
});
