/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global define*/

/**
 * Parser module.
 */

define({
    name: 'helpers/parser',
    def: function helpersParser() {
        'use strict';

        /**
         * Parses battery info and puts it into array of properties.
         * @param {Object} battery Battery information object
         * @returns {array}
         */
        function parseBatteryInfo(battery) {
            var batteryLevel = Math.floor(battery.level * 100) + '%',
                batteryInfo = [
                    {text: 'Level', value: batteryLevel},
                    {text: 'Charging', value: battery.isCharging}
                ];

            return batteryInfo;
        }

        /**
         * Parses CPU info and puts it into array of properties.
         * @param {Object} cpu CPU information object
         * @returns {array}
         */
        function parseCpuInfo(cpu) {
            var cpuLoad = Math.floor(cpu.load.toFixed(2) * 100) + '%',
                cpuInfo = [
                    {text: 'Load', value: cpuLoad}
                ];

            return cpuInfo;
        }

        /**
         * Parses device orientation info and puts it into array of properties.
         * @param {Object} orientation Device orientation information object
         * @returns {array}
         */
        function parseOrientationInfo(orientation) {
            var orientationInfo = [
                {text: 'Orientation', value: orientation.status},
                {text: 'Auto rotation', value: orientation.isAutoRotation}
            ];

            return orientationInfo;
        }

        /**
         * Parses display info and puts it into array of properties.
         * @param {Object} display Display information object
         * @returns {array}
         */
        function parseDisplayInfo(display) {
            var resolutionWidth = display.resolutionWidth + ' px',
                resolutionHeight = display.resolutionHeight + ' px',
                dotsPerInchWidth = display.dotsPerInchWidth + ' dots',
                dotsPerInchHeight = display.dotsPerInchHeight + ' dots',
                physicalWidth = display.physicalWidth + ' mm',
                physicalHeight = display.physicalHeight + ' mm',
                brightness = Math.floor(display.brightness * 100) + '%',
                displayInfo = [
                    {text: 'Resolution', divider: true},
                    {text: 'Width', value: resolutionWidth},
                    {text: 'Height', value: resolutionHeight},
                    {text: 'Dots per inch', divider: true},
                    {text: 'Width', value: dotsPerInchWidth},
                    {text: 'Height', value: dotsPerInchHeight},
                    {text: 'Physical size', divider: true},
                    {text: 'Width', value: physicalWidth},
                    {text: 'Height', value: physicalHeight},
                    {text: 'Others', divider: true},
                    {text: 'Brightness', value: brightness}
                ];

            return displayInfo;
        }

        /**
         * Converts specified value from bytes to megabytes.
         * @param {number} size Size in bytes.
         * @returns {number} Size in megabytes.
         */
        function sizeInMB(size) {
            return Math.round(size / 1048576 * 100) / 100;
        }

        /**
         * Parses storage info and puts it into array of properties.
         * @param {Object} storage Storage information object
         * @returns {array}
         */
        function parseStorageInfo(storage) {
            var i = 0,
                units = storage.units,
                unitsLen = units.length,
                unit = null,
                capacity = '',
                available = '',
                storageInfo = [];

            for (i = 0; i < unitsLen; i += 1) {
                unit = units[i];
                capacity = sizeInMB(unit.capacity) + ' MB';
                available = sizeInMB(unit.availableCapacity) + ' MB';

                storageInfo.push(
                    {text: 'Storage ' + (i + 1), divider: true},
                    {text: 'Type', value: unit.type},
                    {text: 'Capacity', value: capacity},
                    {text: 'Available', value: available},
                    {text: 'Removable', value: unit.isRemoveable}
                );
            }

            return storageInfo;
        }

        /**
         * Parses packages info and puts it into array of properties.
         * @param {Object} packages Packages information object
         * @returns {array}
         */
        function parsePackagesInfo(packages) {
            var i = 0,
                packagesLen = packages.length,
                entry = null,
                packagesInfo = [];


            for (i = 0; i < packagesLen; i += 1) {
                entry = packages[i];

                packagesInfo.push(
                    {text: entry.name, value: entry.version,
                        attribute: entry.id}
                );
            }

            return packagesInfo;
        }

        /**
         * Parses package details and puts it into array of properties.
         * @param {Object} details Package details object
         * @returns {array}
         */
        function parsePackageDetails(details) {
            var totalSize = sizeInMB(details.totalSize) + ' MB',
                totalSizeLine = totalSize + '(' + details.totalSize + ' B)',
                dataSize = sizeInMB(details.dataSize) + ' MB',
                dataSizeLine = dataSize + '(' + details.dataSize + ' B)',
                packageDetails = [
                    {text: 'ID', value: details.id},
                    {text: 'Name', value: details.name || 'N/A'},
                    {text: 'Version', value: details.version || 'N/A'},
                    {text: 'Total size', value: totalSizeLine},
                    {text: 'Data size', value: dataSizeLine},
                    {text: 'Last modified', value: details.lastModified},
                    {text: 'Author', value: details.author || 'N/A'},
                    {text: 'Description', value: details.description || 'N/A'},
                    {text: 'App IDs', value: details.appIds.toString()}
                ];

            return packageDetails;
        }

        return {
            parseBatteryInfo: parseBatteryInfo,
            parseCpuInfo: parseCpuInfo,
            parseOrientationInfo: parseOrientationInfo,
            parseDisplayInfo: parseDisplayInfo,
            parseStorageInfo: parseStorageInfo,
            parsePackagesInfo: parsePackagesInfo,
            parsePackageDetails: parsePackageDetails
        };
    }
});
