/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global define, tizen, console*/

/**
 * Module system info
 */

define({
    name: 'models/systemInfo',
    def: function modelsSystemInfo() {
        'use strict';

        /**
         * Displays error message.
         * @param {Error} error
         */
        function onError(error) {
            console.error('Error: ', error.message);
        }

        /**
         * Requests installed packages information.
         * @param {function} onSuccess Success callback
         */
        function getPackagesInfo(onSuccess) {
            try {
                tizen.package.getPackagesInfo(onSuccess);
            } catch (error) {
                onError(error);
            }
        }

        /**
         * Requests installed package details.
         * @param {string} packageId Package id
         * @param {function} onSuccess Success callback
         */
        function getPackageDetails(packageId, onSuccess) {
            try {
                onSuccess(tizen.package.getPackageInfo(packageId));
            } catch (error) {
                onError(error);
            }
        }

        /**
         * Requests system property value.
         * @param {string} property Property identifier
         * @param {function} onSuccess Success callback
         */
        function getSystemProperty(property, onSuccess) {
            try {
                tizen.systeminfo.getPropertyValue(property,
                    onSuccess, onError);
            } catch (error) {
                onError(error);
            }
        }

        /**
         * Adds property value change listener.
         * @param {string} property Property identifier
         * @param {function} onChange Value change callback
         * @returns {number} Listener id
         */
        function addPropertyListener(property, onChange) {
            try {
                return tizen.systeminfo
                    .addPropertyValueChangeListener(property, onChange);
            } catch (error) {
                onError(error);
            }
        }

        /**
         * Removes current property value change listener.
         * @param {number} listenerId Listener id
         */
        function removePropertyListener(listenerId) {
            try {
                tizen.systeminfo.removePropertyValueChangeListener(listenerId);
            } catch (error) {
                onError(error);
            }
        }

        /**
         * Requests device capability value.
         * @param {string} capability Capability identifier
         * @returns {boolean}
         */
        function getCapability(capability) {
            var isSupported = false;

            try {
                isSupported = tizen.systeminfo.getCapability(capability);
            } catch (error) {
                onError(error);
            }

            return isSupported;
        }

        return {
            getPackagesInfo: getPackagesInfo,
            getPackageDetails: getPackageDetails,
            getSystemProperty: getSystemProperty,
            addPropertyListener: addPropertyListener,
            removePropertyListener: removePropertyListener,
            getCapability: getCapability
        };
    }
});
