/*
 * Samsung API
 * Copyright (c) 2009-2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "digital-watch.h"

Eina_Bool _set_info_time_and_date(void *data, watch_time_h watch_time)
{
	const char *DAY_OF_WEEK_STR[7] = { "Sun", "Mon", "Tue", "Wed", "Thur", "Fri", "Sat"};
	const char *MONTH_STR[12] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sept", "Oct", "Nov", "Dec"};

	appdata_s *ad = data;
	char time_str[BUFFER_LENGTH] = { 0 };
	char date_str[BUFFER_LENGTH] = { 0 };
	char ampm_str[BUFFER_LENGTH] = { 0 };
	char *time_str_p = NULL;
	int hour24 = 0;
	int minute = 0;
	int day_of_week = 0;
	int month = 0;
	int day = 0 ;

	retvm_if(!watch_time, EINA_FALSE, "watch_time is NULL");
	retvm_if(!ad, EINA_FALSE, "appdata is NULL");

	int ret = watch_time_get_current_time(&watch_time);
	if (ret != APP_ERROR_NONE) {
		_E("failed to get current time. err = %d", ret);
	}

	watch_time_get_hour24(watch_time, &hour24);
	watch_time_get_minute(watch_time, &minute);
	watch_time_get_day_of_week(watch_time, &day_of_week);
	watch_time_get_month(watch_time, &month);
	watch_time_get_day(watch_time, &day);

	/* time */
	if (hour24 >= 0 && hour24 < 12) {
		snprintf(ampm_str, sizeof(ampm_str), "%s", "AM");
	} else {
		snprintf(ampm_str, sizeof(ampm_str), "%s", "PM");
	}

	sprintf(time_str, "%d:%02d", hour24%12, minute);
	time_str_p = g_strdup_printf("%s<font_size=24><font=Tizen:style=Bold> %s</font></font_size>", time_str, ampm_str);

	elm_object_part_text_set(ad->layout, "textblock_time", time_str_p);

	if (!time_str_p) {
		g_free(time_str_p);
	}

	/* date */
	sprintf(date_str, "%s %d %s", DAY_OF_WEEK_STR[day_of_week-1], day, MONTH_STR[month-1]);
	elm_object_part_text_set(ad->layout, "default_text_date", date_str);

	return ECORE_CALLBACK_RENEW;
}

static void _destroy_base_gui(appdata_s *ad)
{
	ret_if(!ad);

	if (ad->image) evas_object_del(ad->image);
	if (ad->layout) evas_object_del(ad->layout);
	if (ad->conform) evas_object_del(ad->conform);
	if (ad->win) evas_object_del(ad->win);
}

static Evas_Object *_create_win(appdata_s *ad)
{
	Evas_Object *win = NULL;
	int ret = 0;

	ret = watch_app_get_elm_win(&win);
	if (ret != APP_ERROR_NONE) {
		_E("failed to get window. err = %d", ret);
		return NULL;
	}

	elm_win_title_set(win, "digital-watch");
	elm_win_borderless_set(win, EINA_TRUE);
	elm_win_alpha_set(win, EINA_FALSE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_HIDE);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_BG_TRANSPARENT);
	elm_win_prop_focus_skip_set(win, EINA_TRUE);
	elm_win_role_set(win, "no-effect");

	evas_object_resize(win, ad->w, ad->h);
	evas_object_show(win);
	return win;
}


static Evas_Object *_create_layout(appdata_s *ad)
{
	Evas_Object *layout = NULL;

	layout = elm_layout_add(ad->win);
	elm_layout_file_set(layout, EDJ_FILE_PATH, "layout_digital_watch");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_content_set(ad->win, ad->layout);

	evas_object_size_hint_align_set(layout, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_min_set(layout, ad->w, ad->h);
	evas_object_resize(layout, ad->w, ad->h);
	evas_object_show(layout);
	return layout;
}


static Evas_Object *_create_background(appdata_s *ad)
{
	Evas_Object *bg = NULL;
	Eina_Bool ret = EINA_TRUE;

	bg = elm_bg_add(ad->layout);
	ret = elm_bg_file_set(bg, IMG_DIR"bg.jpg", NULL);
	retvm_if (!ret, NULL,"Fail to set the background image" );

	elm_bg_option_set(bg, ELM_BG_OPTION_CENTER);
	elm_object_part_content_set(ad->layout, "bg_image", bg);
	evas_object_show(bg);
	return bg;
}

static void _create_base_gui(appdata_s *ad)
{
	ret_if(!ad);

	/* Window */
	ad->win = _create_win(ad);
	ret_if (!ad->win);

	/* Layout */
	ad->layout = _create_layout(ad);
	ret_if (!ad->layout);

	/* background image */
	ad->image = _create_background(ad);

	/* Show window after base gui is set up */


}

void lang_changed(app_event_info_h event_info, void* user_data)
{
	// Takes necessary actions when language setting is changed
	char *locale = NULL;

	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	ret_if(!locale);

	elm_language_set(locale);
	free(locale);

	return;
}

void region_changed(app_event_info_h event_info, void* user_data)
{
	// Takes necessary actions when region setting is changed
}

void low_battery(app_event_info_h event_info, void* user_data)
{
	// Takes necessary actions when system is running on low battery
	watch_app_exit();
}

void low_memory(app_event_info_h event_info, void* user_data)
{
	// Takes necessary actions when system is running on low memory
	watch_app_exit();
}

void device_orientation(app_event_info_h event_info, void* user_data)
{
	// Takes necessary actions when system is running on low memory
}

static bool app_create(int width, int height, void* user_data)
{
	/* Hook to take necessary actions before main event loop starts
	   Initialize UI resources and application's data */

	appdata_s *ad = user_data;
	app_event_handler_h handlers[5] = {NULL, };
	watch_time_h watch_time = NULL;

	// Register callbacks for each system event
	if (watch_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, lang_changed, NULL) != APP_ERROR_NONE) {
		 dlog_print(DLOG_ERROR, LOG_TAG, "watch_app_add_event_handler () is failed");
	}
	if (watch_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, region_changed, NULL) != APP_ERROR_NONE) {
		 dlog_print(DLOG_ERROR, LOG_TAG, "watch_app_add_event_handler () is failed");
	}
	if (watch_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, low_battery, NULL) != APP_ERROR_NONE) {
		 dlog_print(DLOG_ERROR, LOG_TAG, "watch_app_add_event_handler () is failed");
	}
	if (watch_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY,low_memory, NULL) != APP_ERROR_NONE) {
		 dlog_print(DLOG_ERROR, LOG_TAG, "watch_app_add_event_handler () is failed");
	}
	if (watch_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, device_orientation, NULL) != APP_ERROR_NONE) {
		 dlog_print(DLOG_ERROR, LOG_TAG, "watch_app_add_event_handler () is failed");
	}

	dlog_print(DLOG_ERROR, LOG_TAG, "app_create");

	ad->w = width ;
	ad->h = height;

	_create_base_gui(ad);

	_set_info_time_and_date(ad, watch_time);

	return true;
}

static void app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void app_terminate(void *data)
{
	/* Release all resources. */
	appdata_s *ad = data;
	ret_if(!ad);

	_destroy_base_gui(ad);
}

void app_time_tick(watch_time_h watch_time, void* user_data)
{
	appdata_s *ad = user_data;

	_set_info_time_and_date(ad, watch_time);
}

void app_ambient_tick(watch_time_h watch_time, void* user_data)
{
	appdata_s *ad = user_data;

	_set_info_time_and_date(ad, watch_time);
}

void app_ambient_changed(bool ambient_mode, void* user_data)
{
	if (ambient_mode) {
		// Prepare to enter the ambient mode
	}
	else {
		// Prepare to exit the ambient mode
	}
}



int main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	watch_app_lifecycle_callback_s event_callback = {0,};

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;
	event_callback.time_tick = app_time_tick;
	event_callback.ambient_tick = app_ambient_tick;
	event_callback.ambient_changed = app_ambient_changed;

	ret = watch_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		_E("watch_app_main() is failed. err = %d", ret);
	}

	return ret;
}
