/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var canvas = null,
    context = null,
    centerX = document.width / 2,
    centerY = document.width / 2,
    watchRadius = document.width / 2;

function drawCircle(x, y, radius, color) {
    context.beginPath();
    context.fillStyle = color;
    context.arc(x, y, radius, 0, 2 * Math.PI);
    context.fill();
    context.closePath();
}

function drawNeedle(angle, start, end, width, color) {
    var dxi = watchRadius * Math.cos(angle) * start,
        dyi = watchRadius * Math.sin(angle) * start,
        dxf = watchRadius * Math.cos(angle) * end,
        dyf = watchRadius * Math.sin(angle) * end;

    // Draw the needle
    context.beginPath();
    context.lineWidth = width;
    context.strokeStyle = color;
    context.moveTo(centerX + dxi, centerY + dyi);
    context.lineTo(centerX + dxf, centerY + dyf);
    context.stroke();
    context.closePath();
}

function drawWatchLayout() {
    var angle = 0,
        i = 1,
        j = 1,
        grd = null;

    // Clear canvas
    context.clearRect(0, 0, context.canvas.width, context.canvas.height);
    // Draw a background circle
    drawCircle(centerX, centerY, watchRadius, "#000000");
    grd = context.createLinearGradient(0, 0, watchRadius * 2, 0);
    grd.addColorStop(0, "#000000");
    grd.addColorStop(0.5, "#454545");
    grd.addColorStop(1, "#000000");
    context.fillStyle = grd;

    drawCircle(centerX, centerY, watchRadius * 0.945, grd);
    drawCircle(centerX, centerY, watchRadius * 0.7, "#000000");

    // Draw the divider
    // 60 unit divider
    for (i = 1; i <= 60; i++) {
        angle = (i - 15) * (Math.PI * 2) / 60;
        drawNeedle(angle, 0.95, 1.0, 1, "#c4c4c4");
    }

    // 12 unit line
    for (j = 1; j <= 12; j++) {
        angle = (j - 3) * (Math.PI * 2) / 12;
        drawNeedle(angle, 0.7, 0.945, 10, "#c4c4c4");
    }

    // write texts
    context.font = '25px Courier';
    context.textAlign = 'center';
    context.textBaseline = 'middle';
    context.fillStyle = '#999999';
    context.fillText('TIZEN WATCH', centerX, centerY - (watchRadius * 0.4));
}

function drawWatchContent(hour, minute, second, date) {
    // Draw a hour needle
    drawNeedle(Math.PI * (((hour + minute / 60) / 6) - 0.5), 0, 0.50, 3, "#454545");

    // Draw a minute needle
    drawNeedle(Math.PI * (((minute + second / 60) / 30) - 0.5), 0, 0.70, 3, "#454545");

    // Draw a minute/hour circle
    drawCircle(centerX, centerY, 8, "#454545");

    // Draw a second needle
    context.shadowOffsetX = 4;
    context.shadowOffsetY = 4;
    drawNeedle(Math.PI * ((second / 30) - 0.5), -0.10, 0.85, 1, "#c4c4c4");

    // Draw a second circle
    context.shadowOffsetX = 0;
    context.shadowOffsetY = 0;
    drawCircle(centerX, centerY, 5, "#c4c4c4");

    // Draw a center circle
    drawCircle(centerX, centerY, 2, "#454545");

    // Draw a text for date
    context.font = '25px Courier';
    context.textAlign = 'center';
    context.textBaseline = 'middle';
    context.fillStyle = '#999999';
    context.fillText(date, centerX, centerY + (watchRadius * 0.5));
}

function drawWatch() {
    var datetime = tizen.time.getCurrentDateTime(),
        hour = datetime.getHours(),
        minute = datetime.getMinutes(),
        second = datetime.getSeconds(),
        date = datetime.getDate();

    // Draw a basic layout
    drawWatchLayout();

    // Draw a watch content
    drawWatchContent(hour, minute, second, date);
}

window.onload = function() {
    canvas = document.querySelector("#myCanvas");
    context = canvas.getContext("2d");

    // Set a canvas square
    canvas.width = document.width;
    canvas.height = canvas.width;

    // Start main loop
    setInterval(function() {
        drawWatch();
    }, 1000);

    // add eventListener for tizenhwkey
    window.addEventListener('tizenhwkey', function(e) {
        if (e.keyName === 'back') {
            tizen.application.getCurrentApplication().exit();
        }
    });
};
