/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global app: true, model, ui, EquationInvalidFormatError, CalculationError,
  InfinityError, console, systeminfo, tizen*/
/*jslint evil: true*/

/**
 * app module
 */
app = {

    SEPARATOR: ',',

    init: function init() {
        'use strict';
        systeminfo.init();
        model.init();
        ui.init();
        this.refreshEquation();
    },

    refreshEquation: function refreshEquation() {
        'use strict';
        ui.showEquation(model.equation);
    },

    operatorKeys: {
        'add': '+',
        'sub': '-',
        'mul': '*',
        'div': '/'
    },

    processKey: function processKey(key) {
        'use strict';
        var keys = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
        if (ui.isResultVisible()) {
            if (
                Object.keys(this.operatorKeys).indexOf(key) === -1 &&
                    key !== 'del' &&
                    key !== 'eql' &&
                    key !== 'sign'
            ) {
                model.resetEquation();
            }
        }
        ui.clearResult();
        if (keys.indexOf(key) !== -1) {
            this.pushDigits(key);
        } else if (Object.keys(this.operatorKeys).indexOf(key) !== -1) {
            model.addOperator(this.operatorKeys[key]);
        } else if (key === 'dec') {
            model.addDecimal();
        } else if (key === 'del') {
            model.deleteLast();
        } else if (key === 'c') {
            model.resetEquation();
        } else if (key === 'sign') {
            model.changeSign();
        }
        if (key === 'eql' && !model.isEmpty()) {
            this.calculate();
        }
        this.refreshEquation();
    },

    pushDigits: function pushDigits(key) {
        'use strict';
        if (!model.addDigit(key)) {
            ui.showError('Only 10 digits available');
        }
    },

    addSeparators: function addSeparators(equationString) {
        'use strict';
        /*jslint unparam: true*/
        var negative = false;
        if (model.isNegativeComponent(equationString)) {
            equationString = RegExp.$2;
            negative = true;
        }
        equationString = equationString.replace(
            new RegExp('^(\\-?)([0-9]+)', 'g'),
            function replacer(match, sign, p1) {
                var p1array;
                p1 = p1.split('').reverse().join('');
                p1array = p1.match(new RegExp('.{1,3}', 'g'));
                p1 = p1array.join(app.SEPARATOR);
                p1 = p1.split('').reverse().join('');
                return sign + p1;
            }
        );
        return negative ? '(-' + equationString + ')' : equationString;
    },

    calculate: function calculate() {
        'use strict';
        var result = '';
        try {
            result = model.calculate();
            result = this.addSeparators(result);
            ui.showResult('=&nbsp;' + result);
        } catch (e) {
            if (e instanceof EquationInvalidFormatError) {
                ui.showResult('Wrong format');
            } else if (e instanceof CalculationError) {
                ui.showResult('Invalid operation');
            } else if (e instanceof InfinityError) {
                ui.showResult(
                    (e.positive ? '' : '&minus;') + '&infin;'
                );
            } else {
                ui.showError('Unknown error.');
                console.warn(e);
            }
        }
    },

    onLowBattery: function onLowBattery() {
        'use strict';
        tizen.application.getCurrentApplication().exit();
    }
};
