/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global document, ui: true, Image, app, tizen, window*/

/**
 * ui module
 */

ui = {
    LONGTAP_DELAY: 400,
    LONGTAP_REPEAT_INTERVAL: 20,
    longTapRepeatTimers: {},
    init: function init() {
        'use strict';
        this.resultElement = document.getElementById('result');
        this.resultValueElement = document.getElementById('resultvalue');
        this.equationElement = document.getElementById('equation');
        this.preloadImages();
        this.bindEvents();
        this.error = false;
        this.result = false;
        // disable multitouch
        document.body.addEventListener('touchstart', this.filterTap, true);
        document.body.addEventListener('touchend', this.filterTap, true);
    },

    imagesToPreload: [
        'img/key_0_press.png',
        'img/key_1_press.png',
        'img/key_2_press.png',
        'img/key_3_press.png',
        'img/key_4_press.png',
        'img/key_5_press.png',
        'img/key_6_press.png',
        'img/key_7_press.png',
        'img/key_8_press.png',
        'img/key_9_press.png',
        'img/key_c_press.png',
        'img/key_del_press.png',
        'img/key_div_press.png',
        'img/key_mul_press.png',
        'img/key_sub_press.png',
        'img/key_add_press.png',
        'img/key_dec_press.png',
        'img/key_eql_press.png'
    ],

    imagesCache: [],

    preloadImages: function preloadImages() {
        'use strict';
        var i, image;
        for (i = this.imagesToPreload.length - 1; i >= 0; i -= 1) {
            image = new Image();
            image.src = this.imagesToPreload[i];
            this.imagesCache.push(image);
        }
    },

    filterTap: function filterTap(ev) {
        'use strict';
        // disable multitouch
        if (ev.touches.length > 1) {
            ev.stopPropagation();
            ev.preventDefault();
        }
    },

    clearLongTapRepeatTimers: function clearLongTapRepeatTimers(key) {
        'use strict';
        if (this.longTapRepeatTimers['start' + key]) {
            window.clearTimeout(this.longTapRepeatTimers['start' + key]);
            this.longTapRepeatTimers['start' + key] = null;
        }

        if (this.longTapRepeatTimers['repeat' + key]) {
            window.clearInterval(this.longTapRepeatTimers['repeat' + key]);
            this.longTapRepeatTimers['repeat' + key] = null;
        }
    },

    bindEvents: function bindEvents() {
        'use strict';
        var numpad = document.getElementById('numpad'),
            self = this;

        numpad.addEventListener('touchstart', function onTouchStart(e) {
            var key = '',
                target = e.target,
                classList = target.classList;


            if (!classList.contains('key') && !classList.contains('longkey')) {
                return;
            }
            classList.add('press');
            key = target.id.replace(/key_/, '');
            if (classList.contains('long-tap-repeat')) {
                self.longTapRepeatTimers['start' + key] = window.setTimeout(
                    function longtapStart() {
                        app.processKey(key);
                        self.longTapRepeatTimers['repeat' + key] =
                            window.setInterval(
                                function longtapRepeat() {
                                    app.processKey(key);
                                },
                                self.LONGTAP_REPEAT_INTERVAL
                            );
                    },
                    self.LONGTAP_DELAY
                );
            } else {
                app.processKey(key);
            }

        });
        numpad.addEventListener('touchend', function onTouchEnd(e) {
            var key = '',
                target = e.target,
                classList = target.classList;
            if (!classList.contains('key') &&
                !classList.contains('longkey')) {
                return;
            }
            classList.remove('press');
            key = target.id.replace(/key_/, '');
            if (classList.contains('long-tap-repeat') &&
                    !self.longTapRepeatTimers['repeat' + key]) {
                if (e.touches.length === 0) {
                    app.processKey(key);
                }
            }
            self.clearLongTapRepeatTimers(key);
        });
        numpad.addEventListener('touchcancel', function onTouchCancel(e) {
            var key = '',
                target = e.target,
                classList = target.classList;
            if (!classList.contains('key') &&
                !classList.contains('longkey')) {
                return;
            }
            classList.remove('press');
            key = target.id.replace(/key_/, '');
            self.clearLongTapRepeatTimers(key);
        });
        document.addEventListener('tizenhwkey', function onTizenHwKey(e) {
            if (e.keyName === 'back') {
                tizen.application.getCurrentApplication().exit();
            }
        });
    },

    operatorDisplays: {
        '+': '+',
        '-': '&minus;',
        '*': '&times;',
        '/': '&divide;'
    },

    showEquation: function showEquation(equation) {
        'use strict';
        var e, element, elementText, span, equationElement, length;

        equationElement = document.getElementById('equation');

        equationElement.innerHTML = '';

        length = equation.length;
        for (e = 0; e < length; e += 1) {
            element = equation[e];
            span = document.createElement('span');
            elementText = element;
            if (Object.keys(this.operatorDisplays).indexOf(element) !== -1) {
                span.className = 'operator';
                elementText = this.operatorDisplays[element];
            } else {
                elementText = app.addSeparators(elementText);
            }
            elementText = elementText.replace(/-/g, '&minus;');
            span.innerHTML = elementText;
            equationElement.appendChild(span);
        }
    },

    /**
     * Shows string in result element.
     * @param {string} result
     * @private
     */
    show: function show(result) {
        'use strict';

        if (result === '') {
            return this.clear();
        }

        this.equationElement.classList.add('top');
        this.resultValueElement.innerHTML = result.replace(/-/g, '&minus;');
    },

    /**
     * Clears result element.
     * @private
     */
    clear: function clear() {
        'use strict';
        this.equationElement.classList.remove('top');
        this.resultValueElement.innerHTML = '';
    },

    /**
     * Shows error in result element.
     * @param {string} error
     */
    showError: function showError(error) {
        'use strict';
        this.show(error);
        this.error = true;
    },

    /**
     * Shows result in result element.
     * @param {string} result
     */
    showResult: function showResult(result) {
        'use strict';
        this.show(result);
        this.result = true;
    },

    /**
     * Clears result element and flags.
     */
    clearResult: function clearResult() {
        'use strict';
        this.clear();
        this.result = false;
        this.error = false;
    },

    /**
     * Returns true for result, false for empty result
     * @return {boolean}
     */
    isResultVisible: function isResultVisible() {
        'use strict';
        return this.result;
    }

};
