/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global define*/

/**
 * Date helpers module.
 * @namespace Calendar/helpers/date
 * @memberof Calendar/helpers
 */
define({
    name: 'helpers/date',
    def: function helpersDate() {
        'use strict';

        /**
         * Names of months.
         * @type {string[]}
         */
        var monthNames = [
                'January',
                'February',
                'March',
                'April',
                'May',
                'June',
                'July',
                'August',
                'September',
                'October',
                'November',
                'December'
            ];

        /**
         * Returns current datetime.
         * @memberof Calendar/helpers/date
         * @return {Date}
         */
        function getCurrentDateTime() {
            return new Date();
        }

        /**
         * Clones date.
         * @memberof Calendar/helpers/date
         * @param {Date} date
         * @return {Date}
         */
        function clone(date) {
            return new Date(date);
        }

        /**
         * Returns numbers of days in month.
         * @memberof Calendar/helpers/date
         * @param {Date} date
         * @return {number}
         */
        function daysInMonth(date) {
            var newDate = clone(date);

            newDate.setDate(1);
            newDate.setMonth(newDate.getMonth() + 1);

            return (
                new Date(
                    newDate.getFullYear(),
                    newDate.getMonth(),
                    0
                )
            ).getDate();
        }

        /**
         * Check if new date has in range day of old date.
         * @memberof Calendar/helpers/date
         * @param {Date} oldDate
         * @param {Date} newDate
         * @return {number}
         */
        function getCorrectDay(oldDate, newDate) {
            var day = 1;

            if (daysInMonth(newDate) < oldDate.getDate()) {
                day = daysInMonth(newDate);
            } else {
                day = oldDate.getDate();
            }

            return day;
        }

        /**
         * Returns next month.
         * @memberof Calendar/helpers/date
         * @param {Date} date
         * @return {Date}
         */
        function getNextMonth(date) {
            var newDate = clone(date);

            newDate.setDate(1);
            newDate.setMonth(date.getMonth() + 1);
            newDate.setDate(getCorrectDay(date, newDate));

            return newDate;
        }

        /**
         * Returns previous month.
         * @memberof Calendar/helpers/date
         * @param {Date} date
         * @return {Date}
         */
        function getPreviousMonth(date) {
            var newDate = clone(date);

            newDate.setDate(1);
            newDate.setMonth(date.getMonth() - 1);
            newDate.setDate(getCorrectDay(date, newDate));

            return newDate;
        }

        /**
         * Returns name of month.
         * @memberof Calendar/helpers/date
         * @param {object/number} date
         * @return {string}
         */
        function getMonthName(date) {
            var monthName = '';
            if (date instanceof Date) {
                monthName = monthNames[date.getMonth()];
            } else {
                monthName = monthNames[date];
            }
            return monthName;
        }

        /**
         * Returns position of first day of month.
         * @memberof Calendar/helpers/date
         * @param {Date} date
         * @return {number}
         */
        function getFirstDay(date) {
            var newDate = clone(date);

            newDate.setDate(1);
            return newDate.getDay();
        }

        /**
         * Prepares date to data format.
         * @memberof Calendar/helpers/date
         * @param {Date} data
         * @return {string}
         */
        function toDataString(data) {
            return data.getFullYear() + ',' +
                data.getMonth() + ',' + data.getDate();
        }

        return {
            clone: clone,
            daysInMonth: daysInMonth,
            getFirstDay: getFirstDay,
            getPreviousMonth: getPreviousMonth,
            getNextMonth: getNextMonth,
            getMonthName: getMonthName,
            getCurrentDateTime: getCurrentDateTime,
            toDataString: toDataString
        };
    }
});
