/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*global tizen, localStorage, console, Helpers, $ */

/**
 * @class Model
 */
function Model() {
    'use strict';
    this.init();
}

(function strict() {
    'use strict';

    Model.prototype = {

        /**
         * @type {Helpers}
         */
        helpers: null,

        /**
         * API module initialization
         */
        init: function Model_init() {
            console.log('Model_init');
            this.helpers = new Helpers();
        },

        /**
         * Returns URL of downloading file by its ID.
         * @param {number} downloadId
         * @returns {string}
         */
        getDownloadUrl: function Model_getDownloadUrl(downloadId) {
            var storageObjectString = localStorage[downloadId];

            if (storageObjectString) {
                return JSON.parse(storageObjectString).url;
            }
            return '';
        },

        /**
         * Handles download start event.
         * @param {function} callback
         * @param {number} downloadId
         * @param {string} url
         */
        onDownloadStarted: function onDownloadStarted(
            callback,
            downloadId,
            url
        ) {
            var fileName = this.helpers.getSourceName(url);

            this.updateRecord(downloadId, {
                'fileName': fileName,
                'url': url,
                'state': 0
            });

            if (typeof callback === 'function') {
                callback(downloadId, url);
            }
        },

        /**
         * Handles download progress event.
         * @param {function} callback
         * @param {number} downloadId
         * @param {number} receivedSize
         * @param {number} totalSize
         */
        onDownloadProgress: function Model_onDownloadProgress(
            callback,
            downloadId,
            receivedSize,
            totalSize
        ) {
            this.updateRecord(downloadId, {
                'receivedSize': receivedSize,
                'state': Math.floor((receivedSize / totalSize) * 100)
            });

            if (typeof callback === 'function') {
                callback(downloadId, receivedSize, totalSize);
            }
        },

        /**
         * Handles download pause event.
         * @param {function} callback
         * @param {number} downloadId
         */
        onDownloadPaused: function Model_onDownloadPaused(
            callback,
            downloadId
        ) {
            if (typeof callback === 'function') {
                callback(downloadId);
            }
        },

        /**
         * Handles download canceled event.
         * @param {function} callback
         * @param {number} downloadId
         */
        onDownloadCanceled: function Model_onDownloadCanceled(
            callback,
            downloadId
        ) {
            this.setDownloadFileProperties(downloadId, {
                needsRestart: true
            });

            if (typeof callback === 'function') {
                callback(downloadId);
            }
        },

        /**
         * Handles download completed event.
         * @param {function} callback
         * @param {number} downloadId
         * @param {string} fullPath
         */
        onDownloadCompleted: function Model_onDownloadCompleted(
            callback,
            downloadId,
            fullPath
        ) {
            var fileName = this.helpers.getSourceName(fullPath);

            this.updateRecord(downloadId, {'fileName': fileName});

            if (typeof callback === 'function') {
                callback(downloadId, fullPath);
            }
        },

        /**
         * Handles download failed event.
         * @param {function} callback
         * @param {number} downloadId
         * @param {WebAPIError} error
         */
        onDownloadFailed: function Model_onDownloadFailed(
            callback,
            downloadId,
            error
        ) {
            this.setDownloadFileProperties(downloadId, {
                needsRestart: true,
                state: 'failed'
            });

            if (typeof callback === 'function') {
                callback(downloadId, error);
            }
        },

        /**
         * Creates download callback objects.
         * Injects model callbacks to specified callbacks object.
         * @param {object} eventHandlers
         * @return {DownloadCallback}
         */
        createDownloadCallback: function createDownloadCallback(eventHandlers) {
            return {
                onprogress: this.onDownloadProgress
                    .bind(this, eventHandlers.onprogress),
                onpaused: this.onDownloadPaused
                    .bind(this, eventHandlers.onpaused),
                oncanceled: this.onDownloadCanceled
                    .bind(this, eventHandlers.oncanceled),
                oncompleted: this.onDownloadCompleted
                    .bind(this, eventHandlers.oncompleted),
                onfailed: this.onDownloadFailed
                    .bind(this, eventHandlers.onfailed)
            };
        },

        /**
         * Returns 'true' if 'needsRestart' flag is set on file object in
         * local storage, 'false' otherwise.
         *
         * @param {number} downloadId
         * @return {boolean}
         */
        fileNeedsRestart: function Model_fileNeedsRestart(downloadId) {
            var storageObjectString = localStorage[downloadId],
                storageObject = null;

            if (storageObjectString) {
                storageObject = JSON.parse(storageObjectString);
                return storageObject.needsRestart === true;
            }
            return false;
        },

        /**
         * Sets properties to object that contains information about downloading
         * file.
         * @param {number} downloadId
         * @param {object} properties
         */
        setDownloadFileProperties:
            function Model_setDownloadFileProperties(downloadId, properties) {
                var storageObjectString = localStorage[downloadId],
                    storageObject = null,
                    propertyName = null;

                if (storageObjectString) {
                    storageObject = JSON.parse(storageObjectString);
                    for (propertyName in properties) {
                        storageObject[propertyName] = properties[propertyName];
                    }
                    localStorage[downloadId] = JSON.stringify(storageObject);
                }
            },

        /**
         * Starts downloading process for given resource and configuration.
         *
         * @param {string} url Resource URL.
         * @param {object} eventHandlers Object with download event handlers.
         * @param {function} onSuccess Success callback function.
         */
        startDownloading: function Model_startDownloading(
            url,
            eventHandlers,
            onSuccess
        ) {
            console.log('Model_startDownloading: ' + url);
            var downloadRequest, downloadId;

            try {
                downloadRequest = new tizen.DownloadRequest(url, 'downloads');
                downloadId = tizen.download.start(downloadRequest,
                    this.createDownloadCallback(eventHandlers));

                this.onDownloadStarted(onSuccess, downloadId, url);
            } catch (error) {
                console.error('Start download error', error);
            }
        },

        /**
         * Pauses download operation with specified ID
         * @param {number} downloadId Download ID
         */
        pauseDownloading: function Model_pauseDownloading(downloadId) {
            console.log('Model_pauseDownloading', downloadId);
            try {
                tizen.download.pause(downloadId);
            } catch (error) {
                console.error('Pause download error', error);
            }
        },

        /**
         * Resumes previously paused operation.
         *
         * @param {number} downloadId Download ID
         * @param {object} eventHandlers Object with download events handlers
         * @param {function} listenerCallback
         */
        resumeDownloading: function Model_resumeDownloading(
            downloadId,
            eventHandlers,
            listenerCallback
        ) {
            console.log('Model_resumeDownloading: ', downloadId);
            tizen.download.resume(downloadId);
            if (eventHandlers !== null) {
                tizen.download.setListener(downloadId,
                    this.createDownloadCallback(eventHandlers));
                listenerCallback(downloadId);
            }
        },

        /**
         * Cancels download operation with specified ID
         * @param {number} downloadId Download ID
         */
        cancelDownloading: function Model_cancelDownloading(downloadId) {
            console.log('Model_cancelDownloading: ' + downloadId);
            try {
                tizen.download.cancel(downloadId);
            } catch (error) {
                console.error('Cancel download error', error);
            }
        },

        /**
         * Deletes download operation with specified ID
         * @param {number} downloadId Download ID
         * @return {boolean}
         */
        deleteDownloading: function Model_deleteDownloading(downloadId) {
            console.log('Model_deleteDownloading:', downloadId);
            try {
                return delete localStorage[downloadId];
            } catch (error) {
                console.error('Delete download error', error);
                return false;
            }
        },

        /**
         * Removes all downloads that are finished or failed.
         * @return {Array} Identifiers of removed downloads.
         */
        clearAllFinished: function Model_clearAllFinished() {
            var removed = [],
                downloadId = null,
                downloadObject = null;

            for (downloadId in localStorage) {
                downloadObject = JSON.parse(localStorage[downloadId]);

                if (downloadObject.state === 100 ||
                        downloadObject.state === 'failed') {
                    localStorage.removeItem(downloadId);
                    removed.push(downloadId);
                }
            }

            return removed;
        },

        /**
         * Pauses all active download operations.
         * @param {function} successCallback Success callback function.
         */
        pauseAllActiveDownloads: function Model_pauseAllActiveDownloads(
            successCallback
        ) {
            console.log('Model_pauseAllActiveDownloads');
            var id = null;

            for (id in localStorage) {
                try {
                    if (localStorage.hasOwnProperty(id) &&
                        tizen.download.getState(id) === 'DOWNLOADING') {
                        tizen.download.pause(id);
                    }
                } catch (error) {
                    console.error('Pause all active downloads error', error);
                }
            }
            if (typeof successCallback === 'function') {
                successCallback();
            }
        },

        /**
         * Updates download record in LocalStorage database
         * @param {number} id Download ID
         * @param {object} rData Object that stores detailed download data.
         * @param {string} rData.fileName File name.
         * @param {string} rData.url Resource URL.
         * @param {number} rData.state Download operation state.
         * @param {number} rData.receivedSize
         */
        updateRecord: function Model_updateRecord(id, rData) {
            console.log('Model_updateRecord', id, rData);

            if (id === undefined || typeof id !== 'number') {
                console.error('Update records error: id must be the number');
            }

            var localStorageObject = JSON.parse(localStorage.getItem(id)) || {};

            localStorageObject.id = id;

            if (rData.fileName !== undefined &&
                    typeof rData.fileName === 'string') {
                localStorageObject.fileName = rData.fileName;
            }

            if (rData.url !== undefined && typeof rData.url === 'string') {
                localStorageObject.url = rData.url;
            }

            if (rData.state !== undefined) {
                localStorageObject.state = rData.state;
            }

            if (rData.receivedSize !== undefined) {
                localStorageObject.receivedSize = rData.receivedSize;
            }

            if (localStorageObject.creationTime === undefined) {
                localStorageObject.creationTime = new Date().getTime();
            }

            // save record data to local storage
            localStorage.setItem(id, JSON.stringify(localStorageObject));
        },

        /**
         * Resolves download folder and pass list of files to the success
         * callback function
         * @param {function} successCallback Success callback function
         */
        getDownloads: function Model_getDownloads(successCallback) {
            console.log('Model_refreshDownloadsList');
            var self = this;

            try {
                tizen.filesystem.resolve(
                    'downloads',
                    function onResolveSuccess(dir) {
                        dir.listFiles(
                            function onListFilesSuccess(nodes) {
                                successCallback(self.filterDownloads(nodes));
                            },
                            function onListFilesError(error) {
                                console.error(
                                    'refreshDownloadsList dir.listFiles: ',
                                    error
                                );
                            }
                        );
                    },
                    function onResolveError(error) {
                        console.error(
                            'refreshDownloadsList filesystem.resolve',
                            error
                        );
                    },
                    'r'
                );
            } catch (error) {
                console.error('refreshDownloadsList', error);
            }
        },

        /**
         * Filters given array with nodes and returns only files
         *
         * @param {File[]} nodes
         * @return {object} files
         */
        filterDownloads: function Model_filterDownloads(nodes) {
            console.log('Model_checkDownloadFolder: ', nodes);
            var i, len = nodes.length, files = {};
            for (i = 0; i < len; i += 1) {
                if (nodes[i].isFile) {
                    files[nodes[i].name] = nodes[i].fileSize;
                }
            }
            return files;
        },

        /**
         * Shows folder content using ApplicationControl feature.
         * Method is checking whether MyFiles application is installed.
         * If so, it is used for displaying folder contents.
         */
        showFolder: function Model_showFolder() {
            console.log('Model_showFolder');
            var i,
                len,
                appId,
                service,
                onListInstalledApps = function onGetAppsSuccess(applications) {
                    len = applications.length;
                    for (i = 0; i < len; i += 1) {
                        if (applications[i].id.indexOf('MyFiles') !== -1) {
                            appId = applications[i].id;
                        }
                    }
                    service = new tizen.ApplicationControl(
                        'http://tizen.org/appcontrol/operation/view'
                    );

                    try {
                        tizen.application.launchAppControl(
                            service,
                            appId,
                            function onShowFolderSuccess() {
                                console.log('showFolder success');
                            },
                            function onShowFolderError(error) {
                                console.error(
                                    'showFolder launchAppControl error',
                                    error
                                );
                            }
                        );
                    } catch (error) {
                        console.error('showFolder launchAppControl', error);
                    }
                };

            try {
                tizen.application.getAppsInfo(onListInstalledApps);
            } catch (error) {
                console.error('showFolder error', error);
            }
        },

        /**
         * Launches specified file using ApplicationControl feature.
         * File will be opened if there is installed at least one application
         * that handles view operation for given MIME type.
         *
         * @param {string} fileName File name
         * @param {string} mime File MIME type
         */
        launchFile: function Model_launchFile(fileName, mime) {
            console.log('Model_launchFile: ' + fileName);
            var service;
            service = new tizen.ApplicationControl(
                'http://tizen.org/appcontrol/operation/view',
                'file:///opt/usr/media/Downloads/' + fileName,
                mime,
                null
            );
            try {
                tizen.application.launchAppControl(
                    service,
                    null,
                    function onLaunchFileSuccess() {
                        console.log('Launch file success');
                    },
                    function onLaunchFileError(error) {
                        console.error(
                            'Launch file launchAppControl error: ',
                            error
                        );
                    }
                );
            } catch (error) {
                console.error('Launch file error', error.message);
            }
        },

        /**
         * Returns list of download history.
         * @return {object[]}
         */
        getDownloadList: function Model_getDownloadList() {
            var list = [],
                downloadId = null;

            for (downloadId in localStorage) {
                list.push(JSON.parse(localStorage.getItem(downloadId)));
            }

            return list.sort(function compare(item1, item2) {
                return item2.creationTime - item1.creationTime;
            });
        }
    };
}());
