#include <Eina.h>
#include "$(appName).h"
#include "model.h"
#include "controller.h"
#include "view/view.h"

static void __app_event_low_battery_cb(app_event_info_h event_info, void *user_data)
{
	app_event_low_battery_status_e status;

	if (!model_get_app_event_low_battery_status(event_info, &status))
		return;

	switch (status) {
	case APP_EVENT_LOW_BATTERY_POWER_OFF:
		view_update_event_low_battery("Power off state");
		break;
	case APP_EVENT_LOW_BATTERY_CRITICAL_LOW:
		view_update_event_low_battery("Critical low");
		break;
	default:
		view_update_event_low_battery("Unknown state");
	}
}

static void __app_event_low_memory_cb(app_event_info_h event_info, void *user_data)
{
	app_event_low_memory_status_e status;

	if (!model_get_app_event_low_memory_status(event_info, &status))
		return;

	switch (status) {
	case APP_EVENT_LOW_MEMORY_NORMAL:
		view_update_event_low_memory("Normal level");
		break;
	case APP_EVENT_LOW_MEMORY_SOFT_WARNING:
		view_update_event_low_memory("Soft warning");
		break;
	case APP_EVENT_LOW_MEMORY_HARD_WARNING:
		view_update_event_low_memory("Hard warning");
		break;
	default:
		view_update_event_low_memory("Unknown state");
	}
}

static void __app_event_device_orientation_changed_cb(app_event_info_h event_info, void *user_data)
{
	app_device_orientation_e orientation;

	if (!model_get_app_event_device_orientation(event_info, &orientation))
		return;

	switch (orientation) {
	case APP_DEVICE_ORIENTATION_0:
		view_update_event_orientation_changed("0 [deg]: natural position");
		break;
	case APP_DEVICE_ORIENTATION_90:
		view_update_event_orientation_changed("90 [deg]: leftside up");
		break;
	case APP_DEVICE_ORIENTATION_180:
		view_update_event_orientation_changed("180 [deg]: upside down");
		break;
	case APP_DEVICE_ORIENTATION_270:
		view_update_event_orientation_changed("270 [deg]: rightside up");
		break;
	default:
		view_update_event_orientation_changed("Unknown orientation");
	}
}

static void __app_event_language_changed_cb(app_event_info_h event_info, void *user_data)
{
	char *language = NULL;

	if (!model_get_app_event_language(event_info, &language))
		return;

	view_update_event_language_changed(language);
}

static void __app_event_region_format_changed_cb(app_event_info_h event_info, void *user_data)
{
	char *region = NULL;

	if (!model_get_app_event_region_format(event_info, &region))
		return;

	view_update_event_region_format_changed(region);
}

static void __app_event_suspended_state_changed_cb(app_event_info_h event_info, void *user_data)
{
	app_suspended_state_e state;

	if (!model_get_app_event_suspended_state(event_info, &state))
		return;

	switch (state) {
	case APP_SUSPENDED_STATE_WILL_ENTER:
		view_update_event_suspended_state("Will enter the suspended state");
		break;
	case APP_SUSPENDED_STATE_DID_EXIT:
		view_update_event_suspended_state("Exited from the suspended state");
		break;
	default:
		view_update_event_suspended_state("Unknown state");
	}
}

static bool __add_event_handlers(void)
{
	return (model_add_event_handler(APP_EVENT_LOW_BATTERY, __app_event_low_battery_cb) &&
			model_add_event_handler(APP_EVENT_LOW_MEMORY, __app_event_low_memory_cb) &&
			model_add_event_handler(APP_EVENT_DEVICE_ORIENTATION_CHANGED, __app_event_device_orientation_changed_cb) &&
			model_add_event_handler(APP_EVENT_LANGUAGE_CHANGED, __app_event_language_changed_cb) &&
			model_add_event_handler(APP_EVENT_REGION_FORMAT_CHANGED, __app_event_region_format_changed_cb) &&
			model_add_event_handler(APP_EVENT_SUSPENDED_STATE_CHANGED, __app_event_suspended_state_changed_cb));
}

static void __remove_event_handlers(void)
{
	model_remove_event_handler(APP_EVENT_LOW_BATTERY);
	model_remove_event_handler(APP_EVENT_LOW_MEMORY);
	model_remove_event_handler(APP_EVENT_DEVICE_ORIENTATION_CHANGED);
	model_remove_event_handler(APP_EVENT_LANGUAGE_CHANGED);
	model_remove_event_handler(APP_EVENT_REGION_FORMAT_CHANGED);
	model_remove_event_handler(APP_EVENT_SUSPENDED_STATE_CHANGED);
}

static void __init_view_app(void)
{
	char *id = NULL;
	char *name = NULL;
	char *version = NULL;

	if (model_get_app_id(&id) &&
		model_get_app_name(&name) &&
		model_get_app_version(&version))
		view_update_application_tab(id, name, version);

	char *paths[PATHS_LIST_ITEMS_COUNT] = { NULL };

	if (model_get_app_data_path(&paths[PIT_DATA]) &&
		model_get_app_cache_path(&paths[PIT_CACHE]) &&
		model_get_app_resource_path(&paths[PIT_RESOURCE]) &&
		model_get_app_shared_data_path(&paths[PIT_SHARED_DATA]) &&
		model_get_app_shared_resource_path(&paths[PIT_SHARED_RESOURCE]) &&
		model_get_app_shared_trusted_path(&paths[PIT_SHARED_TRUSTED]) &&
		model_get_app_external_data_path(&paths[PIT_EXTERNAL_DATA]) &&
		model_get_app_external_cache_path(&paths[PIT_EXTERNAL_CACHE]) &&
		model_get_app_external_shared_data_path(&paths[PIT_EXTERNAL_SHARED_DATA]) &&
		model_get_app_tep_resource_path(&paths[PIT_TEP_RESOURCE])) {
		int i;
		view_update_paths_tab(paths);

		for (i = 0; i < PATHS_LIST_ITEMS_COUNT; i++)
			free(paths[i]);
	}
}

static void __paths_popup_opened_cb(const char *path)
{
	Eina_List *files_list = NULL;
	Eina_List *list = NULL;
	char *file_name = NULL;

	if (!model_get_files_list(path, &files_list))
		return;

	view_clear_paths_tab_files_list();

	EINA_LIST_FOREACH(files_list, list, file_name) {
		view_update_paths_tab_files_list(file_name);
		free(file_name);
	}

	eina_list_free(files_list);
}

bool controller_init(viewdata_s *vd, modeldata_s *md)
{
	if (!model_init(md))
		return false;

	if (!__add_event_handlers())
		return false;

	if (!view_create_base_gui(vd, __paths_popup_opened_cb))
		return false;

	__init_view_app();

	return true;
}

void controller_finit(viewdata_s *vd)
{
	view_destroy_base_gui(vd);
	__remove_event_handlers();
}

void controller_log(log_priority pri, const char *format, ...)
{
	va_list vl;

	va_start(vl, format);
	dlog_vprint(pri, LOG_TAG, format, vl);
	va_end(vl);
}
