#include "$(appName).h"
#include "view/generic.h"
#include "controller.h"

static void __get_app_resource(const char *edj_file_in, char *edj_path_out, int edj_path_max)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, edj_path_max, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

Evas_Object *view_generic_create_window(Evas_Smart_Cb on_delete_cb)
{
	Evas_Object *win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	if (!win) {
		controller_log(DLOG_ERROR, "Function elm_win_util_standard_add() failed.");
		return NULL;
	}

	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_autodel_set(win, EINA_TRUE);

	evas_object_smart_callback_add(win, "delete,request", on_delete_cb, NULL);

	return win;
}

Evas_Object *view_generic_create_conformant(Evas_Object *parent)
{
	Evas_Object *conform = NULL;

	if (!parent) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	conform = elm_conformant_add(parent);
	if (!conform) {
		controller_log(DLOG_ERROR, "Function elm_conformant_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(parent, conform);
	evas_object_show(conform);

	return conform;
}

Evas_Object *view_generic_create_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;

	if (!parent || !edj_file_name || !edj_group) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	__get_app_resource(edj_file_name, edj_path, (int)PATH_MAX);

	layout = elm_layout_add(parent);
	if (!layout) {
		controller_log(DLOG_ERROR, "Function elm_layout_add() failed.");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, edj_group)) {
		controller_log(DLOG_ERROR, "Function elm_layout_file_set() failed.");
		evas_object_del(layout);
		return NULL;
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	return layout;
}

Evas_Object *view_generic_create_set_layout(Evas_Object *parent, const char *edj_file_name, const char *edj_group, const char *target_part_name)
{
	Evas_Object *layout = view_generic_create_layout(parent, edj_file_name, edj_group);
	if (!layout)
		return NULL;

	if (!target_part_name) {
		elm_object_content_set(parent, layout);
	} else {
		elm_object_part_content_set(parent, target_part_name, layout);
	}

	return layout;
}

Evas_Object *view_generic_create_box(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *box = NULL;

	if (!parent || !target_part_name) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	box = elm_box_add(parent);
	if (!box) {
		controller_log(DLOG_ERROR, "Function elm_box_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(box, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_part_content_set(parent, target_part_name, box);
	evas_object_show(box);

	return box;
}

Evas_Object *view_generic_create_toolbar(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *toolbar = NULL;

	if (!parent || !target_part_name) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	toolbar = elm_toolbar_add(parent);
	if (!toolbar) {
		controller_log(DLOG_ERROR, "Function elm_toolbar_add() failed.");
		return NULL;
	}

	elm_toolbar_shrink_mode_set(toolbar, ELM_TOOLBAR_SHRINK_EXPAND);
	elm_object_style_set(toolbar, "tabbar");
	elm_toolbar_transverse_expanded_set(toolbar, EINA_TRUE);
	elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_ALWAYS);
	elm_object_part_content_set(parent, target_part_name, toolbar);
	evas_object_size_hint_weight_set(toolbar, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	evas_object_show(toolbar);

	return toolbar;
}

Evas_Object *view_generic_create_entry(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *entry = elm_entry_add(parent);
	if (!entry) {
		controller_log(DLOG_ERROR, "Function elm_entry_add() failed.");
		return NULL;
	}

	elm_entry_line_wrap_set(entry, ELM_WRAP_NONE);
	elm_entry_single_line_set(entry, EINA_TRUE);
	elm_entry_editable_set(entry, EINA_TRUE);
	elm_entry_scrollable_set(entry, EINA_FALSE);
	evas_object_size_hint_weight_set(entry, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(entry, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_part_content_set(parent, target_part_name, entry);
	evas_object_show(entry);

	return entry;
}

Evas_Object *view_generic_create_label(Evas_Object *parent, const char *target_part_name, const char *caption)
{
	Evas_Object *label = NULL;

	if (!parent || !target_part_name) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	label = elm_label_add(parent);
	if (!label) {
		controller_log(DLOG_ERROR, "Function elm_label_add() failed.");
		return NULL;
	}

	elm_object_part_content_set(parent, target_part_name, label);
	elm_object_part_text_set(label, "default", caption);
	evas_object_show(label);

	return label;
}

Evas_Object *view_generic_create_spinner(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *spinner = NULL;

	if (!parent || !target_part_name) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	spinner = elm_spinner_add(parent);
	if (!spinner) {
		controller_log(DLOG_ERROR, "Function elm_spinner_add() failed.");
		return NULL;
	}

	elm_spinner_editable_set(spinner, EINA_FALSE);
	elm_spinner_min_max_set(spinner, 0, 0);
	elm_spinner_wrap_set(spinner, EINA_TRUE);
	evas_object_size_hint_align_set(spinner, 0.5, 0.5);
	evas_object_size_hint_weight_set(spinner, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, spinner);

	evas_object_show(spinner);

	return spinner;
}

Evas_Object *view_generic_create_checkbox(Evas_Object *parent, const char *target_part_name, const char *caption)
{
	Evas_Object *checkbox = NULL;

	if (!parent || !target_part_name || !caption) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	checkbox = elm_check_add(parent);
	if (!checkbox) {
		controller_log(DLOG_ERROR, "Function elm_check_add() failed.");
		return NULL;
	}

	elm_check_state_set(checkbox, EINA_FALSE);
	elm_object_part_text_set(checkbox, "default", caption);
	elm_object_part_content_set(parent, target_part_name, checkbox);

	evas_object_show(checkbox);

	return checkbox;
}

Evas_Object *view_generic_create_genlist(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *list = NULL;

	if (!parent || !target_part_name) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	list = elm_genlist_add(parent);
	if (!list) {
		controller_log(DLOG_ERROR, "Function elm_genlist_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(list, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, list);

	evas_object_show(list);

	return list;
}

Evas_Object *view_generic_create_list(Evas_Object *parent, const char *target_part_name)
{
	Evas_Object *list = NULL;

	if (!parent || !target_part_name) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	list = elm_list_add(parent);
	if (!list) {
		controller_log(DLOG_ERROR, "Function elm_list_add() failed.");
		return NULL;
	}

	elm_list_mode_set(list, ELM_LIST_SCROLL);
	evas_object_size_hint_weight_set(list, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, list);

	evas_object_show(list);

	return list;
}

Elm_Genlist_Item_Class *view_generic_create_genlist_item_class(Elm_Genlist_Item_Text_Get_Cb on_text_get_cb, Elm_Genlist_Item_Del_Cb on_item_del_cb)
{
	static Elm_Genlist_Item_Class *itc = NULL;

	if (!itc) {
		itc = elm_genlist_item_class_new();
		if (!itc) {
			controller_log(DLOG_ERROR, "Function elm_genlist_item_class_new() failed.");
			return NULL;
		}

		itc->item_style = "double_label";
		itc->func.text_get = on_text_get_cb;
		itc->func.content_get = NULL;
		itc->func.state_get = NULL;
		itc->func.del = on_item_del_cb;
	}

	return itc;
}

Evas_Object *view_generic_create_button(Evas_Object *parent, const char *target_part_name, const char *caption, Evas_Smart_Cb on_click_cb, void *data)
{
	Evas_Object *button = NULL;

	if (!parent || !target_part_name || !caption) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	button = elm_button_add(parent);
	if (!button) {
		controller_log(DLOG_ERROR, "Function elm_button_add() failed.");
		return NULL;
	}

	elm_object_part_content_set(parent, target_part_name, button);
	elm_object_text_set(button, caption);

	evas_object_smart_callback_add(button, "clicked", on_click_cb, data);

	evas_object_show(button);

	return button;
}

Evas_Object *view_generic_create_popup(Evas_Object *parent, const char *title)
{
	Evas_Object *popup = elm_popup_add(parent);
	if (!popup) {
		controller_log(DLOG_ERROR, "Function elm_popup_add() failed.");
		return NULL;
	}

	elm_object_part_text_set(popup, "title,text", title);

	return popup;
}
