#include "$(appName).h"
#include "view/defines.h"
#include "view/generic.h"
#include "view/view.h"
#include "controller.h"

struct __path_list_item {
	char *top_label;
	char *bottom_label;
	path_item_type type;
};

typedef struct __path_list_item __path_list_item_s;

static viewdata_s *__viewdata = NULL;
static __path_list_item_s __list_items[PATHS_LIST_ITEMS_COUNT] = {
	{ .top_label = "Data", .bottom_label = NULL, .type = PIT_DATA, },
	{ .top_label = "Cache", .bottom_label = NULL, .type = PIT_CACHE, },
	{ .top_label = "Resource", .bottom_label = NULL, .type = PIT_RESOURCE, },
	{ .top_label = "Shared data", .bottom_label = NULL, .type = PIT_SHARED_DATA, },
	{ .top_label = "Shared resource", .bottom_label = NULL, .type = PIT_SHARED_RESOURCE, },
	{ .top_label = "Shared trusted", .bottom_label = NULL, .type = PIT_SHARED_TRUSTED, },
	{ .top_label = "External data", .bottom_label = NULL, .type = PIT_EXTERNAL_DATA, },
	{ .top_label = "External cache", .bottom_label = NULL, .type = PIT_EXTERNAL_CACHE, },
	{ .top_label = "External shared data", .bottom_label = NULL, .type = PIT_EXTERNAL_SHARED_DATA, },
	{ .top_label = "Tep resource", .bottom_label = NULL, .type = PIT_TEP_RESOURCE, },
};


static void __delete_win_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

static void __layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;

	elm_win_lower(vd->win);
}

static void __toolbar_item_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *current_page = elm_object_part_content_unset(__viewdata->layout_main_panel, PART_MAIN_CONTENT);
	if (current_page)
		evas_object_hide(current_page);

	if (!data)
		return;

	elm_object_part_content_set(__viewdata->layout_main_panel, PART_MAIN_CONTENT, (Evas_Object *)data);
}

static char *__get_paths_item_label_cb(void *data, Evas_Object *obj, const char *part)
{
	__path_list_item_s *item = (__path_list_item_s *)data;

	if (!item)
		return NULL;

	if (!strcmp(part, "elm.text")) {
		return strdup(item->top_label);
	} else if (!strcmp(part, "elm.text.sub")) {
		if (!item->bottom_label)
			return strdup("---");

		return strdup(item->bottom_label);
	}

	return NULL;
}

static void __del_paths_item_cb(void *data, Evas_Object *obj)
{
	__path_list_item_s *item = (__path_list_item_s *)data;

	if (!item)
		return;

	if (item->bottom_label)
		free(item->bottom_label);
}

static void __paths_popup_button_close_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;

	evas_object_del(vd->popup_paths);
	vd->popup_paths = NULL;
}

static Evas_Object *__create_rect_paths_popup(Evas_Object *parent, const char *target_part_name, Evas_Coord width, Evas_Coord height)
{
	Evas *evas = evas_object_evas_get(parent);
	if (!evas) {
		controller_log(DLOG_ERROR, "Function evas_object_evas_get() failed");
		return NULL;
	}

	Evas_Object *rect = evas_object_rectangle_add(evas);
	if (!rect) {
		controller_log(DLOG_ERROR, "Function evas_object_rectangle_add() failed");
		return NULL;
	}

	elm_object_part_content_set(parent, target_part_name, rect);
	evas_object_resize(rect, width, height);
	evas_object_size_hint_min_set(rect, width, height);
	evas_object_size_hint_max_set(rect, width, height);

	evas_object_show(rect);

	return rect;
}

static void __delete_popup_window(void)
{
	if (__viewdata->popup_paths) {
		evas_object_del(__viewdata->popup_paths);
		__viewdata->popup_paths = NULL;
	}
}

static void __create_files_popup(const char *title, const char *path)
{
	__viewdata->popup_paths = view_generic_create_popup(__viewdata->win, title);
	if (!__viewdata->popup_paths)
		return;

	view_generic_create_button(__viewdata->popup_paths, "button1", "Close", __paths_popup_button_close_cb, (void *)__viewdata);

	Evas_Object *layout_popup = view_generic_create_set_layout(__viewdata->popup_paths, EDJ_POPUP_FILE_NAME, GROUP_POPUP, "default");
	if (!layout_popup) {
		__delete_popup_window();
		return;
	}

	Evas_Coord width = 0;
	Evas_Coord height = 0;
	evas_object_geometry_get(__viewdata->layout_paths, NULL, NULL, &width, &height);

	Evas_Object *rect = __create_rect_paths_popup(layout_popup, PART_POPUP_BACKGROUND_RECT, width * 0.9, height * 0.3);
	if (!rect) {
		__delete_popup_window();
		return;
	}

	elm_object_part_text_set(layout_popup, PART_POPUP_PATH, path);
	elm_object_part_text_set(layout_popup, PART_POPUP_FILES_LABEL, "Files list:");

	__viewdata->popup_files_list = view_generic_create_entry(layout_popup, PART_POPUP_FILES_LIST);
	if (!__viewdata->popup_files_list) {
		__delete_popup_window();
		return;
	}

	elm_entry_line_wrap_set(__viewdata->popup_files_list, ELM_WRAP_MIXED);
	elm_entry_single_line_set(__viewdata->popup_files_list, EINA_FALSE);
	elm_entry_editable_set(__viewdata->popup_files_list, EINA_FALSE);
	elm_entry_scrollable_set(__viewdata->popup_files_list, EINA_FALSE);
	elm_entry_text_style_user_push(__viewdata->popup_files_list, "DEFAULT='font_size=27 color=#a0a0a0'");

	evas_object_show(__viewdata->popup_paths);

	view_update_paths_tab_files_list("No files found");
}

static void __paths_item_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	__path_list_item_s *path_item = (__path_list_item_s *)data;

	if (__viewdata->popup_paths)
		evas_object_del(__viewdata->popup_paths);

	__create_files_popup(path_item->top_label, path_item->bottom_label);

	if (__viewdata->popup_opened_cb)
		__viewdata->popup_opened_cb(path_item->bottom_label);
}

static bool __create_main_panel(viewdata_s *vd)
{
	vd->win = view_generic_create_window(__delete_win_request_cb);
	if (!vd->win)
		return false;

	vd->conform = view_generic_create_conformant(vd->win);
	if (!vd->conform)
		return false;

	vd->layout_main_panel = view_generic_create_set_layout(vd->conform, EDJ_MAIN_FILE_NAME, GROUP_MAIN, NULL);
	if (!vd->layout_main_panel)
		return false;

	eext_object_event_callback_add(vd->layout_main_panel, EEXT_CALLBACK_BACK, __layout_back_cb, (void *)vd);

	vd->main_toolbar = view_generic_create_toolbar(vd->layout_main_panel, PART_MAIN_TOOLBAR);
	if (!vd->main_toolbar)
		return false;

	vd->main_toolbar_item_app = elm_toolbar_item_append(vd->main_toolbar, NULL, "Application", __toolbar_item_selected_cb, NULL);
	if (!vd->main_toolbar_item_app)
		return false;

	vd->main_toolbar_item_events = elm_toolbar_item_append(vd->main_toolbar, NULL, "Events", __toolbar_item_selected_cb, NULL);
	if (!vd->main_toolbar_item_events)
		return false;

	vd->main_toolbar_item_paths = elm_toolbar_item_append(vd->main_toolbar, NULL, "Paths", __toolbar_item_selected_cb, NULL);
	if (!vd->main_toolbar_item_paths)
		return false;

	return true;
}

static bool __create_app_view(viewdata_s *vd)
{
	vd->layout_app = view_generic_create_layout(vd->layout_main_panel, EDJ_APP_FILE_NAME, GROUP_APP);
	if (!vd->layout_app)
		return false;

	return true;
}

static bool __create_events_view(viewdata_s *vd)
{
	vd->layout_events = view_generic_create_layout(vd->layout_main_panel, EDJ_EVENTS_FILE_NAME, GROUP_EVENTS);
	if (!vd->layout_events)
		return false;

	return true;
}

static bool __create_paths_view(viewdata_s *vd)
{
	vd->layout_paths = view_generic_create_layout(vd->layout_main_panel, EDJ_PATHS_FILE_NAME, GROUP_PATHS);
	if (!vd->layout_paths)
		return false;

	vd->paths_list = view_generic_create_genlist(vd->layout_paths, PART_PATHS_CONTENT);
	if (!vd->paths_list)
		return false;

	Elm_Genlist_Item_Class *itc = view_generic_create_genlist_item_class(__get_paths_item_label_cb, __del_paths_item_cb);
	if (!itc)
		return false;

	int i;
	for (i = 0; i < PATHS_LIST_ITEMS_COUNT; i++)
		elm_genlist_item_append(vd->paths_list,
									itc,
									(void *)&__list_items[i],
									NULL,
									ELM_GENLIST_ITEM_NONE,
									__paths_item_selected_cb,
									(void *)&__list_items[i]);

	return true;
}

bool view_create_base_gui(viewdata_s *vd, paths_popup_opened_cb callback)
{
	__viewdata = vd;
	__viewdata->popup_opened_cb = callback;

	if (!__create_main_panel(vd)) {
		evas_object_del(vd->win);
		return false;
	}

	if (!__create_app_view(vd)) {
		evas_object_del(vd->win);
		return false;
	}

	if (!__create_events_view(vd)) {
		evas_object_del(vd->win);
		return false;
	}

	if (!__create_paths_view(vd)) {
		evas_object_del(vd->win);
		return false;
	}

	/* The layouts are assigned to the item's data field. These layouts are used in
	 * __toolbar_item_selected_cb() callback function to switch the layout
	 * on toolbar's item selection.
	 */
	elm_object_item_data_set(vd->main_toolbar_item_app, (void *)vd->layout_app);
	elm_object_item_data_set(vd->main_toolbar_item_events, (void *)vd->layout_events);
	elm_object_item_data_set(vd->main_toolbar_item_paths, (void *)vd->layout_paths);

	/* The "Application" tab is marked as selected
	 * (the __toolbar_item_selected_cb() callback function is not
	 * automatically called).
	 */
	elm_toolbar_item_selected_set(vd->main_toolbar_item_app, EINA_TRUE);

	/* Set the vd->layout_app as the content of vd->layout_main_panel layout because the
	 * __toolbar_item_selected_cb() callback function is not called
	 * when elm_toolbar_item_selected_set() is invoked.
	 */
	elm_object_part_content_set(vd->layout_main_panel, PART_MAIN_CONTENT, vd->layout_app);

	evas_object_show(vd->win);

	return true;
}

void view_destroy_base_gui(viewdata_s *vd)
{
	if (!vd)
		return;

	if (vd->layout_app)
		evas_object_del(vd->layout_app);

	if (vd->layout_events)
		evas_object_del(vd->layout_events);

	if (vd->layout_paths)
		evas_object_del(vd->layout_paths);

	if (vd->win)
		evas_object_del(vd->win);

	int i;
	for (i = 0; i < PATHS_LIST_ITEMS_COUNT; i++)
		free(__list_items[i].bottom_label);
}

void view_update_application_tab(const char *name, const char *id, const char *version)
{
	if (name)
		elm_object_part_text_set(__viewdata->layout_app, PART_APP_NAME_PANEL_NAME, name);

	if (id)
		elm_object_part_text_set(__viewdata->layout_app, PART_APP_ID_PANEL_ID, id);

	if (version)
		elm_object_part_text_set(__viewdata->layout_app, PART_APP_VERSION_PANEL_VERSION, version);
}

void view_update_event_low_battery(const char *status_msg)
{
	if (status_msg)
		elm_object_part_text_set(__viewdata->layout_events, PART_EVENTS_BATTERY_PANEL_STATUS, status_msg);
}

void view_update_event_low_memory(const char *status_msg)
{
	if (status_msg)
		elm_object_part_text_set(__viewdata->layout_events, PART_EVENTS_MEMORY_PANEL_STATUS, status_msg);
}

void view_update_event_language_changed(const char *language)
{
	if (language)
		elm_object_part_text_set(__viewdata->layout_events, PART_EVENTS_LANGUAGE_PANEL_STATUS, language);
}

void view_update_event_region_format_changed(const char *region)
{
	if (region)
		elm_object_part_text_set(__viewdata->layout_events, PART_EVENTS_REGION_PANEL_STATUS, region);
}

void view_update_event_orientation_changed(const char *status_msg)
{
	if (status_msg)
		elm_object_part_text_set(__viewdata->layout_events, PART_EVENTS_ORIENTATION_PANEL_STATUS, status_msg);
}

void view_update_event_suspended_state(const char *status_msg)
{
	if (status_msg)
		elm_object_part_text_set(__viewdata->layout_events, PART_EVENTS_SUSPENDED_PANEL_STATUS, status_msg);
}

void view_update_paths_tab(char **paths)
{
	int i;
	for (i = 0; i < PATHS_LIST_ITEMS_COUNT; i++)
		__list_items[i].bottom_label = strdup(paths[i]);
}

void view_clear_paths_tab_files_list(void)
{
	elm_entry_entry_set(__viewdata->popup_files_list, "");
}

void view_update_paths_tab_files_list(const char *file_name)
{
	elm_entry_entry_append(__viewdata->popup_files_list, file_name);
	elm_entry_entry_append(__viewdata->popup_files_list, "<br>");
}
