#include "$(appName).h"
#include "model.h"
#include "view.h"
#include "controller.h"

#define EMAIL_DEFAULT_MESSAGE_SUBJECT "Message from appcontrol"
#define EMAIL_APP_ID "org.tizen.email-composer"

static controller_data_s controller_data = {0, };

static void __app_control_launch_reply_cb(app_control_h request, app_control_h reply, app_control_result_e result, void *user_data)
{
	switch (result) {
		case APP_CONTROL_RESULT_SUCCEEDED:
			controller_print_message(DLOG_INFO, "Launch request result: Operation succeeded");
			break;
		case APP_CONTROL_RESULT_FAILED:
			controller_print_message(DLOG_INFO, "Launch request result: Operation failed by the callee");
			break;
		case APP_CONTROL_RESULT_CANCELED:
			controller_print_message(DLOG_INFO, "Launch request result: Operation canceled by the framework");
			break;
		default:
			controller_print_message(DLOG_INFO, "Launch request result: unknown result type");
	}
}

bool controller_initialize(viewdata_s *vd, modeldata_s *md)
{
	controller_data.viewdata = vd;

	int ret = app_control_create(&controller_data.launch_app_control);
	if (ret != APP_CONTROL_ERROR_NONE) {
		controller_print_message(DLOG_ERROR, "app_control_create() failed. Err = %d", ret);
		return false;
	}

	return (model_create(md) && view_create(vd));
}

void controller_finalize(void)
{
	app_control_destroy(controller_data.launch_app_control);

	model_free_app_control_selection();
	model_free_app_control_apps_list();
}

void controller_print_message(log_priority pri, const char *format, ...)
{
	va_list vl;

	va_start(vl, format);
	dlog_vprint(pri, LOG_TAG, format, vl);
	va_end(vl);
}

static bool app_control_application_matched_cb(app_control_h app_control, const char *appid, void *user_data)
{
	bool new_radio_group = false;

	if (!view_is_shown_popup_window()) {
		new_radio_group = true;
		if (!view_create_popup_window())
			return false;
	}

	if (view_add_popup_window_radio_button(appid, new_radio_group))
		model_add_application_id(appid);

	view_disable_button(controller_data.viewdata, BUTTON_EXECUTE, EINA_FALSE);

	return true;
}

void controller_search_app(int selected_operation_id)
{
	app_control_h app_control = NULL;

	model_free_app_control_selection();

	const char *operation = model_get_operation_name(selected_operation_id);
	if (!operation)
		return;

	int ret = app_control_create(&app_control);
	if (ret != APP_CONTROL_ERROR_NONE) {
		controller_print_message(DLOG_ERROR, "app_control_create() failed. Err = %d", ret);
		return;
	}

	ret = app_control_set_operation(app_control, operation);
	if (ret != APP_CONTROL_ERROR_NONE) {
		app_control_destroy(app_control);
		controller_print_message(DLOG_ERROR, "app_control_set_operation() failed. Err = %d", ret);
		return;
	}

	ret = app_control_foreach_app_matched(app_control, app_control_application_matched_cb, NULL);
	if (ret != APP_CONTROL_ERROR_NONE) {
		app_control_destroy(app_control);
		controller_print_message(DLOG_ERROR, "app_control_foreach_app_matched() failed. Err = %d", ret);
		return;
	}

	app_control_destroy(app_control);
}

void controller_execute(void)
{
	char *operation = NULL;
	char *appid = NULL;

	if (!model_get_app_control_selection(&operation, &appid)) {
		controller_print_message(DLOG_ERROR, "Cannot execute. The operation/appid is not specified");
		return;
	}

	int ret = app_control_set_operation(controller_data.launch_app_control, operation);
	if (ret != APP_CONTROL_ERROR_NONE) {
		controller_print_message(DLOG_ERROR, "app_control_set_operation() failed. Err = %d", ret);
		return;
	}

	ret = app_control_set_app_id(controller_data.launch_app_control, appid);
	if (ret != APP_CONTROL_ERROR_NONE) {
		controller_print_message(DLOG_ERROR, "app_control_set_app_id() failed. Err = %d", ret);
		return;
	}

	ret = app_control_add_extra_data(controller_data.launch_app_control, APP_CONTROL_DATA_TO, COMPOSE_OP_RECIPIENT);
	if (ret == APP_CONTROL_ERROR_NONE)
		ret = app_control_add_extra_data(controller_data.launch_app_control, APP_CONTROL_DATA_CC, COMPOSE_OP_RECIPIENT_CC);
	if (ret == APP_CONTROL_ERROR_NONE)
		ret = app_control_add_extra_data(controller_data.launch_app_control, APP_CONTROL_DATA_BCC, COMPOSE_OP_RECIPIENT_BCC);
	if (ret == APP_CONTROL_ERROR_NONE)
		ret = app_control_add_extra_data(controller_data.launch_app_control, APP_CONTROL_DATA_SUBJECT, COMPOSE_OP_SUBJECT);
	if (ret == APP_CONTROL_ERROR_NONE)
		ret = app_control_add_extra_data(controller_data.launch_app_control, APP_CONTROL_DATA_TEXT, COMPOSE_OP_MESSAGE);

	if (ret != APP_CONTROL_ERROR_NONE) {
		controller_print_message(DLOG_ERROR, "app_control_add_extra_data() failed. Err = %d", ret);
		return;
	}

	ret = app_control_send_launch_request(controller_data.launch_app_control, __app_control_launch_reply_cb, NULL);
	if (ret != APP_CONTROL_ERROR_NONE) {
		controller_print_message(DLOG_ERROR, "app_control_send_launch_request() failed. Err = %d", ret);
		return;
	}

	view_disable_button(controller_data.viewdata, BUTTON_KILL, EINA_FALSE);
}

void controller_terminate(void)
{
	int ret = app_control_send_terminate_request(controller_data.launch_app_control);
	if (ret != APP_CONTROL_ERROR_NONE) {
		controller_print_message(DLOG_ERROR, "app_control_send_terminate_request() failed. Err = %d", ret);
		return;
	}

	model_free_app_control_selection();
	view_set_app_selected(controller_data.viewdata, NULL);
}

void controller_send_message(const char *email_address, const char *email_body)
{
	app_control_h app_control = NULL;

	int ret = app_control_create(&app_control);
	if (ret != APP_CONTROL_ERROR_NONE) {
		controller_print_message(DLOG_ERROR, "app_control_create() failed. Err = %d", ret);
		return;
	}

	ret = app_control_set_launch_mode(app_control, APP_CONTROL_LAUNCH_MODE_GROUP);
	if (ret != APP_CONTROL_ERROR_NONE) {
		app_control_destroy(app_control);
		controller_print_message(DLOG_ERROR, "app_control_set_launch_mode() failed. Err = %d", ret);
		return;
	}

	ret = app_control_set_operation(app_control, APP_CONTROL_OPERATION_COMPOSE);
	if (ret != APP_CONTROL_ERROR_NONE) {
		app_control_destroy(app_control);
		controller_print_message(DLOG_ERROR, "app_control_set_operation() failed. Err = %d", ret);
		return;
	}

	ret = app_control_set_app_id(app_control, EMAIL_APP_ID);
	if (ret != APP_CONTROL_ERROR_NONE) {
		app_control_destroy(app_control);
		controller_print_message(DLOG_ERROR, "app_control_set_app_id() failed. Err = %d", ret);
		return;
	}
	
	ret = app_control_add_extra_data(app_control, APP_CONTROL_DATA_TO, email_address);
	if (ret != APP_CONTROL_ERROR_NONE) {
		app_control_destroy(app_control);
		controller_print_message(DLOG_ERROR, "app_control_add_extra_data() failed. Err = %d", ret);
		return;
	}

	ret = app_control_add_extra_data(app_control, APP_CONTROL_DATA_SUBJECT, EMAIL_DEFAULT_MESSAGE_SUBJECT);
	if (ret != APP_CONTROL_ERROR_NONE) {
		app_control_destroy(app_control);
		controller_print_message(DLOG_ERROR, "app_control_add_extra_data() failed. Err = %d", ret);
		return;
	}

	ret = app_control_add_extra_data(app_control, APP_CONTROL_DATA_TEXT, email_body);
	if (ret != APP_CONTROL_ERROR_NONE) {
		app_control_destroy(app_control);
		controller_print_message(DLOG_ERROR, "app_control_add_extra_data() failed. Err = %d", ret);
		return;
	}

	ret = app_control_send_launch_request(app_control, __app_control_launch_reply_cb, NULL);
	if (ret != APP_CONTROL_ERROR_NONE) {
		app_control_destroy(app_control);
		controller_print_message(DLOG_ERROR, "app_control_send_launch_request() failed. Err = %d", ret);
		return;
	}

	app_control_destroy(app_control);
}
