#include <tizen.h>
#include "$(appName).h"
#include "defines.h"
#include "model.h"
#include "view.h"
#include "controller.h"

#define TEXT_BUFF 256
#define DEFAULT_TEXT_MESSAGE "Dear Developer,<br><br>This is the default message sent from<br>"\
								"appcontrol sample application.<br>"\
								"Feel free to modify this text message in email composer.<br><br>"\
								"Best Regards."
#define DEFAULT_TEXT_MESSAGE_EMAIL "Dear Developer,\n\nThis is the default message sent from\n"\
								"appcontrol sample application.\n"\
								"Feel free to modify this text message in email composer.\n\n"\
								"Best Regards."

viewdata_s *viewdata;


static void __get_edj_path(const char *edj_file_in, char *edj_path_out, int edj_path_max)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, edj_path_max, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

static void __on_controls_app_launch_search_button_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;

	controller_search_app(vd->radio_button_operation_selected);
}

static void __on_controls_app_launch_execute_button_cb(void *data, Evas_Object *obj, void *event_info)
{
	controller_execute();
}

static void __on_controls_app_launch_terminate_button_cb(void *data, Evas_Object *obj, void *event_info)
{
	controller_terminate();
}

static void __window_delete_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

static void __window_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;

	elm_win_lower(vd->win);
}

static void __on_popup_window_button_cancel_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;

	model_free_app_control_apps_list();

	evas_object_del(vd->popup_window);
	vd->popup_window = NULL;
}

static void __on_popup_window_button_select_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;

	model_set_app_control_selection(vd->radio_button_operation_selected, vd->radio_button_application_selected);

	const char *appid = model_get_application_id(vd->radio_button_application_selected);
	if (!appid)
		return;
	view_set_app_selected(vd, appid);

	__on_popup_window_button_cancel_cb(data, obj, event_info);
}


static void __on_controls_message_compose_send_button_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;

	const char *address_to = elm_entry_entry_get(vd->controls_message_compose_address_to_entry);
	if (!address_to || strlen(address_to) == 0) {
		controller_print_message(DLOG_ERROR, "address-to: NOT PROVIDED !.");
		return;
	}

	controller_send_message(address_to, DEFAULT_TEXT_MESSAGE_EMAIL);
}

static Evas_Object *__create_main_window(viewdata_s *vd)
{
	Evas_Object *win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	if (!win) {
		controller_print_message(DLOG_ERROR, "elm_win_util_standard_add() failed.");
		return NULL;
	}

	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);

	evas_object_smart_callback_add(win, "delete,request", __window_delete_request_cb, NULL);
	eext_object_event_callback_add(win, EEXT_CALLBACK_BACK, __window_back_cb, (void*)vd);

	return win;
}

static Evas_Object *__create_conformant(Evas_Object *parent)
{
	Evas_Object *conformant = elm_conformant_add(parent);
	if (!conformant) {
		controller_print_message(DLOG_ERROR, "elm_conformant_add() failed.");
		return NULL;
	}

	elm_win_resize_object_add(parent, conformant);
	evas_object_show(conformant);

	return conformant;
}

static Evas_Object *__create_layout(const char *edj_file_name, const char *edj_group_name, const char *edj_part_name, Evas_Object *parent)
{
	char edj_path[PATH_MAX] = {0, };
	Evas_Object *layout = NULL;

	if (!edj_group_name) {
		controller_print_message(DLOG_ERROR, "Cannot create layout, the group name is empty.");
		return NULL;
	}

	__get_edj_path(edj_file_name, edj_path, (int)PATH_MAX);
	layout = elm_layout_add(parent);
	if (!layout) {
		controller_print_message(DLOG_ERROR, "elm_layout_add() failed.");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, edj_group_name)) {
		controller_print_message(DLOG_ERROR, "elm_layout_file_set() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	if (!edj_part_name) {
		elm_object_content_set(parent, layout);
	} else {
		elm_object_part_content_set(parent, edj_part_name, layout);
	}

	return layout;
}

static Evas_Object *__create_box(Evas_Object *parent, const char *edj_part_name)
{
	Evas_Object *box = elm_box_add(parent);
	if (!box) {
		controller_print_message(DLOG_ERROR, "elm_box_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(box, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_part_content_set(parent, edj_part_name, box);
	evas_object_show(box);

	return box;
}

static Evas_Object *__create_radio_button(Evas_Object *parent, const char *caption, int *rb_selected, bool create_new_group)
{
	static int radio_id = 0;
	static Evas_Object *radio_group = NULL;

	Evas_Object *radio;

	if (create_new_group) {
		radio_id = 0;
		radio_group = NULL;
	}

	radio = elm_radio_add(parent);
	if (!radio) {
		controller_print_message(DLOG_ERROR, "elm_radio_add() failed.");
		return NULL;
	}

	elm_object_part_text_set(radio, "default", caption);
	elm_radio_state_value_set(radio, radio_id);
	elm_radio_value_pointer_set(radio, rb_selected);

	if (radio_id == 0) {
		radio_group = radio;
		elm_box_pack_start(parent, radio);
	} else {
		elm_radio_group_add(radio, radio_group);
		elm_box_pack_end(parent, radio);
	}

	evas_object_size_hint_weight_set(radio, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(radio, EVAS_HINT_FILL, EVAS_HINT_FILL);

	evas_object_show(radio);

	radio_id++;

	return radio;
}

static Evas_Object *__create_button(Evas_Object *parent, const char *edj_part_name, const char *caption, Evas_Smart_Cb on_click_cb, void *data)
{
	Evas_Object *button = elm_button_add(parent);
	if (!button) {
		controller_print_message(DLOG_ERROR, "elm_button_add() failed.");
		return NULL;
	}

	elm_object_part_content_set(parent, edj_part_name, button);
	elm_object_text_set(button, caption);

	evas_object_smart_callback_add(button, "clicked", on_click_cb, data);

	evas_object_show(button);

	return button;
}

static Evas_Object *__create_entry(Evas_Object *parent, const char *edj_part_name)
{
	Evas_Object *entry = elm_entry_add(parent);
	if (!entry) {
		controller_print_message(DLOG_ERROR, "elm_entry_add() failed.");
		return NULL;
	}

	elm_entry_line_wrap_set(entry, ELM_WRAP_NONE);
	elm_entry_single_line_set(entry, EINA_TRUE);
	elm_entry_scrollable_set(entry, EINA_TRUE);
	elm_object_part_content_set(parent, edj_part_name, entry);
	evas_object_show(entry);

	return entry;
}

static Evas_Object *__create_label(Evas_Object *parent, const char *edj_part_name)
{
	Evas_Object *label = elm_label_add(parent);
	if (!label) {
		controller_print_message(DLOG_ERROR, "elm_label_add() failed.");
		return NULL;
	}

	elm_label_line_wrap_set(label, ELM_WRAP_WORD);
	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(label, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_part_content_set(parent, edj_part_name, label);
	evas_object_show(label);

	return label;
}

static bool __create_app_launch_panel(viewdata_s *vd)
{
	vd->controls_app_launch_panel_layout = __create_layout(EDJ_FILE_CONTROLS_APP_LAUNCH_PANEL,
															EDJ_GROUP_CONTROLS_APP_LAUNCH_PANEL,
															PART_CONTROLS_PANEL_APP_LAUNCH_PANEL,
															vd->controls_panel_layout);
	if (!vd->controls_app_launch_panel_layout)
		return false;

	vd->controls_app_launch_box = __create_box(vd->controls_app_launch_panel_layout,
												PART_CONTROLS_APP_LAUNCH_PANEL_OPERATION_RADIOS);
	if (!vd->controls_app_launch_box) {
		evas_object_del(vd->controls_app_launch_panel_layout);
		return false;
	}

	int i;
	for (i = 0; i < APP_CONTROL_OPERATION_COUNT; i++) {
		const char *operation_caption = model_get_operation_caption(i);
		if (!operation_caption)
			continue;

		vd->controls_app_launch_operation_rb[i] = __create_radio_button(vd->controls_app_launch_box,
																		operation_caption,
																		&(vd->radio_button_operation_selected),
																		(bool)(i == 0));
		if (!vd->controls_app_launch_operation_rb[i])
			continue;
	}

	vd->controls_app_launch_search_bt = __create_button(vd->controls_app_launch_panel_layout,
														PART_CONTROLS_APP_LAUNCH_PANEL_SEARCH_BUTTON,
														"Search",
														__on_controls_app_launch_search_button_cb,
														(void*)vd);
	if (!vd->controls_app_launch_search_bt) {
		evas_object_del(vd->controls_app_launch_panel_layout);
		return false;
	}

	vd->controls_app_launch_execute_bt = __create_button(vd->controls_app_launch_panel_layout,
														PART_CONTROLS_APP_LAUNCH_PANEL_LAUNCH_BUTTON,
														"Execute",
														__on_controls_app_launch_execute_button_cb,
														NULL);
	if (!vd->controls_app_launch_execute_bt) {
		evas_object_del(vd->controls_app_launch_panel_layout);
		return false;
	}
	view_disable_button(vd, BUTTON_EXECUTE, EINA_TRUE);

	vd->controls_app_launch_terminate_bt = __create_button(vd->controls_app_launch_panel_layout,
														PART_CONTROLS_APP_LAUNCH_PANEL_TERMINATE_BUTTON,
														"Kill",
														__on_controls_app_launch_terminate_button_cb,
														NULL);
	if (!vd->controls_app_launch_terminate_bt) {
		evas_object_del(vd->controls_app_launch_panel_layout);
		return false;
	}
	view_disable_button(vd, BUTTON_KILL, EINA_TRUE);

	return true;
}

static bool __create_message_compose_panel(viewdata_s *vd)
{
	vd->controls_message_compose_panel_layout = __create_layout(EDJ_FILE_CONTROLS_MESSAGE_COMPOSE_PANEL,
																EDJ_GROUP_CONTROLS_MESSAGE_COMPOSE_PANEL,
																PART_CONTROLS_PANEL_MESSAGE_COMPOSE,
																vd->controls_panel_layout);

	if (!vd->controls_message_compose_panel_layout)
		return false;

	vd->controls_message_compose_address_to_entry = __create_entry(vd->controls_message_compose_panel_layout,
																PART_MESSAGE_COMPOSE_ADDRESS_TO_EDIT_PANEL);
	if (!vd->controls_message_compose_address_to_entry) {
		evas_object_del(vd->controls_message_compose_panel_layout);
		return false;
	}

	vd->controls_message_compose_text_label = __create_label(vd->controls_message_compose_panel_layout,
															PART_MESSAGE_COMPOSE_TEXT_PANEL);
	if (!vd->controls_message_compose_text_label) {
		evas_object_del(vd->controls_message_compose_panel_layout);
		return false;
	}

	elm_object_text_set(vd->controls_message_compose_text_label, DEFAULT_TEXT_MESSAGE);

	vd->controls_message_compose_send_bt = __create_button(vd->controls_message_compose_panel_layout,
														PART_MESSAGE_COMPOSE_SEND_PANEL,
														"Send message",
														__on_controls_message_compose_send_button_cb,
														(void*)vd);
	if (!vd->controls_message_compose_send_bt) {
		evas_object_del(vd->controls_message_compose_panel_layout);
		return false;
	}

	return true;
}

static bool __create_controls_panel(viewdata_s *vd)
{
	vd->controls_panel_layout = __create_layout(EDJ_FILE_CONTROLS_PANEL,
												EDJ_GROUP_CONTROLS_PANEL,
												PART_MAIN_CONTROLS_PANEL,
												vd->main_layout);
	if (!vd->controls_panel_layout)
		return false;

	if (!__create_app_launch_panel(vd))
		return false;

	if (!__create_message_compose_panel(vd))
		return false;

	return true;
}

bool view_create_popup_window(void)
{
	Evas_Object *box = NULL;

	viewdata->popup_window = elm_popup_add(viewdata->win);
	if (!viewdata->popup_window) {
		controller_print_message(DLOG_ERROR, "elm_popup_add() failed.");
		return false;
	}

	box = __create_box(viewdata->popup_window, "default");
	if (!box) {
		evas_object_del(viewdata->popup_window);
		viewdata->popup_window = NULL;
		return false;
	}

	elm_object_part_text_set(viewdata->popup_window, "title,text", "Select an application");

	if (!__create_button(viewdata->popup_window, "button1", "Select", __on_popup_window_button_select_cb, viewdata) ||
		!__create_button(viewdata->popup_window, "button2", "Cancel", __on_popup_window_button_cancel_cb, viewdata)) {
		evas_object_del(viewdata->popup_window);
		viewdata->popup_window = NULL;
		return false;
	}

	evas_object_show(viewdata->popup_window);

	return true;
}

bool view_add_popup_window_radio_button(const char *caption, bool create_new_group)
{
	Evas_Object *box = elm_object_part_content_get(viewdata->popup_window, "default");
	if (!box) {
		controller_print_message(DLOG_ERROR, "elm_object_part_content_get() failed.");
		return false;
	}

	return (__create_radio_button(box, caption, &viewdata->radio_button_application_selected, create_new_group) != NULL);
}

bool view_is_shown_popup_window(void)
{
	return (viewdata->popup_window != NULL);
}

void view_disable_button(viewdata_s *vd, appcontrol_button_type type, Eina_Bool disabled)
{
	Evas_Object *button = NULL;

	switch(type) {
	case BUTTON_SEARCH:
		button = vd->controls_app_launch_search_bt;
		break;
	case BUTTON_EXECUTE:
		button = vd->controls_app_launch_execute_bt;
		break;
	case BUTTON_KILL:
		button = vd->controls_app_launch_terminate_bt;
		break;
	}

	elm_object_disabled_set(button, disabled);
}

void view_set_app_selected(viewdata_s *vd, const char *app_id)
{
	if (!app_id) {
		elm_object_part_text_set(vd->controls_app_launch_panel_layout, PART_CONTROLS_APP_LAUNCH_PANEL_SELECTED_APP, "No app selected.");
		view_disable_button(vd, BUTTON_EXECUTE, EINA_TRUE);
		view_disable_button(vd, BUTTON_KILL, EINA_TRUE);
	}

	if (vd->controls_app_launch_panel_layout)
		elm_object_part_text_set(vd->controls_app_launch_panel_layout, PART_CONTROLS_APP_LAUNCH_PANEL_SELECTED_APP, app_id);
}

bool view_create(viewdata_s *vd)
{
	if (!vd)
		return false;

	viewdata = vd;

	vd->win = __create_main_window(vd);
	if (!vd->win)
		return false;

	vd->conform = __create_conformant(vd->win);
	if (!vd->conform) {
		evas_object_del(vd->win);
		return false;
	}

	vd->main_layout = __create_layout(EDJ_FILE_MAIN,
										EDJ_GROUP_MAIN,
										NULL,
										vd->conform);
	if (!vd->main_layout) {
		evas_object_del(vd->win);
		return false;
	}

	if (!__create_controls_panel(vd)) {
		evas_object_del(vd->win);
		return false;
	}

	evas_object_show(vd->win);

	return true;
}
