/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"

/* search layout app data */
typedef struct _search_data
{
	Evas_Object *editfield;
	Evas_Object *genlist;
	Elm_Genlist_Item_Class* itc;
} search_data;

/* genlist item data */
typedef struct _text_item
{
	char text[100];
} text_item;

/* text database,
 * if you want you can change this to another database such as SQLlite, Oracle and so on */
static const int text_database_size = 20;
static const char *text_database[] =
{
	"orange","banana",
	"tomato","mango",
	"amond","blueberry",
	"melon","peach",
	"cherry","sweety",
	"pear", "avocado",
	"grapes","grapefruit",
	"plum","apricot",
	"kiwi","guava",
	"coconut","papaya"
};

/* genlist is updated by search text
 * if you want you can use another algorithm */
void
update_genlist(const char* search_text, void* data)
{
	int i;
	search_data *sd = data;
	elm_genlist_clear(sd->genlist);

	if(!strcmp(search_text, ""))
	{
		for(i = 0; i < text_database_size; i++)
		{
			text_item *text_data = calloc(sizeof(text_item),1);
			sprintf(text_data->text,"%s",text_database[i]);
			elm_genlist_item_append(sd->genlist, sd->itc, text_data, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
		}
	}
	else
	{
		for(i = 0; i < text_database_size; i++)
		{
			if(strstr(text_database[i], search_text) != NULL)
			{
				text_item *text_data = calloc(sizeof(text_item),1);
				sprintf(text_data->text,"%s",text_database[i]);
				elm_genlist_item_append(sd->genlist, sd->itc, text_data, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
			}
		}
	}
}

/* genlist item callbacks */
static char*
gl_text_get_cb(void *data, Evas_Object *obj, const char *part)
{
	text_item *text_data = data;
	return strdup(text_data->text);
}

static void
gl_del_cb(void *data, Evas_Object *obj)
{
	text_item *text_data = data;
	free(text_data);
}

/* editfield callbacks */
static void
editfield_focused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;
	elm_object_signal_emit(editfield, "elm,state,focused", "");

	if (!elm_entry_is_empty(obj))
		elm_object_signal_emit(editfield, "elm,action,show,button", "");
}

static void
editfield_unfocused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;
	elm_object_signal_emit(editfield, "elm,state,unfocused", "");
	elm_object_signal_emit(editfield, "elm,action,hide,button", "");
}

static void
editfield_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	search_data *sd = data;
	Evas_Object *editfield = sd->editfield;
	const char *search_text = elm_entry_entry_get(obj);

	if (!elm_entry_is_empty(obj) && elm_object_focus_get(obj))
		elm_object_signal_emit(editfield, "elm,action,show,button", "");
	else
		elm_object_signal_emit(editfield, "elm,action,hide,button", "");

	update_genlist(search_text, sd);
}

static void
editfield_clear_button_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *entry = (Evas_Object *)data;
	elm_entry_entry_set(entry, "");
}

Evas_Object*
create_search_field(Evas_Object* parent, void* search_data)
{
	Evas_Object *editfield, *entry, *button;

	/* add an editfield */
	editfield = elm_layout_add(parent);
	elm_layout_theme_set(editfield, "layout", "editfield", "singleline");

	entry = elm_entry_add(editfield);
	elm_entry_single_line_set(entry, EINA_TRUE);
	elm_entry_scrollable_set(entry, EINA_TRUE);
	elm_object_part_text_set(entry, "elm.guide", "Search");
	evas_object_smart_callback_add(entry, "focused", editfield_focused_cb, editfield);
	evas_object_smart_callback_add(entry, "unfocused", editfield_unfocused_cb, editfield);
	evas_object_smart_callback_add(entry, "changed", editfield_changed_cb, search_data);
	evas_object_smart_callback_add(entry, "preedit,changed", editfield_changed_cb, editfield);
	elm_object_part_content_set(editfield, "elm.swallow.content", entry);

	button = elm_button_add(editfield);
	elm_object_style_set(button, "editfield_clear");
	evas_object_smart_callback_add(button, "clicked", editfield_clear_button_clicked_cb, entry);
	elm_object_part_content_set(editfield, "elm.swallow.button", button);

	return editfield;
}

Evas_Object*
create_text_list(Evas_Object* layout)
{
	/* add an genlist to the layout */
	Evas_Object* genlist;
	genlist = elm_genlist_add(layout);

	return genlist;
}

Elm_Genlist_Item_Class*
create_item_class()
{
	/* set genlist item style and callbacks */
	Elm_Genlist_Item_Class* itc;
	itc = elm_genlist_item_class_new();
	itc->item_style = "default";
	itc->func.content_get = NULL;
	itc->func.text_get = gl_text_get_cb;
	itc->func.del = gl_del_cb;

	return itc;
}

static void
search_layout_del_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	search_data *sd = data;
	sd->editfield = NULL;
	sd->genlist = NULL;
	sd->itc = NULL;
	free(sd);
}

void
search_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Evas_Object *layout;
	Evas_Object *nf = ad->nf;
	int i;

	/* create an search app data */
	search_data *sd = malloc(sizeof(search_data));

	/* create an layout */
	layout = elm_layout_add(nf);
	elm_layout_file_set(layout, APP_VIEW_EDJ, "search_layout");
	evas_object_event_callback_add(layout, EVAS_CALLBACK_DEL, search_layout_del_cb, sd);

	/* create an entry */
	sd->editfield = create_search_field(layout, sd);
	elm_object_part_content_set(layout, "elm.swallow.searchbar", sd->editfield);

	/* create an genlist */
	sd->genlist = create_text_list(layout);
	elm_object_part_content_set(layout,"elm.swallow.list", sd->genlist);

	/* create an genlist item class */
	sd->itc = create_item_class();

	/* initialize list data */
	for(i = 0; i < text_database_size; i++)
	{
		/* declare genlist item and set text of the item, then append to the genlist */
		text_item *text_data = calloc(sizeof(text_item),1);
		sprintf(text_data->text,"%s",text_database[i]);
		elm_genlist_item_append(sd->genlist, sd->itc, text_data, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
	}

	elm_naviframe_item_push(nf, "Search", NULL, NULL, layout, NULL);
}
