/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"

static void
searchfield_focused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;
	elm_object_signal_emit(editfield, "elm,state,focused", "");

	if (!elm_entry_is_empty(obj))
		elm_object_signal_emit(editfield, "elm,action,show,button", "");
}

static void
searchfield_unfocused_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;
	elm_object_signal_emit(editfield, "elm,state,unfocused", "");
	elm_object_signal_emit(editfield, "elm,action,hide,button", "");
}

static void
searchfield_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *editfield = (Evas_Object *)data;

	if (!elm_entry_is_empty(obj) && elm_object_focus_get(obj))
		elm_object_signal_emit(editfield, "elm,action,show,button", "");
	else
		elm_object_signal_emit(editfield, "elm,action,hide,button", "");
}

static void
searchfield_clear_button_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *entry = (Evas_Object *)data;

	elm_entry_entry_set(entry, "");
}

static Evas_Object *
create_searchfield_layout(Evas_Object *parent)
{
	Evas_Object *searchfield, *entry, *button;

	searchfield = elm_layout_add(parent);
	elm_layout_theme_set(searchfield, "layout", "searchfield", "singleline");
	evas_object_size_hint_align_set(searchfield, EVAS_HINT_FILL, 0.0);
	evas_object_size_hint_weight_set(searchfield, EVAS_HINT_EXPAND, 0.0);

	entry = elm_entry_add(searchfield);
	elm_entry_single_line_set(entry, EINA_TRUE);
	elm_entry_scrollable_set(entry, EINA_TRUE);
	elm_entry_input_panel_return_key_type_set(entry, ELM_INPUT_PANEL_RETURN_KEY_TYPE_SEARCH);
	elm_object_part_text_set(entry, "elm.guide", "Search Field");
	evas_object_smart_callback_add(entry, "focused", searchfield_focused_cb, searchfield);
	evas_object_smart_callback_add(entry, "unfocused", searchfield_unfocused_cb, searchfield);
	evas_object_smart_callback_add(entry, "changed", searchfield_changed_cb, searchfield);
	evas_object_smart_callback_add(entry, "preedit,changed", searchfield_changed_cb, searchfield);
	elm_object_part_content_set(searchfield, "elm.swallow.content", entry);

	button = elm_button_add(searchfield);
	elm_object_style_set(button, "editfield_clear");
	evas_object_smart_callback_add(button, "clicked", searchfield_clear_button_clicked_cb, entry);
	elm_object_part_content_set(searchfield, "elm.swallow.button", button);

	return searchfield;
}

static Evas_Object*
create_nocontent_layout(Evas_Object *parent, const char *text)
{
	Evas_Object *layout, *scroller;

	/* Scroller */
	scroller = elm_scroller_add(parent);
	elm_scroller_bounce_set(scroller, EINA_FALSE, EINA_TRUE);
	elm_scroller_policy_set(scroller, ELM_SCROLLER_POLICY_OFF, ELM_SCROLLER_POLICY_AUTO);

	/* NoContent Layout */
	layout = elm_layout_add(scroller);
	elm_layout_theme_set(layout, "layout", "nocontents", "default");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(layout, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_object_part_text_set(layout, "elm.text", "No Contents");

	elm_object_content_set(scroller, layout);

	return scroller;
}

static Evas_Object *
create_searchbar_view(appdata_s *ad)
{
	Evas_Object *main_scroller, *main_box, *searchfield, *layout;

	main_scroller = elm_scroller_add(ad->nf);
	elm_scroller_bounce_set(main_scroller, EINA_FALSE, EINA_TRUE);
	evas_object_size_hint_weight_set(main_scroller, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(main_scroller, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(main_scroller);

	main_box = elm_box_add(main_scroller);
	evas_object_size_hint_align_set(main_box, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(main_box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_show(main_box);

	searchfield = create_searchfield_layout(main_box);
	elm_box_pack_end(main_box, searchfield);
	evas_object_show(searchfield);

	layout = create_nocontent_layout(main_box, "No Contents");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(layout, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_pack_end(main_box, layout);
	evas_object_show(layout);

	elm_object_content_set(main_scroller, main_box);

	return main_scroller;
}

void
searchbar_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	Evas_Object *searchbar_view;

	/* Search Bar view */
	searchbar_view = create_searchbar_view(ad);
	elm_naviframe_item_push(ad->nf, "Search Bar", NULL, NULL, searchbar_view, NULL);
}
