#include <package_manager.h>
#include <app_manager.h>
#include <badge.h>
#include "$(appName)-ref.h"
#include "model.h"
#include "controller.h"

static const char*
__model_error_code_to_string(int error_code)
{
	switch (error_code) {
		case BADGE_ERROR_NONE:
			return "BADGE_ERROR_NONE";
		case BADGE_ERROR_INVALID_PARAMETER:
			return "BADGE_ERROR_INVALID_PARAMETER";
		case BADGE_ERROR_OUT_OF_MEMORY:
			return "BADGE_ERROR_OUT_OF_MEMORY";
		case BADGE_ERROR_PERMISSION_DENIED:
			return "BADGE_ERROR_PERMISSION_DENIED";
		case BADGE_ERROR_IO_ERROR:
			return "BADGE_ERROR_IO_ERROR";
		case BADGE_ERROR_FROM_DB:
			return "BADGE_ERROR_FROM_DB";
		case BADGE_ERROR_ALREADY_EXIST:
			return "BADGE_ERROR_ALREADY_EXIST";
		case BADGE_ERROR_FROM_DBUS:
			return "BADGE_ERROR_FROM_DBUS";
		case BADGE_ERROR_NOT_EXIST:
			return "BADGE_ERROR_NOT_EXIST";
		case BADGE_ERROR_SERVICE_NOT_READY:
			return "BADGE_ERROR_SERVICE_NOT_READY";
		case BADGE_ERROR_INVALID_PACKAGE:
			return "BADGE_ERROR_INVALID_PACKAGE";
		default:
			return "Unknown error";
	}
}

static bool
model_package_info_create(const char *pkg_name, const char *app_id, pkginfo_s **pkginfo)
{
	*pkginfo = (pkginfo_s*)malloc(sizeof(pkginfo_s));
	if (!*pkginfo) {
		controller_log(DLOG_ERROR, "Memory allocation failed for pkginfo_s.");
		return false;
	}

	(*pkginfo)->pkg_name = strdup(pkg_name);
	(*pkginfo)->app_id = strdup(app_id);

	return true;
}

static bool
model_app_info_cb(app_info_h app_info, void *user_data)
{
	Eina_List **list = (Eina_List**)user_data;

	char *pkg_name = NULL;
	int ret = app_info_get_package(app_info, &pkg_name);
	if (ret != APP_MANAGER_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function app_info_get_package() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		return true;
	}

	bool nodisplay = true;
	ret = app_info_is_nodisplay(app_info, &nodisplay);
	if (ret != APP_MANAGER_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function app_info_is_nodisplay() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		return true;
	}

	if (nodisplay) {
		free(pkg_name);
		return true;
	}

	char *app_id = NULL;
	ret = app_info_get_app_id(app_info, &app_id);
	if (ret != APP_MANAGER_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function app_info_get_app_id() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		free(pkg_name);
		return true;
	}

	pkginfo_s *pkginfo = NULL;
	if (model_package_info_create(pkg_name, app_id, &pkginfo)) {
		*list = eina_list_append(*list, (void*)pkginfo);
	}

	free(app_id);
	free(pkg_name);

	return true;
}

bool
model_packages_list_get(Eina_List **list)
{
	*list = NULL;

	int ret = app_manager_foreach_app_info(model_app_info_cb, (void*)list);
	if (ret != APP_MANAGER_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function app_manager_foreach_app_info() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		return false;
	}

	return true;
}

void
model_packages_list_free(Eina_List *list)
{
	Eina_List *it = NULL;
	pkginfo_s *pkginfo = NULL;

	EINA_LIST_FOREACH(list, it, pkginfo) {
		if (!pkginfo) {
			continue;
		}

		if (pkginfo->pkg_name) {
			free(pkginfo->pkg_name);
			pkginfo->pkg_name = NULL;
		}
		if (pkginfo->app_id) {
			free(pkginfo->app_id);
			pkginfo->app_id = NULL;
		}
		free(pkginfo);
		pkginfo = NULL;
	}

	list = eina_list_free(list);
}

bool
model_packages_certificate_verify(const char *app_id)
{
	package_manager_compare_result_type_e cmp_res = PACKAGE_MANAGER_COMPARE_MISMATCH;

	int ret = package_manager_compare_app_cert_info(APP_ID, app_id, &cmp_res);
	if (ret != PACKAGE_MANAGER_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function package_manager_compare_app_cert_info() failed with error %d.", ret);
		return false;
	}

	return (cmp_res == PACKAGE_MANAGER_COMPARE_MATCH);
}

bool
model_badge_count_get(const char *app_id, unsigned int *badge_count)
{
	int ret = badge_get_count(app_id, badge_count);
	if (ret != BADGE_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function badge_get_count() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		return false;
	}

	return true;
}

bool
model_badge_count_set(const char *app_id, unsigned int badge_count)
{
	int ret = badge_set_count(app_id, badge_count);
	if (ret != BADGE_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function badge_set_count() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		return false;
	}

	return true;
}

bool
model_badge_display_get(const char *app_id, bool *display)
{
	unsigned int is_display = 0;

	*display = (bool)is_display;

	int ret = badge_get_display(app_id, &is_display);
	if (ret != BADGE_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function badge_get_display() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		return false;
	}

	*display = (bool)is_display;

	return true;
}

bool
model_badge_display_set(const char *app_id, bool display)
{
	unsigned int display_val = 0;

	if (display) {
		display_val = 1;
	}

	int ret = badge_set_display(app_id, display_val);
	if (ret != BADGE_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function badge_set_display() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		return false;
	}

	return true;
}

bool
model_badge_add(const char *app_id)
{
	int ret = badge_add(app_id);
	if (ret != BADGE_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function badge_add() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		return false;
	}

	return true;
}

bool
model_badge_remove(const char *app_id)
{
	int ret = badge_remove(app_id);
	if (ret != BADGE_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function badge_remove() failed with error %d (%s).", ret, __model_error_code_to_string(ret));
		return false;
	}

	return true;
}
