#include <tizen.h>
#include "$(appName)-ref.h"
#include "view.h"
#include "controller.h"
#include "defines.h"

#define BADGE_COUNT_MAX 2000

static viewdata_s *viewdata = NULL;
/*
 * We need to store the selected item's pointer because elm_genlist_selected_item_get() function does not work.
 */
static Elm_Object_Item *item_selected = NULL;

static void
view_badge_update_button_click_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s*)data;

	if (!item_selected) {
		controller_log(DLOG_ERROR, "No item is selected.");
		return;
	}

	pkginfo_s *pkginfo = (pkginfo_s*)elm_object_item_data_get(item_selected);
	if (!pkginfo) {
		controller_log(DLOG_ERROR, "Function elm_object_item_data_get() failed.");
		return;
	}

	unsigned int curr_badge_count = controller_badge_count_get(pkginfo->app_id);
	int new_badge_count = (int)elm_spinner_value_get(vd->badge_counter_spinner);

	if (curr_badge_count == new_badge_count) {
		return;
	} else if (curr_badge_count == 0) {
		controller_badge_new(pkginfo->app_id, new_badge_count);
	} else if (new_badge_count == 0) {
		controller_badge_remove(pkginfo->app_id);
	} else {
		controller_badge_update(pkginfo->app_id, new_badge_count);
	}

	elm_genlist_item_update(item_selected);
}

static void
view_available_apps_item_selected_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s*)data;

	item_selected = (Elm_Object_Item*)event_info;
	if (!item_selected) {
		controller_log(DLOG_ERROR, "No item is selected.");
		return;
	}

	pkginfo_s *pkginfo = (pkginfo_s*)elm_object_item_data_get(item_selected);
	if (!pkginfo) {
		controller_log(DLOG_ERROR, "Function elm_object_item_data_get() failed.");
		return;
	}

	unsigned int badge_count = controller_badge_count_get(pkginfo->app_id);

	elm_spinner_value_set(vd->badge_counter_spinner, (double)badge_count);

	elm_genlist_item_update(item_selected);

	elm_object_signal_emit(vd->layout_badge_update_panel,
							SIGNAL_BADGE_UPDATE_PANEL_APP_NAME_SHOW,
							PART_BADGE_UPDATE_PANEL_APP_NAME);

	elm_object_part_text_set(viewdata->layout_badge_update_panel,
							PART_BADGE_UPDATE_PANEL_APP_NAME,
							pkginfo->app_id);
}

static void
view_available_apps_item_unselected_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s*)data;

	elm_object_signal_emit(vd->layout_badge_update_panel,
							SIGNAL_BADGE_UPDATE_PANEL_APP_NAME_HIDE,
							PART_BADGE_UPDATE_PANEL_APP_NAME);

	elm_object_part_text_set(viewdata->layout_badge_update_panel,
							PART_BADGE_UPDATE_PANEL_APP_NAME,
							TEXT_LABEL_BADGE_UPDATE_PANEL_APP_NAME_DEFAULT);

	item_selected = NULL;
}

void
view_win_delete_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

void
view_layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = data;
	/* Let window go to hide state. */
	elm_win_lower(vd->win);
}

static void
app_resource_get(const char *edj_file_in, char *edj_path_out, int edj_path_max)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, edj_path_max, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

static Evas_Object*
view_window_create(void)
{
	Evas_Object *win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	if (!win) {
		controller_log(DLOG_ERROR, "Function elm_win_util_standard_add() failed.");
		return NULL;
	}

	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);
	elm_win_autodel_set(win, EINA_TRUE);

	evas_object_smart_callback_add(win, "delete,request", view_win_delete_request_cb, NULL);

	return win;
}

static Evas_Object*
view_conformant_create(Evas_Object *parent)
{
	if (!parent) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	Evas_Object *conform = elm_conformant_add(parent);
	if (!conform) {
		controller_log(DLOG_ERROR, "Function elm_conformant_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(parent, conform);
	evas_object_show(conform);

	return conform;
}

static Evas_Object*
view_layout_create(Evas_Object *parent, const char *edj_file_name, const char *edj_group, const char *target_part_name)
{
	if (!parent || !edj_file_name || !edj_group) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	char edj_path[PATH_MAX] = {0, };

	app_resource_get(edj_file_name, edj_path, (int)PATH_MAX);

	Evas_Object *layout = elm_layout_add(parent);
	if (!layout) {
		controller_log(DLOG_ERROR, "Function elm_layout_add() failed.");
		return NULL;
	}

	if (!elm_layout_file_set(layout, edj_path, edj_group)) {
		controller_log(DLOG_ERROR, "Function elm_layout_file_set() failed.");
		evas_object_del(layout);
		return NULL;
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	if (!target_part_name) {
		elm_object_content_set(parent, layout);
	} else {
		elm_object_part_content_set(parent, target_part_name, layout);
	}

	return layout;
}

static Evas_Object*
view_spinner_create(Evas_Object *parent)
{
	if (!parent) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	Evas_Object *spinner = elm_spinner_add(parent);
	if (!spinner) {
		controller_log(DLOG_ERROR, "Function elm_spinner_add() failed.");
		return NULL;
	}

	elm_spinner_editable_set(spinner, EINA_FALSE);

	elm_spinner_min_max_set(spinner, 0, BADGE_COUNT_MAX);
	elm_object_part_content_set(parent, PART_BADGE_UPDATE_PANEL_BADGE_COUNTER_EDIT, spinner);

	evas_object_show(spinner);

	return spinner;
}

static Evas_Object*
view_button_create(Evas_Object *parent, const char *target_part_name, const char *caption, Evas_Smart_Cb on_click_cb, void *data)
{
	if (!parent || !target_part_name || !caption) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	Evas_Object *button = elm_button_add(parent);
	if (!button) {
		controller_log(DLOG_ERROR, "Function elm_button_add() failed.");
		return NULL;
	}

	elm_object_part_content_set(parent, target_part_name, button);
	elm_object_text_set(button, caption);

	evas_object_smart_callback_add(button, "clicked", on_click_cb, data);

	evas_object_show(button);

	return button;
}

static Evas_Object*
view_genlist_create(Evas_Object *parent, const char *target_part_name)
{
	if (!parent || !target_part_name) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	Evas_Object *list = elm_genlist_add(parent);
	if (!list) {
		controller_log(DLOG_ERROR, "Function elm_genlist_add() failed.");
		return NULL;
	}

	evas_object_size_hint_weight_set(list, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(parent, target_part_name, list);

	evas_object_show(list);

	return list;
}

static char*
view_genlist_item_label_get(void *data, Evas_Object *obj, const char *part)
{
	pkginfo_s *pkginfo = (pkginfo_s*)data;

	if (!pkginfo || !pkginfo->pkg_name || !pkginfo->app_id) {
		return NULL;
	}

	if (!strcmp(part, "elm.text")) {
		return strdup(pkginfo->pkg_name);
	} else if (!strcmp(part, "elm.text.sub")) {
		unsigned int badge_count = controller_badge_count_get(pkginfo->app_id);
		char buff[255] = {0,};

		snprintf(buff, sizeof(buff), "Badge count = %u", badge_count);

		return strdup(buff);
	}

	return NULL;
}

bool
view_genlist_item_add(pkginfo_s *pkginfo)
{
	if (!pkginfo) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return false;
	}

	static Elm_Genlist_Item_Class *itc = NULL;

	if (!viewdata->available_apps_list) {
		controller_log(DLOG_ERROR, "Function view_genlist_item_add() failed.");
		return false;
	}

	if (!itc) {
		itc = elm_genlist_item_class_new();
		if (!itc) {
			controller_log(DLOG_ERROR, "Function elm_genlist_item_class_new() failed.");
			return false;
		}

		itc->item_style = "double_label";
		itc->func.text_get = view_genlist_item_label_get;
		itc->func.content_get = NULL;
		itc->func.state_get = NULL;
		itc->func.del = NULL;
	}

	Elm_Object_Item *item = elm_genlist_item_append(viewdata->available_apps_list, itc, (void*)pkginfo, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
	if (!item) {
		controller_log(DLOG_ERROR, "Function elm_genlist_item_append() failed.");
		return false;
	}

	elm_object_item_data_set(item, (void*)pkginfo);

	return true;
}

static bool
view_main_panel_create(viewdata_s *vd)
{
	vd->win = view_window_create();
	if (!vd->win) {
		return false;
	}

	vd->conform = view_conformant_create(vd->win);
	if (!vd->conform) {
		return false;
	}

	vd->layout_main_panel = view_layout_create(vd->conform, EDJ_MAIN_FILE_NAME, GROUP_MAIN, NULL);
	if (!vd->layout_main_panel) {
		return false;
	}
	eext_object_event_callback_add(vd->layout_main_panel, EEXT_CALLBACK_BACK, view_layout_back_cb, (void*)vd);

	return true;
}

static bool
view_update_panel_create(viewdata_s *vd)
{
	vd->layout_badge_update_panel = view_layout_create(vd->layout_main_panel,
														EDJ_BADGE_UPDATE_PANEL_FILE_NAME,
														GROUP_BADGE_UPDATE_PANEL,
														PART_MAIN_PANEL_BADGE_UPDATE_PANEL);
	if (!vd->layout_badge_update_panel) {
		return false;
	}

	vd->badge_counter_spinner = view_spinner_create(vd->layout_badge_update_panel);
	if (!vd->badge_counter_spinner) {
		return false;
	}

	return true;
}

static bool
view_available_apps_panel_create(viewdata_s *vd)
{
	vd->layout_available_apps_panel = view_layout_create(vd->layout_main_panel,
														EDJ_APPS_LIST_PANEL_FILE_NAME,
														GROUP_APPS_LIST_PANEL,
														PART_MAIN_PANEL_APPS_LIST_PANEL);
	if (!vd->layout_available_apps_panel) {
		return false;
	}

	vd->available_apps_list = view_genlist_create(vd->layout_available_apps_panel,
													PART_APPS_LIST_PANEL_LIST);
	if (!vd->available_apps_list) {
		return false;
	}

	evas_object_smart_callback_add(vd->available_apps_list, "selected", view_available_apps_item_selected_cb, (void*)vd);
	evas_object_smart_callback_add(vd->available_apps_list, "unselected", view_available_apps_item_unselected_cb, (void*)vd);

	return true;
}

static bool
view_buttons_panel_create(viewdata_s *vd)
{
	vd->layout_buttons_panel = view_layout_create(vd->layout_main_panel,
													EDJ_BUTTONS_PANEL_FILE_NAME,
													GROUP_BUTTONS_PANEL,
													PART_MAIN_PANEL_BUTTON_PANEL);
	if (!vd->layout_buttons_panel) {
		return false;
	}

	vd->update_button = view_button_create(vd->layout_buttons_panel,
											PART_BUTTONS_PANEL_BUTTON,
											"Update",
											view_badge_update_button_click_cb,
											(void*)vd);
	if (!vd->update_button) {
		return false;
	}

	return true;
}

bool
view_base_gui_create(viewdata_s *vd)
{
	viewdata = vd;

	if (!view_main_panel_create(vd) ||
		!view_update_panel_create(vd) ||
		!view_available_apps_panel_create(vd) ||
		!view_buttons_panel_create(vd)) {
		evas_object_del(vd->win);
		return false;
	}

	evas_object_show(vd->win);

	return true;
}
