#include <stdio.h>
#include <Eina.h>
#include "$(appName)-ref.h"
#include "controller/controller.h"
#include "model/model-message-port.h"
#include "model/model-bundle.h"

#define MSG_BUFF_SIZE 256
#define BUNDLE_HEADER_KEY "HEADER"
#define BUNDLE_HEADER_DATA_KEY "HEADER_DATA"

static const char *__data_sink_get_value_type_name(int value_type)
{
	const char *type_name = "";

	switch (value_type) {
	case BUNDLE_TYPE_NONE:
		type_name = "none";
		break;
	case BUNDLE_TYPE_ANY:
		type_name = "any";
		break;
	case BUNDLE_TYPE_STR:
		type_name = "string";
		break;
	case BUNDLE_TYPE_STR_ARRAY:
		type_name = "strings array";
		break;
	case BUNDLE_TYPE_BYTE:
		type_name = "byte";
		break;
	case BUNDLE_TYPE_BYTE_ARRAY:
		type_name = "bytes array";
		break;
	default:
		type_name = "unknown";
		break;
	}

	return type_name;
}

static void controller_data_sink_bundle_foreach_cb(const char *key,
													const int type,
													const bundle_keyval_t *kv,
													void *user_data)
{
	int value_type = 0;
	char buff[MSG_BUFF_SIZE] = {0,};
	const char *type_name = NULL;
	void *value = NULL;

	if (controller_check_same_string(key, BUNDLE_HEADER_KEY) ||
		controller_check_same_string(key, BUNDLE_HEADER_DATA_KEY))
		return;

	if (model_bundle_get_keyval_type(kv, &value_type)) {
		type_name = __data_sink_get_value_type_name(value_type);
	} else {
		type_name = "unknown";
	}

	if (model_bundle_get_keyval_basic_val(kv, &value)) {
		if (value_type == BUNDLE_TYPE_BYTE) {
			snprintf(buff,
						MSG_BUFF_SIZE,
						"     - %s = %d (%s)",
						key,
						*((int *)value),
						type_name);
		} else if (value_type == BUNDLE_TYPE_STR) {
			snprintf(buff,
						MSG_BUFF_SIZE,
						"     - %s = %s (%s)",
						key,
						(char *)value,
						type_name);
		}
	} else {
		snprintf(buff,
					MSG_BUFF_SIZE,
					"     - %s = N/A (%s)",
					key,
					type_name);
	}

	controller_append_message(buff);
}

static void controller_data_sink_message_received_cb(int local_port_id,
													const char *remote_app_id,
													const char *remote_port,
													bool trusted_remote_port,
													bundle *message,
													void *user_data)
{
	char buff[MSG_BUFF_SIZE] = {0,};
	int items_count = 0;
	int header_value = 0;
	const char **header_data = NULL;
	int header_size = 0;

	if (!message) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return;
	}

	controller_append_message("---=== MSG RECEIVED ===---");

	if (model_bundle_get_count(message, &items_count)) {
		snprintf(buff, MSG_BUFF_SIZE, "Bundle items count: %d", items_count);
	} else {
		snprintf(buff, MSG_BUFF_SIZE, "Bundle items count: ERROR");
	}

	controller_append_message(buff);

	if (!model_bundle_get_byte(message, BUNDLE_HEADER_KEY, &header_value))
		return;

	controller_append_message("Header data:");

	if ((bool)header_value) {
		if (!model_bundle_get_string_array(message,
											BUNDLE_HEADER_DATA_KEY,
											&header_data,
											&header_size))
			return;

		int i;
		for (i = 0; i < header_size; i++) {
			int val_type = 0;
			model_bundle_get_type(message, header_data[i], &val_type);
			const char *type_name = __data_sink_get_value_type_name(
																	val_type);

			snprintf(buff,
						MSG_BUFF_SIZE,
						"     - Key: %s (%s)",
						header_data[i],
						type_name);

			controller_append_message(buff);
		}
	} else {
		controller_append_message("     - N/A");
	}

	controller_append_message("Data:");

	if (!model_bundle_foreach(message,
								controller_data_sink_bundle_foreach_cb)) {
		controller_append_message("     - N/A");
		return;
	}
}

bool controller_data_sink_create(void)
{
	if (model_message_port_check_exists())
		return true;

	return model_message_port_create(controller_data_sink_message_received_cb);
}

bool controller_data_sink_destroy(void)
{
	if (!model_message_port_check_exists())
		return true;

	return model_message_port_destroy();
}
