#include "$(appName)-ref.h"
#include "model/model-bundle.h"
#include "controller/controller.h"

static const char *__error_code_to_string(int error_code)
{
	switch (error_code) {
	case BUNDLE_ERROR_NONE:
		return "BUNDLE_ERROR_NONE";
	case BUNDLE_ERROR_OUT_OF_MEMORY:
		return "BUNDLE_ERROR_OUT_OF_MEMORY";
	case BUNDLE_ERROR_INVALID_PARAMETER:
		return "BUNDLE_ERROR_INVALID_PARAMETER";
	case BUNDLE_ERROR_KEY_NOT_AVAILABLE:
		return "BUNDLE_ERROR_KEY_NOT_AVAILABLE";
	case BUNDLE_ERROR_KEY_EXISTS:
		return "BUNDLE_ERROR_KEY_EXISTS";
	default:
		return "Unknown error";
	}
}

bool model_bundle_create(bundle **bundle_obj)
{
	*bundle_obj = bundle_create();
	if (!bundle_obj) {
		controller_log(DLOG_ERROR, "Function bundle_create() failed.");
		return false;
	}

	return true;
}

bool model_bundle_destroy(bundle *bundle_obj)
{
	int ret = bundle_free(bundle_obj);
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
						"Function bundle_free() failed with error: %d (%s).",
						ret,
						__error_code_to_string(ret));
		return false;
	}

	return true;
}

bool model_bundle_get_count(bundle *bundle_obj, int *count)
{
	*count = bundle_get_count(bundle_obj);
	int ret = get_last_result();
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
					"Function bundle_get_count() failed with error: %d (%s).",
					ret,
					__error_code_to_string(ret));
		return false;
	}

	return true;
}

bool model_bundle_get_type(bundle *bundle_obj, const char *key, int *type)
{
	*type = bundle_get_type(bundle_obj, key);
	int ret = get_last_result();
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
					"Function bundle_get_type() failed with error: %d (%s).",
					ret,
					__error_code_to_string(ret));
		return false;
	}

	return true;
}

bool model_bundle_foreach(bundle *bundle_obj, bundle_iterator_t func_cb)
{
	if (!func_cb) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return false;
	}

	bundle_foreach(bundle_obj, func_cb, NULL);

	return true;
}

bool model_bundle_add_string(bundle **bundle_obj,
								const char *key,
								const char *value)
{
	int ret = bundle_add_str(*bundle_obj, key, value);
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
						"Function bundle_add_str() failed with error: %d (%s).",
						ret,
						__error_code_to_string(ret));
		return false;
	}

	return true;
}

bool model_bundle_get_string(bundle *bundle_obj, const char *key, char **value)
{
	*value = NULL;

	int ret = bundle_get_str(bundle_obj, key, value);
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
						"Function bundle_get_str() failed with error: %d (%s).",
						ret,
						__error_code_to_string(ret));
		return false;
	}

	return true;
}

bool model_bundle_add_string_array(bundle **bundle_obj,
										const char *key,
										char **value,
										int value_count)
{
	if (!value) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return false;
	}

	int ret = bundle_add_str_array(*bundle_obj,
										key,
										(const char **)value,
										value_count);
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
				"Function bundle_add_str_array() failed with error: %d (%s).",
				ret,
				__error_code_to_string(ret));
		return false;
	}

	return true;
}

bool model_bundle_get_string_array(bundle *bundle_obj,
										const char *key,
										const char ***value,
										int *value_len)
{
	*value_len = 0;
	*value = bundle_get_str_array(bundle_obj, key, value_len);
	int ret = get_last_result();
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
				"Function bundle_get_str_array() failed with error: %d (%s).",
				ret,
				__error_code_to_string(ret));
		return false;
	}

	return true;
}

bool model_bundle_add_byte(bundle **bundle_obj, const char *key, int value)
{
	int ret = bundle_add_byte(*bundle_obj, key, &value, sizeof(value));
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
					"Function bundle_add_byte() failed with error: %d (%s).",
					ret,
					__error_code_to_string(ret));
		return false;
	}

	return true;
}

bool model_bundle_get_byte(bundle *bundle_obj, const char *key, int *value)
{
	*value = 0;

	void *byte_val = NULL;
	size_t size_val = 0;
	int ret = bundle_get_byte(bundle_obj, key, &byte_val, &size_val);
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
					"Function bundle_get_byte() failed with error: %d (%s).",
					ret,
					__error_code_to_string(ret));
		return false;
	}

	if (size_val > 0 && size_val <= sizeof(int))
		*value = *((int *)byte_val);

	return true;
}

bool model_bundle_get_keyval_type(const bundle_keyval_t *kv, int *type)
{
	*type = bundle_keyval_get_type((bundle_keyval_t *)kv);
	int ret = get_last_result() ;
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
				"Function bundle_keyval_get_type() failed with error: %d (%s).",
				ret,
				__error_code_to_string(ret));
		return false;
	}

	return true;
}

bool model_bundle_get_keyval_basic_val(const bundle_keyval_t *kv, void **value)
{
	size_t value_size = 0;
	int ret = bundle_keyval_get_basic_val((bundle_keyval_t *)kv,
												value,
												&value_size);
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR,
			"Function bundle_keyval_get_basic_val() failed with error: %d (%s).",
			ret,
			__error_code_to_string(ret));
		return false;
	}

	return true;
}
