#include "$(appName)-ref.h"
#include "controller/controller.h"
#include "controller/controller-data-source.h"
#include "view/defines.h"
#include "view/generic.h"
#include "view/view.h"

#define CHAR_BUFFER_SIZE 256

typedef struct _bundletype {
	char *caption;
	bundle_value_type_t type;
} bundletype_s;

bundletype_s bundletypes[BUNDLE_VALUE_TYPE_MAX] = {
	{ .caption = "Byte", .type = BUNDLE_VALUE_TYPE_BYTE, },
	{ .caption = "String", .type = BUNDLE_VALUE_TYPE_STRING, },
};

static viewdata_s *viewdata = NULL;
static int __radio_selected = 0;


static Elm_Object_Item *__append_genlist_item(bundledata_s *bundledata);
static Elm_Object_Item *__prepend_genlist_item(bundledata_s *bundledata);


static bool __string_to_int(const char *string, int *value)
{
	if (!value) {
		controller_log(DLOG_INFO, "Wrong input arguments.");
		return false;
	}

	*value = atoi(string);

	if (*value == 0)
		return (strlen(string) == 1 && string[0] == '0');

	return true;
}

static char *__get_key_string(viewdata_s *vd)
{
	const char *string = elm_entry_entry_get(vd->data_source_key_entry);

	if (!string)
		return NULL;

	return strdup(string);
}

static char *__get_value_string(viewdata_s *vd)
{
	const char *string = elm_entry_entry_get(vd->data_source_value_entry);

	if (!string)
		return NULL;

	return strdup(string);
}

static int __get_type_index(viewdata_s *vd)
{
	return __radio_selected;
}

static bool __get_input_data(viewdata_s *vd,
									char **key,
									char **value,
									bundle_value_type_t *value_type)
{
	bool ret = false;

	*key = NULL;
	*value = NULL;
	*value_type = BUNDLE_VALUE_TYPE_MAX;

	if (!vd) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return false;
	}

	*key = __get_key_string(vd);
	*value = __get_value_string(vd);
	*value_type = (bundle_value_type_t)__get_type_index(vd);

	ret = (*key &&
			*value &&
			*value_type >= BUNDLE_VALUE_TYPE_BYTE &&
			*value_type < BUNDLE_VALUE_TYPE_MAX);

	if (!ret) {
		if (*key)
			free(*key);

		if (*value)
			free(*value);
	}

	return ret;
}

static bool __get_input_data_value_pointer(char *str_value,
												bundle_value_type_t value_type,
												void **ptr_value,
												int *val_size)
{
	static int int_value = 0;

	*val_size = 0;
	*ptr_value = NULL;

	switch (value_type) {
	case BUNDLE_VALUE_TYPE_BYTE:
		if (__string_to_int(str_value, &int_value)) {
			*val_size = sizeof(int);
			*ptr_value = &int_value;
		}
		break;
	case BUNDLE_VALUE_TYPE_STRING:
		*val_size = strlen(str_value) + 1;
		*ptr_value = str_value;
		break;
	default:
		controller_log(DLOG_ERROR, "Unknown value type index.");
		break;
	}

	return (val_size > 0);
}

static bool __input_data_to_bundledata(viewdata_s *vd,
											bundledata_s **bundledata)
{
	int val_size = 0;
	void *ptr_value = NULL;
	char *key = NULL;
	char *str_value = NULL;
	bundle_value_type_t value_type = BUNDLE_VALUE_TYPE_MAX;

	*bundledata = NULL;

	if (!__get_input_data(vd, &key, &str_value, &value_type))
		return false;

	if (__get_input_data_value_pointer(str_value,
											value_type,
											&ptr_value,
											&val_size))
		*bundledata = model_create_bundledata(key,
												ptr_value,
												val_size,
												value_type);

	free(key);
	free(str_value);

	return (*bundledata != NULL);
}

static bundledata_s *__create_bundle_header_bundledata(void)
{
	char *str_value = "all-bundles";
	void *ptr_value = NULL;
	int val_size = 0;

	if (!__get_input_data_value_pointer(str_value,
											BUNDLE_VALUE_TYPE_STRING,
											&ptr_value,
											&val_size))
		return NULL;

	return model_create_bundledata("Header",
									ptr_value,
									val_size,
									BUNDLE_VALUE_TYPE_STRING);
}

static void __delete_win_request_cb(void *data,
									Evas_Object *obj,
									void *event_info)
{
	ui_app_exit();
}

static void __layout_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;

	/* Let window go to hide state. */
	elm_win_lower(vd->win);
}

static void __toolbar_item_selected_cb(void *data,
											Evas_Object *obj,
											void *event_info)
{
	Evas_Object *current_page = elm_object_part_content_unset(
											viewdata->layout_main_panel,
											PART_MAIN_CONTENT);
	if (current_page)
		evas_object_hide(current_page);

	if (!data)
		return;

	elm_object_part_content_set(viewdata->layout_main_panel,
									PART_MAIN_CONTENT,
									(Evas_Object *)data);
}

static void __checkbox_changed_cb(void *data,
										Evas_Object *obj,
										void *event_info)
{
	Elm_Object_Item *item = NULL;
	viewdata_s *vd = (viewdata_s *)data;

	if (elm_check_state_get(obj)) {
		bundledata_s *bundledata = __create_bundle_header_bundledata();

		if (bundledata)
			item = __prepend_genlist_item(bundledata);
	} else {
		item = elm_genlist_first_item_get(vd->data_source_list);
		if (!item)
			return;

		elm_object_item_del(item);
		item = elm_genlist_first_item_get(vd->data_source_list);
	}

	if (!item)
		return;

	elm_genlist_item_bring_in(item, ELM_GENLIST_ITEM_SCROLLTO_TOP);
}

static void __button_add_clicked_cb(void *data,
										Evas_Object *obj,
										void *event_info)
{
	viewdata_s *vd = (viewdata_s *)data;
	bundledata_s *bundledata = NULL;

	if (!vd) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return;
	}

	if (!__input_data_to_bundledata(vd, &bundledata))
		return;

	model_add_list_item(bundledata);

	__append_genlist_item(bundledata);
}

static void __button_send_clicked_cb(void *data,
											Evas_Object *obj,
											void *event_info)
{
	if (!data) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return;
	}

	viewdata_s *vd = (viewdata_s *)data;

	bool include_header = (bool)elm_check_state_get(vd->data_source_checkbox);

	controller_data_source_send_message(include_header);
}

static char *__get_genlist_item_label_cb(void *data,
												Evas_Object *obj,
												const char *part)
{
	bundledata_s *bundledata = (bundledata_s *)data;
	char buff[CHAR_BUFFER_SIZE] = {0,};

	if (!bundledata || !bundledata->key || !bundledata->value)
		return NULL;

	if (!strcmp(part, "elm.text")) {
		switch ((bundle_value_type_t)bundledata->type) {
		case BUNDLE_VALUE_TYPE_BYTE:
			snprintf(buff,
						CHAR_BUFFER_SIZE,
						"%s : %d",
						bundledata->key,
						*((int *)bundledata->value));
			break;
		case BUNDLE_VALUE_TYPE_STRING:
			snprintf(buff,
						CHAR_BUFFER_SIZE,
						"%s : %s",
						bundledata->key,
						(char *)bundledata->value);
			break;
		default:
			return NULL;
		}

		return strdup(buff);
	} else if (!strcmp(part, "elm.text.sub")) {
		switch ((bundle_value_type_t)bundledata->type) {
		case BUNDLE_VALUE_TYPE_BYTE:
			return strdup("type: byte");
			break;
		case BUNDLE_VALUE_TYPE_STRING:
			return strdup("type: string");
			break;
		default:
			return NULL;
		}
	}

	return NULL;
}

static void __del_genlist_item_cb(void *data, Evas_Object *obj)
{
	bundledata_s *bundledata = (bundledata_s *)data;

	if (!bundledata)
		return;

	if (bundledata->key)
		free(bundledata->key);

	if (bundledata->value)
		free(bundledata->value);

	free(bundledata);
}

static Elm_Object_Item *__append_genlist_item(bundledata_s *bundledata)
{
	if (!bundledata) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	Elm_Genlist_Item_Class *itc = view_generic_create_genlist_item_class(
										__get_genlist_item_label_cb,
										__del_genlist_item_cb);
	if (!itc)
		return NULL;

	Elm_Object_Item *item = elm_genlist_item_append(viewdata->data_source_list,
													itc,
													(void *)bundledata,
													NULL,
													ELM_GENLIST_ITEM_NONE,
													NULL,
													NULL);
	if (!item) {
		controller_log(DLOG_ERROR, "Function elm_genlist_item_append() failed.");
		return NULL;
	}

	elm_genlist_item_bring_in(item, ELM_GENLIST_ITEM_SCROLLTO_TOP);

	return item;
}

static Elm_Object_Item *__prepend_genlist_item(bundledata_s *bundledata)
{
	if (!bundledata) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return NULL;
	}

	Elm_Genlist_Item_Class *itc = view_generic_create_genlist_item_class(
											__get_genlist_item_label_cb,
											__del_genlist_item_cb);
	if (!itc)
		return NULL;

	Elm_Object_Item *item = NULL;
	Elm_Object_Item *first_item = elm_genlist_first_item_get(
											viewdata->data_source_list);
	if (first_item) {
		item = elm_genlist_item_insert_before(viewdata->data_source_list,
												itc,
												(void *)bundledata,
												NULL,
												first_item,
												ELM_GENLIST_ITEM_NONE,
												NULL,
												NULL);
		if (!item) {
			controller_log(DLOG_ERROR,
							"Function elm_genlist_item_insert_before() failed.");
			return NULL;
		}

		return item;
	}

	item = elm_genlist_item_append(viewdata->data_source_list,
									itc,
									(void *)bundledata,
									NULL,
									ELM_GENLIST_ITEM_NONE,
									NULL,
									NULL);

	if (!item) {
		controller_log(DLOG_ERROR, "Function elm_genlist_item_append() failed.");
		return NULL;
	}

	return item;
}

static bool __create_main_panel(viewdata_s *vd)
{
	vd->win = view_generic_create_window(__delete_win_request_cb);
	if (!vd->win)
		return false;

	vd->conform = view_generic_create_conformant(vd->win);
	if (!vd->conform)
		return false;

	vd->layout_main_panel = view_generic_create_set_layout(vd->conform,
															EDJ_MAIN_FILE_NAME,
															GROUP_MAIN,
															NULL);
	if (!vd->layout_main_panel)
		return false;

	eext_object_event_callback_add(vd->layout_main_panel,
									EEXT_CALLBACK_BACK,
									__layout_back_cb,
									(void *)vd);

	vd->main_toolbar = view_generic_create_toolbar(vd->layout_main_panel,
													PART_MAIN_TOOLBAR);
	if (!vd->main_toolbar)
		return false;

	vd->main_toolbar_item_data_source = elm_toolbar_item_append(vd->main_toolbar,
												NULL,
												"Data source",
												__toolbar_item_selected_cb,
												NULL);
	if (!vd->main_toolbar_item_data_source)
		return false;

	vd->main_toolbar_item_data_sink = elm_toolbar_item_append(vd->main_toolbar,
												NULL,
												"Data sink",
												__toolbar_item_selected_cb,
												NULL);
	if (!vd->main_toolbar_item_data_sink)
		return false;

	return true;
}

static bool __create_data_source_edit(viewdata_s *vd)
{
	if (!vd) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return false;
	}

	vd->layout_data_source_edit = view_generic_create_set_layout(
										vd->layout_data_source,
										EDJ_DATA_SOURCE_EDIT_PANEL_FILE_NAME,
										GROUP_DATA_SOURCE_EDIT_PANEL,
										PART_DATA_SOURCE_EDIT_PANEL);
	if (!vd->layout_data_source_edit)
		return false;

	vd->data_source_key_entry = view_generic_create_entry(
										vd->layout_data_source_edit,
										PART_DATA_SOURCE_EDIT_PANEL_KEY_PANEL_ENTRY);
	if (!vd->data_source_key_entry)
		return false;

	vd->data_source_value_entry = view_generic_create_entry(
								vd->layout_data_source_edit,
								PART_DATA_SOURCE_EDIT_PANEL_VALUE_PANEL_ENTRY);
	if (!vd->data_source_value_entry)
		return false;

	vd->data_source_type_box = view_generic_create_box(
								vd->layout_data_source_edit,
								PART_DATA_SOURCE_EDIT_PANEL_TYPE_PANEL_ENTRY);
	if (!vd->data_source_type_box)
		return false;

	int i;

	for (i = 0; i < BUNDLE_VALUE_TYPE_MAX; i++) {
		char caption[256] = {0,};
		snprintf(caption, 256, "<font color=#ffffff>%s</font>", bundletypes[i].caption);
		view_generic_create_radio(vd->data_source_type_box, caption, i, &__radio_selected);
	}

	return true;
}

static bool __create_view_data_source_list(viewdata_s *vd)
{
	if (!vd) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return false;
	}

	vd->layout_data_source_list = view_generic_create_set_layout(
										vd->layout_data_source,
										EDJ_DATA_SOURCE_LIST_PANEL_FILE_NAME,
										GROUP_DATA_SOURCE_LIST_PANEL,
										PART_DATA_SOURCE_LIST_PANEL);
	if (!vd->layout_data_source_list)
		return false;

	vd->data_source_checkbox = view_generic_create_checkbox(
										vd->layout_data_source_list,
										PART_DATA_SOURCE_LIST_PANEL_CHECKBOX,
										"");
	if (!vd->data_source_checkbox)
		return false;

	evas_object_smart_callback_add(vd->data_source_checkbox,
									"changed",
									__checkbox_changed_cb,
									(void *)vd);

	vd->data_source_list = view_generic_create_genlist(
									vd->layout_data_source_list,
									PART_DATA_SOURCE_LIST_PANEL_LIST);
	if (!vd->data_source_list)
		return false;

	return true;
}

static bool __create_data_source_buttons(viewdata_s *vd)
{
	if (!vd) {
		controller_log(DLOG_ERROR, "Wrong input arguments.");
		return false;
	}

	vd->layout_data_source_buttons = view_generic_create_set_layout(
										vd->layout_data_source,
										EDJ_DATA_SOURCE_BUTTONS_PANEL_FILE_NAME,
										GROUP_DATA_SOURCE_BUTTONS_PANEL,
										PART_DATA_SOURCE_BUTTONS_PANEL);
	if (!vd->layout_data_source_buttons)
		return false;

	vd->data_source_button_add = view_generic_create_button(
										vd->layout_data_source_buttons,
										PART_DATA_SOURCE_BUTTONS_PANEL_ADD,
										"Add",
										__button_add_clicked_cb,
										(void *)vd);
	if (!vd->data_source_button_add)
		return false;

	vd->data_source_button_send = view_generic_create_button(
										vd->layout_data_source_buttons,
										PART_DATA_SOURCE_BUTTONS_PANEL_SEND,
										"Send",
										__button_send_clicked_cb,
										(void *)vd);
	if (!vd->data_source_button_send)
		return false;

	return true;
}

static bool __create_view_data_source_content(viewdata_s *vd)
{
	vd->layout_data_source = view_generic_create_layout(
													vd->layout_main_panel,
													EDJ_DATA_SOURCE_FILE_NAME,
													GROUP_DATA_SOURCE);
	if (!vd->layout_data_source)
		return false;

	if (!__create_data_source_edit(vd))
		return false;

	if (!__create_view_data_source_list(vd))
		return false;

	if (!__create_data_source_buttons(vd))
		return false;

	return true;
}

static bool __create_data_sink_content(viewdata_s *vd)
{
	vd->layout_data_sink = view_generic_create_layout(vd->layout_main_panel,
														EDJ_DATA_SINK_FILE_NAME,
														GROUP_DATA_SINK);
	if (!vd->layout_data_sink)
		return false;

	vd->sink_entry = view_generic_create_entry(vd->layout_data_sink,
												PART_DATA_SINK_ENTRY);
	if (!vd->sink_entry)
		return false;

	elm_entry_single_line_set(vd->sink_entry, EINA_FALSE);
	elm_entry_editable_set(vd->sink_entry, EINA_FALSE);
	elm_entry_scrollable_set(vd->sink_entry, EINA_TRUE);

	return true;
}

bool view_create_base_gui(viewdata_s *vd)
{
	viewdata = vd;

	if (!__create_main_panel(vd)) {
		evas_object_del(vd->win);
		return false;
	}

	if (!__create_view_data_source_content(vd)) {
		evas_object_del(vd->win);
		return false;
	}

	if (!__create_data_sink_content(vd)) {
		evas_object_del(vd->win);
		return false;
	}

	/* The layout is assigned to the item's data field. This layout is used in
	 * view_toolbar_item_selected_cb() callback function to switch the layout
	 * on toolbar's item selection.
	 */
	elm_object_item_data_set(vd->main_toolbar_item_data_source,
								(void *)vd->layout_data_source);
	elm_object_item_data_set(vd->main_toolbar_item_data_sink,
								(void *)vd->layout_data_sink);

	/* The "Data source" tab is marked as selected
	 * (the view_toolbar_item_selected_cb() callback function is not
	 * automatically called).
	 */
	elm_toolbar_item_selected_set(vd->main_toolbar_item_data_source, EINA_TRUE);

	/* Set the vd->layout_data_source as the content of
	 * viewdata->layout_main_panel layout because the
	 * view_toolbar_item_selected_cb() callback function is not called
	 * when elm_toolbar_item_selected_set() is invoked.
	 */
	elm_object_part_content_set(viewdata->layout_main_panel,
								PART_MAIN_CONTENT,
								vd->layout_data_source);

	evas_object_show(vd->win);

	return true;
}

void view_destroy_base_gui(viewdata_s *vd)
{
	if (!vd)
		return;

	if (vd->layout_data_source)
		evas_object_del(vd->layout_data_source);

	if (vd->layout_data_sink)
		evas_object_del(vd->layout_data_sink);

	if (vd->win)
		evas_object_del(vd->win);
}

void view_add_sink_message(const char *message)
{
	if (!message)
		return;

	elm_entry_entry_append(viewdata->sink_entry, message);
}
