/*
 * Copyright 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "main-app.h"

#include "model/calc.h"

#include "view/main-view.h"

#include "utils/ui-utils.h"
#include "utils/logger.h"

#include <stdlib.h>
#include <elm_app.h>

#define BASE_DISPLAY_SCALE 1.8

/* app event callbacks */
static bool _on_create_cb(void *user_data);
static void _on_terminate_cb(void *user_data);
static void _on_region_fmt_changed_cb(app_event_info_h event_info, void *user_data);

static Evas_Object *_add_naviframe(app_data *app);
static void _navi_back_cb(void *data, Evas_Object *obj, void *event_info);

app_data *app_create()
{
	elm_app_base_scale_set(BASE_DISPLAY_SCALE);

	return calloc(1, sizeof(app_data));
}

void app_destroy(app_data *app)
{
	free(app);
}

int app_run(app_data *app, int argc, char **argv)
{
	RETVM_IF(!app, -1, "app is NULL");

	ui_app_lifecycle_callback_s cbs = {
		.create = _on_create_cb,
		.terminate = _on_terminate_cb,
	};
	return ui_app_main(argc, argv, &cbs, app);
}

static Evas_Object *_add_naviframe(app_data *app)
{
	Evas_Object *result = NULL;
	Evas_Object *parent = win_get_host_layout(app->win);
	if (parent) {
		result = ui_utils_navi_add(parent, _navi_back_cb, app);
		if (result) {
			win_set_layout(app->win, result);
		}
	}
	return result;
}

static bool _on_create_cb(void *user_data)
{
	app_data *app = user_data;
	RETVM_IF(!app, false, "ad is NULL");

	int res = ui_app_add_event_handler(&(app->event_handler), APP_EVENT_REGION_FORMAT_CHANGED,
			_on_region_fmt_changed_cb, app);
	RETVM_IF(res != APP_ERROR_NONE, false, "Add event handler failed: %d", res);

	result_type result = RES_OK;
	app->model = calc_create(&result);
	RETVM_IF(!app->model, false, "calc instance creation failed: %d", result);

	app->win = win_create();
	RETVM_IF(!app->win, false, "Window creation failed.");

	app->navi = _add_naviframe(app);
	RETVM_IF(!app->navi, false, "Naviframe creation failed.");

	main_view_add(app);
	return true;
}

static void _navi_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	app_data *app = data;
	RETM_IF(!app, "app is NULL");

	if (elm_naviframe_top_item_get(obj) == elm_naviframe_bottom_item_get(obj)) {
		win_lower(app->win);
	} else {
		elm_naviframe_item_pop(obj);
	}
}

static void _on_terminate_cb(void *user_data)
{
	app_data *app = user_data;
	RETM_IF(!app, "app is NULL");

	ui_app_remove_event_handler(app->event_handler);

	win_destroy(app->win);
	app->win = NULL;
	app->navi = NULL;

	calc_destroy(app->model);
	app->model = NULL;
}

static void _on_region_fmt_changed_cb(app_event_info_h event_info, void *user_data)
{
	app_data *app = user_data;
	RETM_IF(!app, "app is NULL");

	if (app->region_fmt_change_cb) {
		app->region_fmt_change_cb(app->region_fmt_change_cb_data);
	}
}
