/*
 * Copyright 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "display.h"

#include "utils/utils.h"
#include "utils/logger.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

static const double CALC_MAX_MULTIPLIER   = 1.0e+11; /* 10 ^ (CALC_MAX_DIGITS - 1) */
static const double CALC_NUMBER_ABS_LIMIT = 1.0e+12; /* 10 ^ CALC_MAX_DIGITS */

static void _display_reset_props(display *obj);
static void _display_set_str(display *obj, const char *new_str);
static bool _display_is_empty(display *obj);
static void _display_notify_change(display *obj);

void display_reset(display *obj)
{
	RETM_IF(!obj, "obj is NULL");

	_display_reset_props(obj);
	_display_set_str(obj, "0");
}

void _display_reset_props(display *obj)
{
	obj->len = 1;
	obj->dig_count = 1;
	obj->is_neg = false;
	obj->has_dec_point = false;
}

void _display_set_str(display *obj, const char *new_str)
{
	if (strcmp(new_str, obj->str) != 0) {
		strcpy(obj->str, new_str);
		_display_notify_change(obj);
	}
}

void display_set_dec_point_char(display *obj, char new_dec_point_char)
{
	RETM_IF(!obj, "obj is NULL");

	if (new_dec_point_char != obj->dec_point_char) {
		if (obj->has_dec_point) {
			char *pos = strchr(obj->str, obj->dec_point_char);
			if (pos) {
				*pos = new_dec_point_char;
				_display_notify_change(obj);
			}
		}
		obj->dec_point_char = new_dec_point_char;
	}
}

result_type display_set_number(display *obj, double value)
{
	RETVM_IF(!obj, RES_INTERNAL_ERROR, "obj is NULL");
	RETVM_IF(fabs(value) >= CALC_NUMBER_ABS_LIMIT, RES_NUMER_IS_TOO_LONG, "Number is too long: %f", value)

	char new_str[DISP_STR_SIZE] = {'\0'};

	bool is_neg = (value < 0.0);

	long long whole = (long long)(value);
	int whole_len = (0 != whole) ? (snprintf(new_str, DISP_STR_SIZE, "%lld", whole) - is_neg) : 1;
	int fract_len = DISP_MAX_DIGITS - whole_len;

	if (0 == whole) {
		long long round_mant = 0;
		if (is_neg) {
			round_mant = utils_round((value - 1.0) * CALC_MAX_MULTIPLIER);
		} else {
			round_mant = utils_round((value + 1.0) * CALC_MAX_MULTIPLIER);
		}
		snprintf(new_str + 1, DISP_STR_SIZE - 1, "%lld", round_mant);
		 --new_str[is_neg + 1];
	} else if (fract_len >= 0) {
		long long mult = 1;
		int i = 0;
		for (i = 0; i < fract_len; ++i) {
			mult *= 10;
		}
		long long round_mant = utils_round(value * (double)(mult));
		int c = snprintf(new_str + 1, DISP_STR_SIZE - 1, "%lld", round_mant);
		if (c > DISP_MAX_DIGITS + is_neg) {
			++whole_len;
			--fract_len;
			new_str[c] = '\0';
		}
	}

	RETVM_IF(fract_len < 0, RES_NUMER_IS_TOO_LONG, "Number is too long: %f", value)

	obj->len = DISP_MAX_DIGITS + is_neg + 1;
	obj->dig_count = DISP_MAX_DIGITS;
	obj->is_neg = is_neg;
	obj->has_dec_point = true;

	memmove(new_str, new_str + 1, whole_len + is_neg);
	new_str[whole_len + is_neg] = obj->dec_point_char;

	while ('0' == new_str[obj->len - 1]) {
		--obj->len;
		--obj->dig_count;
		new_str[obj->len] = '\0';
	}
	if (obj->dec_point_char == new_str[obj->len - 1]) {
		--obj->len;
		new_str[obj->len] = '\0';
		obj->has_dec_point = false;
	}
	if (is_neg && (2 == obj->len) && ('0' == new_str[1])) {
		new_str[0] = '0';
		new_str[1] = '\0';
		obj->len = 1;
		obj->is_neg = false;
	}

	_display_set_str(obj, new_str);

	return RES_OK;
}

double display_get_number(display *obj)
{
	RETVM_IF(!obj, 0.0, "obj is NULL");

	return strtod(obj->str, NULL);
}

result_type display_enter_key(display *obj, key_type key, bool reset)
{
	RETVM_IF(!obj, RES_INTERNAL_ERROR, "obj is NULL");

	RETVM_IF((CALC_KEY_DEC != key) && ((key < CALC_KEY_0) || (key > CALC_KEY_9)),
			RES_ILLEGAL_ARGUMENT, "Invalid key: %d", key);

	char new_str[DISP_STR_SIZE] = {'\0'};

	if (reset) {
		_display_reset_props(obj);
		strcpy(new_str, "0");
	} else {
		RETVM_IF(DISP_MAX_DIGITS == obj->dig_count, RES_MAX_DIGITS_REACHED, "Display is full");

		strcpy(new_str, obj->str);
	}

	if (CALC_KEY_DEC != key) {
		char dig_char = '0' + (key - CALC_KEY_0);
		if ((1 == obj->len) && ('0' == new_str[0])) {
			new_str[0] = dig_char;
		} else {
			new_str[obj->len] = dig_char;
			++obj->len;
			new_str[obj->len] = '\0';
			++obj->dig_count;
		}
	} else {
		RETVM_IF(obj->has_dec_point, RES_DOUBLE_DOT, "Double decimal point entered");

		new_str[obj->len] = obj->dec_point_char;
		++obj->len;
		new_str[obj->len] = '\0';
		obj->has_dec_point = true;
	}

	_display_set_str(obj, new_str);

	return RES_OK;
}

result_type display_erase(display *obj)
{
	RETVM_IF(!obj, RES_INTERNAL_ERROR, "obj is NULL");
	RETVM_IF(_display_is_empty(obj), RES_DISPLAY_IS_EMPTY, "Dipsplay is empty");

	if ((1 == obj->dig_count) && ((!obj->has_dec_point) || ('0' == obj->str[obj->is_neg]))) {
		display_reset(obj);
		return RES_OK;
	}

	--obj->len;

	if (obj->has_dec_point && (obj->str[obj->len] == obj->dec_point_char)) {
		obj->has_dec_point = false;
	} else {
		--obj->dig_count;
	}

	obj->str[obj->len] = '\0';
	_display_notify_change(obj);

	return RES_OK;
}

void display_negate(display *obj)
{
	RETM_IF(!obj, "obj is NULL");

	if (!_display_is_empty(obj)) {
		if (obj->is_neg) {
			memmove(obj->str, obj->str + 1, obj->len);
			obj->len--;
			obj->is_neg = false;
		} else {
			++obj->len;
			memmove(obj->str + 1, obj->str, obj->len);
			obj->str[0] = '-';
			obj->is_neg = true;
		}
		_display_notify_change(obj);
	}
}

bool _display_is_empty(display *obj)
{
	return ((1 == obj->len) && ('0' == obj->str[0]));
}

void display_set_str(display *obj, const char *new_str)
{
	RETM_IF(!obj, "obj is NULL");
	RETM_IF(!new_str, "new_str is NULL");

	_display_set_str(obj, new_str);
}

void _display_notify_change(display *obj)
{
	if (obj->change_cb) {
		obj->change_cb(obj->change_cb_data);
	}
}
