/*
 * Copyright 2014-2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include "view/main-view.h"

#include "main-app.h"

#include "model/calc.h"

#include "utils/ui-utils.h"
#include "utils/utils.h"
#include "utils/resources.h"
#include "utils/logger.h"

#define EO_ALIGN_HINT_CENTER 0.5

/* General constant values */
enum {
	WIDGETS_COUNT = 20, /* Total number of all kinds of widgets on the view */
	BUTTONS_COUNT = WIDGETS_COUNT - 1, /* Count of the buttons */

	CIRCLE_QUARTER_DEG = 90,
	CIRCLE_HALF_QUARTER_DEG = CIRCLE_QUARTER_DEG / 2
};

struct _main_view_data;

/* Data for the button */
typedef struct {
	struct _main_view_data *view; /* Pointer to the parent view structure */
	key_type key; /* Key value assigned with the button */

} main_view_btn_data;

/* Internal data structure of the main_view object */
typedef struct _main_view_data {
	app_data *app;
	calc *model;
	Evas_Object *window; /* elm_win object */
	int orientation; /* current orientation of the layout */

	Evas_Object *layout; /* root layout of the view (single cell elm_table) */
	Evas_Object *grid; /* content of the layout, holds and aligns the widgets */
	Evas_Object *widgets[WIDGETS_COUNT]; /* widgets in the grid (used for fast orientation change) */

	Evas_Object *label; /* display elm_label object (update display number and message) */
	Evas_Object *dec_btn; /* decimal elm_button object (update decimal point character) */

	/* Data structures for the buttons (avoid dynamic data allocations) */
	main_view_btn_data btns_data[BUTTONS_COUNT];

} main_view_data;

/* Parameters for widget creation */
typedef struct {
	key_type key; /* key type of the button or CALC_KEY_NONE for label */
	const char *text; /* text label for the button or NULL for label */

} widget_params;

/* Widget position and size on the view grid */
typedef struct {
	int x, y;
	int w, h;

} widget_pos;

/* Initialises allocated main_view instance */
static result_type _main_view_init(main_view_data *data, app_data *app);
/* Makes the layout and grid objects on specified parent object */
static result_type _main_view_make_layout(main_view_data *data, Evas_Object *parent);
/* Makes widgets on the layout. Widget is not packed to a grid yet */
static result_type _main_view_make_widgets(main_view_data *data);
/* Changes view orientations and repacks the widgets */
static void _main_view_set_orientation(main_view_data *data, int orientation);
/* Repacks previously created widgets to the grid according to the current orientation */
static void _main_view_repack_widgets(main_view_data *data);

/* Destroys and deallocates main_view object */
static void _main_view_del(main_view_data *data);

/* Updates region format dependent properties */
static void _main_view_update_region_fmt(main_view_data *data);
/* Updates display label object from the model */
static void _main_view_update_disp_label(main_view_data *data);
/* Handles key press by the model. Shows error popups. */
static void _main_view_handle_key_press(main_view_data *data, key_type key);
/* Creates and shows message popup with OK button */
static void _main_view_create_msg_popup(main_view_data *data, const char *msg);

/* Handler for the model display change event */
static void _main_view_display_changed_cb(void *data);
/* Handler for the app region format change event */
static void _main_view_region_fmt_changed_cb(void *data);
/* Handler for the window rotation change event */
static void _main_view_win_rotate_cb(void *data, Evas_Object *obj, void *event_info);
/* Handler for the view layout free event */
static void _main_view_layout_free_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);

/* Handler for the view button clock event */
static void _main_view_btn_click_cb(void *data, Evas_Object *obj, void *event_info);

/* Handler for the popup close event (OK click or back button press) */
static void _main_view_popup_close_cb(void *data, Evas_Object *obj, void *event_info);

/* Widget creation parameters. Order must be the same as in other arrays. */
static const widget_params WIDGET_PARAMS[] = {
	{CALC_KEY_NONE,   NULL},
	{CALC_KEY_RESULT, "="},
	{CALC_KEY_SIGN,   "+/-"},
	{CALC_KEY_ERASE,  "<font_size=48>\xE2\x86\x90</font_size>"},
	{CALC_KEY_RESET,  "<b>C</b>"},

	{CALC_KEY_0,      "0"},
	{CALC_KEY_1,      "1"},
	{CALC_KEY_2,      "2"},
	{CALC_KEY_3,      "3"},
	{CALC_KEY_4,      "4"},
	{CALC_KEY_5,      "5"},
	{CALC_KEY_6,      "6"},
	{CALC_KEY_7,      "7"},
	{CALC_KEY_8,      "8"},
	{CALC_KEY_9,      "9"},
	{CALC_KEY_DEC,    NULL},

	{CALC_KEY_ADD,    "+"},
	{CALC_KEY_SUB,    "-"},
	{CALC_KEY_MUL,    "*"},
	{CALC_KEY_DIV,    "/"}
};

/* View layout constants */
enum {
	/* Common parameters */
	VIEW_BORDER_WIDTH = 10, /* Border around all the grid with widgets */
	WIDGET_PADDING = 3, /* Padding around all widgets (adds extra pixels to the border) */
	DISP_NUMBER_RIGHT_PADDING = 20, /* Padding between the right edge of the display text and grid cell virtual edje */
	DISP_NUMBER_TEXT_SIZE = 56,

	/* Portrait parameters */

	PORT_DISP_HEIGHT = 4, /* Height of the button in the logical grid units */
	PORT_DISP_GAP = 6, /* Space between display */
	PORT_BTN1_HEIGHT = 4, /* Height of the first row of the buttons */
	PORT_AUX_GAP = 1, /* Space between first and second row of the buttons */
	PORT_BTN2_HEIGHT = 4, /* Height of the other rows of the buttons */

	/* Y offsets of the widget rows for portrait orientation */
	PORT_OFFS0 = 0,
	PORT_OFFS1 = PORT_OFFS0 + PORT_DISP_HEIGHT + PORT_DISP_GAP,
	PORT_OFFS2 = PORT_OFFS1 + PORT_BTN1_HEIGHT + PORT_AUX_GAP,
	PORT_OFFS3 = PORT_OFFS2 + PORT_BTN2_HEIGHT,
	PORT_OFFS4 = PORT_OFFS3 + PORT_BTN2_HEIGHT,
	PORT_OFFS5 = PORT_OFFS4 + PORT_BTN2_HEIGHT,
	PORT_OFFS6 = PORT_OFFS5 + PORT_BTN2_HEIGHT,

	/* Size of the grid in logical units for portrait orientation */
	PORT_GRID_W = 4,
	PORT_GRID_H = PORT_OFFS6 + PORT_BTN2_HEIGHT,

	/* Landscape parameters */

	LAND_DISP_HEIGHT = 4,
	LAND_DISP_GAP = 1,
	LAND_BTN_HEIGHT = 4,

	LAND_OFFS0 = 0,
	LAND_OFFS1 = LAND_OFFS0 + LAND_DISP_HEIGHT + LAND_DISP_GAP,
	LAND_OFFS2 = LAND_OFFS1 + LAND_BTN_HEIGHT,
	LAND_OFFS3 = LAND_OFFS2 + LAND_BTN_HEIGHT,

	LAND_GRID_W = 7,
	LAND_GRID_H = LAND_OFFS3 + LAND_BTN_HEIGHT
};

/* Widgets sizes and positions for portrait orientation. */
/* Order must be the same as in other arrays. */
static const widget_pos PORT_WPOS[] = {
	{0, PORT_OFFS0, 4, PORT_DISP_HEIGHT},
	{3, PORT_OFFS5, 1, PORT_BTN2_HEIGHT * 2},
	{0, PORT_OFFS2, 1, PORT_BTN2_HEIGHT},
	{2, PORT_OFFS1, 2, PORT_BTN1_HEIGHT},
	{0, PORT_OFFS1, 1, PORT_BTN1_HEIGHT},

	{0, PORT_OFFS6, 2, PORT_BTN2_HEIGHT},
	{0, PORT_OFFS5, 1, PORT_BTN2_HEIGHT},
	{1, PORT_OFFS5, 1, PORT_BTN2_HEIGHT},
	{2, PORT_OFFS5, 1, PORT_BTN2_HEIGHT},
	{0, PORT_OFFS4, 1, PORT_BTN2_HEIGHT},
	{1, PORT_OFFS4, 1, PORT_BTN2_HEIGHT},
	{2, PORT_OFFS4, 1, PORT_BTN2_HEIGHT},
	{0, PORT_OFFS3, 1, PORT_BTN2_HEIGHT},
	{1, PORT_OFFS3, 1, PORT_BTN2_HEIGHT},
	{2, PORT_OFFS3, 1, PORT_BTN2_HEIGHT},
	{2, PORT_OFFS6, 1, PORT_BTN2_HEIGHT},

	{3, PORT_OFFS3, 1, PORT_BTN2_HEIGHT * 2},
	{3, PORT_OFFS2, 1, PORT_BTN2_HEIGHT},
	{2, PORT_OFFS2, 1, PORT_BTN2_HEIGHT},
	{1, PORT_OFFS2, 1, PORT_BTN2_HEIGHT}
};

/* Widgets sizes and positions for landscape orientation. */
/* Order must be the same as in other arrays. */
static const widget_pos LAND_WPOS[] = {
	{1, LAND_OFFS0, 4, LAND_DISP_HEIGHT},
	{5, LAND_OFFS3, 2, LAND_BTN_HEIGHT},
	{0, LAND_OFFS1, 1, LAND_BTN_HEIGHT},
	{5, LAND_OFFS0, 2, LAND_DISP_HEIGHT},
	{0, LAND_OFFS0, 1, LAND_DISP_HEIGHT},

	{0, LAND_OFFS2, 1, LAND_BTN_HEIGHT * 2},
	{1, LAND_OFFS3, 1, LAND_BTN_HEIGHT},
	{2, LAND_OFFS3, 1, LAND_BTN_HEIGHT},
	{3, LAND_OFFS3, 1, LAND_BTN_HEIGHT},
	{1, LAND_OFFS2, 1, LAND_BTN_HEIGHT},
	{2, LAND_OFFS2, 1, LAND_BTN_HEIGHT},
	{3, LAND_OFFS2, 1, LAND_BTN_HEIGHT},
	{1, LAND_OFFS1, 1, LAND_BTN_HEIGHT},
	{2, LAND_OFFS1, 1, LAND_BTN_HEIGHT},
	{3, LAND_OFFS1, 1, LAND_BTN_HEIGHT},
	{4, LAND_OFFS3, 1, LAND_BTN_HEIGHT},

	{4, LAND_OFFS1, 1, LAND_BTN_HEIGHT * 2},
	{5, LAND_OFFS1, 1, LAND_BTN_HEIGHT * 2},
	{6, LAND_OFFS2, 1, LAND_BTN_HEIGHT},
	{6, LAND_OFFS1, 1, LAND_BTN_HEIGHT}
};

/* Static assertions to ensure that all arrays has WIDGETS_COUNT size */
typedef char WIDGET_PARAMS_COUNT_CHECK[((sizeof(WIDGET_PARAMS) / sizeof(WIDGET_PARAMS[0])) == WIDGETS_COUNT) ? 1 : -1];
typedef char PORT_WPOS_COUNT_CHECK[((sizeof(PORT_WPOS) / sizeof(PORT_WPOS[0])) == WIDGETS_COUNT) ? 1 : -1];
typedef char LAND_WPOS_COUNT_CHECK[((sizeof(LAND_WPOS) / sizeof(LAND_WPOS[0])) == WIDGETS_COUNT) ? 1 : -1];

void main_view_add(app_data *app)
{
	RETM_IF(!app, "app is NULL");

	main_view_data *data = calloc(1, sizeof(main_view_data));
	RETM_IF(!data, "Failed to allocate the instance");

	result_type result = _main_view_init(data, app);
	if (RES_OK != result) {
		_main_view_del(data);
	}
}

void _main_view_layout_free_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	main_view_data *view_data = data;
	view_data->layout = NULL; /* NULL the layout so we do not delete it twice in _main_view_del() */
	_main_view_del(view_data); /* Delete self when layout is deleted by parent object */
}

result_type _main_view_init(main_view_data *data, app_data *app)
{
	RETVM_IF(!app->model, RES_ILLEGAL_ARGUMENT, "app->model is NULL");
	RETVM_IF(!app->win, RES_ILLEGAL_ARGUMENT, "app->win is NULL");
	RETVM_IF(!app->win->win, RES_ILLEGAL_ARGUMENT, "app->win->win is NULL");
	RETVM_IF(!app->navi, RES_ILLEGAL_ARGUMENT, "app->navi is NULL");

	data->app = app;
	data->model = app->model;
	data->window = app->win->win;
	data->orientation = elm_win_rotation_get(data->window);

	result_type result = RES_INTERNAL_ERROR;

	result = _main_view_make_layout(data, app->navi);
	RETVM_IF(RES_OK != result, result, "Failed to make layout: %d", result);

	result = _main_view_make_widgets(data);
	RETVM_IF(RES_OK != result, result, "Failed to make widgets: %d", result);

	_main_view_repack_widgets(data);

	Elm_Object_Item *navi_item = elm_naviframe_item_push(app->navi,
			STR_MAIN_TITLE, NULL, NULL, data->layout, NULL);
	RETVM_IF(!navi_item, RES_OUT_OF_MEMORY, "Failed to push naviframe item");

	evas_object_smart_callback_add(data->window, "wm,rotation,changed", _main_view_win_rotate_cb, data);
	calc_set_display_change_cb(data->model, _main_view_display_changed_cb, data);
	app->region_fmt_change_cb = _main_view_region_fmt_changed_cb;
	app->region_fmt_change_cb_data = data;

	return RES_OK;
}

result_type _main_view_make_layout(main_view_data *data, Evas_Object *parent)
{
	int scaled_border_padding = (int)utils_round(ELM_SCALE_SIZE(VIEW_BORDER_WIDTH));

	/* Create single cell table so the internal grid will support padding */
	Evas_Object *layout = elm_table_add(parent);
	RETVM_IF(!layout, RES_OUT_OF_MEMORY, "Failed to create elm_table");

	data->layout = layout;
	elm_table_homogeneous_set(layout, EINA_TRUE);
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_event_callback_add(layout, EVAS_CALLBACK_FREE, _main_view_layout_free_cb, data);
	evas_object_show(layout);

	/* Add background to the view with default theme color */
	Evas_Object *bg = elm_bg_add(layout);
	RETVM_IF(!bg, RES_OUT_OF_MEMORY, "Failed to create elm_bg");

	elm_table_pack(layout, bg, 0, 0, 1, 1);
	evas_object_size_hint_align_set(bg, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(bg);

	/* Create grid for holding and aligning the widgets */
	Evas_Object *grid = elm_grid_add(layout);
	RETVM_IF(!grid, RES_OUT_OF_MEMORY, "Failed to create elm_grid");

	data->grid = grid;
	elm_table_pack(layout, grid, 0, 0, 1, 1);
	evas_object_size_hint_align_set(grid, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_padding_set(grid,
			scaled_border_padding, scaled_border_padding,
			scaled_border_padding, scaled_border_padding);
	evas_object_show(grid);

	return RES_OK;
}

result_type _main_view_make_widgets(main_view_data *data)
{
	int scaled_widget_padding = (int)utils_round(ELM_SCALE_SIZE(WIDGET_PADDING));
	int scaled_number_padding = (int)utils_round(ELM_SCALE_SIZE(DISP_NUMBER_RIGHT_PADDING));

	int btn_idx = 0;
	int i = 0;
	for (i = 0; i < WIDGETS_COUNT; ++i) {
		const widget_params *params = &WIDGET_PARAMS[i];

		/* For each widget in the grid's cell we create additional single cell table */
		/* This will allow to add paddings for widgets because grid does not support paddings */
		Evas_Object *table = elm_table_add(data->grid);
		RETVM_IF(!table, RES_OUT_OF_MEMORY, "Failed to create elm_table");

		data->widgets[i] = table; /* Registering widget in the array so we can late pack it to the grid */
		elm_table_homogeneous_set(table, EINA_TRUE);
		evas_object_show(table);

		bool is_button = (CALC_KEY_NONE != params->key);
		Evas_Object *obj = is_button ? elm_button_add(table) : elm_label_add(table);
		RETVM_IF(!obj, RES_OUT_OF_MEMORY, "Failed to create widget object");

		elm_table_pack(table, obj, 0, 0, 1, 1);

		if (is_button) {
			RETVM_IF(BUTTONS_COUNT == btn_idx, RES_INTERNAL_ERROR, "Too many buttons in params!");

			main_view_btn_data *btn_data = &data->btns_data[btn_idx];
			++btn_idx;
			btn_data->view = data;
			btn_data->key = params->key;
			evas_object_smart_callback_add(obj, "clicked", _main_view_btn_click_cb, btn_data);

			if (CALC_KEY_DEC == params->key) {
				elm_object_text_set(obj, calc_get_dec_point_str(data->model));
			} else {
				elm_object_text_set(obj, params->text);
			}

			evas_object_size_hint_align_set(obj, EVAS_HINT_FILL, EVAS_HINT_FILL);
			evas_object_size_hint_padding_set(obj,
					scaled_widget_padding, scaled_widget_padding,
					scaled_widget_padding, scaled_widget_padding);
		} else {
			data->label = obj;
			_main_view_update_disp_label(data);
			evas_object_size_hint_align_set(obj, 1, EO_ALIGN_HINT_CENTER);
			evas_object_size_hint_padding_set(obj,
					scaled_widget_padding, scaled_widget_padding + scaled_number_padding,
					scaled_widget_padding, scaled_widget_padding);
		}

		evas_object_show(obj);
	}

	return RES_OK;
}

void _main_view_set_orientation(main_view_data *data, int orientation)
{
	if (orientation != data->orientation) {
		data->orientation = orientation;
		_main_view_repack_widgets(data);
	}
}

void _main_view_repack_widgets(main_view_data *data)
{
	/* Remove all widget from the grid without deleting */
	elm_grid_clear(data->grid, EINA_FALSE);

	/* Obtaining widget position params and resizing the grid according to the orientation */
	const widget_pos *pos_array = NULL;
	if ((((abs(data->orientation) + CIRCLE_HALF_QUARTER_DEG) / CIRCLE_QUARTER_DEG) & 1) == 0) {
		pos_array = PORT_WPOS;
		elm_grid_size_set(data->grid, PORT_GRID_W, PORT_GRID_H);
	} else {
		pos_array = LAND_WPOS;
		elm_grid_size_set(data->grid, LAND_GRID_W, LAND_GRID_H);
	}

	/* Packing widgets to the grid using information from the array */
	int i = 0;
	for (i = 0; i < WIDGETS_COUNT; ++i) {
		elm_grid_pack(data->grid, data->widgets[i],
				pos_array[i].x, pos_array[i].y,
				pos_array[i].w, pos_array[i].h);
	}
}

void _main_view_del(main_view_data *data)
{
	/* Unregistering all call-backs */
	evas_object_smart_callback_del(data->window, "wm,rotation,changed", _main_view_win_rotate_cb);
	calc_set_display_change_cb(data->model, NULL, NULL);
	data->app->region_fmt_change_cb = NULL;
	data->app->region_fmt_change_cb_data = NULL;

	/* Deleting layout (do not need NULL check) */
	evas_object_del(data->layout);

	/* Deallocation data structure */
	free(data);
}

void _main_view_handle_key_press(main_view_data *data, key_type key)
{
	result_type result = calc_handle_key_press(data->model, key);
	if (RES_MAX_DIGITS_REACHED == result) {
		char msg[DISP_STR_SIZE] = {'\0'};
		snprintf(msg, DISP_STR_SIZE, "<p align=center>"STR_POPUP_MAX_INPUT"</p>", DISP_MAX_DIGITS);
		_main_view_create_msg_popup(data, msg);
	}
}

void _main_view_create_msg_popup(main_view_data *data, const char *msg)
{
	Evas_Object *popup = elm_popup_add(data->window);
	elm_popup_align_set(popup, ELM_NOTIFY_ALIGN_FILL, 1.0);

	elm_object_part_text_set(popup, "default", msg);

	Evas_Object *ok_btn = elm_button_add(popup);
	elm_object_text_set(ok_btn, "OK");
	elm_object_part_content_set(popup, "button1", ok_btn);

	evas_object_show(ok_btn);
	evas_object_show(popup);

	evas_object_smart_callback_add(ok_btn, "clicked", _main_view_popup_close_cb, popup);
	eext_object_event_callback_add(popup, EEXT_CALLBACK_BACK, _main_view_popup_close_cb, popup);
}

void _main_view_update_region_fmt(main_view_data *data)
{
	RETM_IF(!data->dec_btn, "dec_btn is NULL");

	calc_update_region_fmt(data->model);

	elm_object_text_set(data->dec_btn, calc_get_dec_point_str(data->model));
}

void _main_view_update_disp_label(main_view_data *data)
{
	RETM_IF(!data->label, "label is NULL");

	char buff[DISP_STR_SIZE] = {'\0'};
	snprintf(buff, DISP_STR_SIZE, "<font_size=%d>%s</font_size>",
			(int)DISP_NUMBER_TEXT_SIZE, calc_get_display_str(data->model));
	elm_object_text_set(data->label, buff);
}

void _main_view_display_changed_cb(void *data)
{
	_main_view_update_disp_label((main_view_data *)data);
}

void _main_view_region_fmt_changed_cb(void *data)
{
	_main_view_update_region_fmt((main_view_data *)data);
}

void _main_view_win_rotate_cb(void *data, Evas_Object *obj, void *event_info)
{
	main_view_data *view_data = data;
	_main_view_set_orientation(view_data, elm_win_rotation_get(view_data->window));
}

void _main_view_btn_click_cb(void *data, Evas_Object *obj, void *event_info)
{
	main_view_btn_data *btn_data = data;
	_main_view_handle_key_press(btn_data->view, btn_data->key);
}

void _main_view_popup_close_cb(void *data, Evas_Object *obj, void *event_info)
{
	evas_object_del((Evas_Object *)data);
}
