/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "model/edit-contact.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "model/contacts-utils/contacts-record-utils.h"
#include "model/common.h"
#include "utils/logger.h"

#include <stdlib.h>
#include <time.h>


sc_ec_contact_edit *sc_ec_create_new_contact()
{
	bool is_ok = false;
	sc_ec_contact_edit *edit_contact = malloc(sizeof(sc_ec_contact_edit));
	if (edit_contact) {
		is_ok = sc_db_utils_is_success(contacts_record_create(_contacts_contact._uri, &edit_contact->contact_record));
		edit_contact->is_new_contact = true;
	}

	if (!is_ok) {
		sc_ec_destruct_edit_contact(edit_contact);
	}
	return edit_contact;
}

sc_ec_contact_edit *sc_ec_create_edit_contact(int contact_id)
{
	bool is_ok = false;
	sc_ec_contact_edit *edit_contact = malloc(sizeof(sc_ec_contact_edit));
	if (edit_contact) {
		is_ok = sc_db_utils_is_success(contacts_db_get_record(_contacts_contact._uri, contact_id, &edit_contact->contact_record));
		edit_contact->is_new_contact = false;
	}

	if (!is_ok) {
		sc_ec_destruct_edit_contact(edit_contact);
	}
	return edit_contact;
}

void sc_ec_destruct_edit_contact(sc_ec_contact_edit *edit_contact)
{
	RETM_IF(NULL == edit_contact, "edit contact is null");

	if (NULL != edit_contact->contact_record) {
		contacts_record_destroy(edit_contact->contact_record, true);
	}

	free(edit_contact);
}

int sc_ec_save_contact(sc_ec_contact_edit *edit_contact)
{
	RETVM_IF(NULL == edit_contact, false, "edit contact is null");
	RETVM_IF(NULL == edit_contact->contact_record, false, "contact_record is null");
	int ret_value = 0;
	if (edit_contact->is_new_contact) {
		sc_db_utils_is_success(contacts_db_insert_record(edit_contact->contact_record, &ret_value));
	} else {
		if (sc_db_utils_is_success(contacts_db_update_record(edit_contact->contact_record))) {
			sc_db_utils_is_success(contacts_record_get_int(edit_contact->contact_record, _contacts_contact.id, &ret_value));
		}
	}
	return ret_value;
}

bool sc_ec_get_contact_field_str_val(sc_common_contact_fields field_type, contacts_record_h contact_record, char **value)
{
	RETVM_IF(field_type >= CF_MAX || field_type < CF_FIRST_NAME, false, "incorrect field type");
	RETVM_IF(NULL == contact_record, false, "record is null");
	RETVM_IF(NULL == value, false, "pointer to value is null");
	unsigned parent_property_id = 0, child_property_id = 0;
	sc_common_get_contact_properties(field_type, &parent_property_id, NULL, &child_property_id);
	return sc_record_utils_get_child_str_p(contact_record, parent_property_id, child_property_id, value);
}

bool sc_ec_get_contact_field_int_val(sc_common_contact_fields field_type, contacts_record_h contact_record, int *value)
{
	RETVM_IF(field_type >= CF_MAX || field_type < CF_FIRST_NAME, false, "incorrect field type");
	RETVM_IF(NULL == contact_record, false, "record is null");
	RETVM_IF(NULL == value, false, "pointer to value is null");
	unsigned parent_property_id = 0, child_property_id = 0;
	sc_common_get_contact_properties(field_type, &parent_property_id, NULL, &child_property_id);
	return sc_record_utils_get_child_int(contact_record, parent_property_id, child_property_id, value);
}

bool sc_ec_set_contact_field_str_val(sc_common_contact_fields field_type, contacts_record_h contact_record, const char *value)
{
	RETVM_IF(field_type >= CF_MAX || field_type < CF_FIRST_NAME, false, "incorrect field type");
	RETVM_IF(NULL == contact_record, false, "parent_record is null");
	RETVM_IF(NULL == value, false, "value is null");
	unsigned parent_property_id = 0, child_property_id = 0;
	const char *child_uri = NULL;
	sc_common_get_contact_properties(field_type, &parent_property_id, &child_uri, &child_property_id);
	return sc_record_utils_set_child_str(contact_record, parent_property_id, child_uri, child_property_id, value);
}

bool sc_ec_set_contact_field_int_val(sc_common_contact_fields field_type, contacts_record_h contact_record, int value)
{
	RETVM_IF(field_type >= CF_MAX || field_type < CF_FIRST_NAME, false, "incorrect field type");
	RETVM_IF(NULL == contact_record, false, "parent_record is null");
	unsigned parent_property_id = 0, child_property_id = 0;
	const char *child_uri = NULL;
	sc_common_get_contact_properties(field_type, &parent_property_id, &child_uri, &child_property_id);
	return sc_record_utils_set_child_int(contact_record, parent_property_id, child_uri, child_property_id, value);
}

bool sc_ec_is_exist_first_or_last_name(contacts_record_h contact_record)
{
	RETVM_IF(NULL == contact_record, false, "contact_record is null");
	char *first = NULL;
	char *last = NULL;
	sc_record_utils_get_child_str_p(contact_record, _contacts_contact.name, _contacts_name.first, &first);
	if (first && *first) {
		return true;
	}

	sc_record_utils_get_child_str_p(contact_record, _contacts_contact.name, _contacts_name.last, &last);
	if (last && *last) {
		return true;
	}
	return false;
}

struct tm sc_ec_get_date(contacts_record_h record)
{
	struct tm date = {0};
	int int_date = 0;
	if (sc_ec_get_contact_field_int_val(CF_BIRTHDAY, record, &int_date)) {
		date = sc_common_int_to_date(int_date);
	} else {
		time_t now = time(NULL);
		localtime_r(&now, &date);
	}

	return date;
}
