/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "utils/ui-utils.h"
#include "utils/logger.h"

#define PART_POPUP_BUTTON1      "button1"
#define PART_POPUP_BUTTON2      "button2"
#define PART_POPUP_BUTTON3      "button3"
#define STYLE_POPUP_BUTTON      "popup_button/default"
#define BG_COLOR                255, 255, 255

static void _editfield_focused_cb(void *data, Evas_Object *obj, void *event_info)
{
	elm_object_signal_emit(data, "elm,state,focused", "");
}

static void _editfield_unfocused_cb(void *data, Evas_Object *obj, void *event_info)
{
	elm_object_signal_emit(data, "elm,state,unfocused", "");
}

static void _editfield_button_cb(void *data, Evas_Object *obj, void *event_info)
{
	elm_entry_entry_set(data, "");
	elm_object_focus_set(data, EINA_TRUE);
}

static void _editfield_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	if (elm_entry_is_empty(obj)) {
		elm_object_signal_emit(data, "elm,action,hide,button", "");
	} else {
		elm_object_signal_emit(data, "elm,action,show,button", "");
	}
}

Evas_Object *ui_utils_editfield_add(Evas_Object *parent, const char *guide_text, bool multiline)
{
	Evas_Object *layout = elm_layout_add(parent);
	Evas_Object *entry = elm_entry_add(layout);

	if (multiline) {
		elm_layout_theme_set(layout, "layout", "editfield", "multiline");
		elm_entry_single_line_set(entry, EINA_FALSE);
	} else {
		elm_layout_theme_set(layout, "layout", "editfield", "singleline");
		elm_entry_single_line_set(entry, EINA_TRUE);
		elm_entry_scrollable_set(entry, EINA_TRUE);
	}

	elm_object_part_text_set(entry, PART_ENTRY_GUIDE, guide_text);
	elm_object_part_content_set(layout, "elm.swallow.content", entry);

	Evas_Object *button = elm_button_add(layout);
	elm_object_style_set(button, "editfield_clear");
	elm_object_focus_allow_set(button, EINA_FALSE);
	elm_object_part_content_set(layout, "elm.swallow.button", button);
	evas_object_smart_callback_add(button, EVENT_CLICKED, _editfield_button_cb, entry);

	evas_object_smart_callback_add(entry, EVENT_FOCUSED, _editfield_focused_cb, layout);
	evas_object_smart_callback_add(entry, EVENT_UNFOCUSED, _editfield_unfocused_cb, layout);
	evas_object_smart_callback_add(entry, EVENT_CHANGED, _editfield_changed_cb, layout);

	return layout;
}

Evas_Object *ui_utils_navi_add(Evas_Object *parent, Eext_Event_Cb back_cb, void *cb_data)
{
	RETVM_IF(NULL == parent, NULL, "parent is null");
	Evas_Object *navi = elm_naviframe_add(parent);
	eext_object_event_callback_add(navi, EEXT_CALLBACK_BACK, back_cb, cb_data);
	eext_object_event_callback_add(navi, EEXT_CALLBACK_MORE, eext_naviframe_more_cb, NULL);
	elm_naviframe_prev_btn_auto_pushed_set(navi, EINA_FALSE);
	return navi;
}

Evas_Object *ui_utils_layout_add(Evas_Object *parent, Evas_Object_Event_Cb destroy_cb, void *cb_data)
{
	Evas_Object *layout = elm_layout_add(parent);
	RETVM_IF(NULL == layout, NULL, "layout is null");

	elm_layout_theme_set(layout, "layout", "application", "default");

	Evas_Object *bg = elm_bg_add(layout);
	elm_bg_color_set(bg, BG_COLOR);
	elm_object_part_content_set(layout, PART_BG, bg);

	evas_object_event_callback_add(layout, EVAS_CALLBACK_FREE, destroy_cb, cb_data);
	return layout;
}

Evas_Object *ui_utils_toolbar_add(Evas_Object *parent)
{
	Evas_Object *toolbar = elm_toolbar_add(parent);
	RETVM_IF(NULL == toolbar, NULL, "toolbar is null");

	elm_toolbar_shrink_mode_set(toolbar, ELM_TOOLBAR_SHRINK_EXPAND);
	elm_toolbar_transverse_expanded_set(toolbar, EINA_TRUE);
	return toolbar;
}

Evas_Object *ui_utils_popup_button_add(Evas_Object *popup, const char *text, Evas_Smart_Cb func, void *data)
{
	RETVM_IF(NULL == popup, NULL, "popup is null");
	static const char *button_parts[] = {
		PART_POPUP_BUTTON1,
		PART_POPUP_BUTTON2,
		PART_POPUP_BUTTON3,
		NULL
	};
	const char **part = button_parts;
	Evas_Object *button = NULL;

	while (*part && elm_object_part_content_get(popup, *part)) {
		++part;
	}

	RETVM_IF(NULL == part, NULL, "No empty button parts left");

	button = elm_button_add(popup);
	RETVM_IF(NULL == button, NULL, "Failed to create button");
	elm_object_style_set(button, STYLE_POPUP_BUTTON);
	elm_object_text_set(button, text);
	elm_object_part_content_set(popup, *part, button);
	evas_object_smart_callback_add(button, EVENT_CLICKED, func, data);
	return button;
}

static void _popup_cancel_cb(void *data, Evas_Object *obj, void *event_info)
{
	evas_object_del(data);
}

Evas_Object *ui_utils_popup_add(Evas_Object *parent, const char *title, const char *cancel_btn)
{
	RETVM_IF(!title, NULL, "title is NULL");
	RETVM_IF(!cancel_btn, NULL, "cancel_btn is NULL");

	Evas_Object *popup = elm_popup_add(parent);
	RETVM_IF(NULL == popup, NULL, "elm_popup_add() failed");
	elm_object_part_text_set(popup, PART_POPUP_TITLE, title);

	elm_popup_align_set(popup, ELM_NOTIFY_ALIGN_FILL, 1.0);
	evas_object_size_hint_weight_set(popup, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	eext_object_event_callback_add(popup, EEXT_CALLBACK_BACK, _popup_cancel_cb, popup);
	ui_utils_popup_button_add(popup, cancel_btn, _popup_cancel_cb, popup);
	evas_object_show(popup);

	return popup;
}
