/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


#include "view/contact-details-view.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "model/common.h"
#include "model/contact-details.h"
#include "utils/logger.h"
#include "utils/ui-utils.h"
#include "view/view-common.h"
#include "view/edit-contact-view.h"

#include <contacts.h>
#include <Elementary.h>

#define CT_DETAILS_VIEW_DATA                    "ct_details_view_data"

static const char *_caption_fav                 = "Fav.";
static const char *_caption_green_fav           = "<color=#0F0>Fav.</>";
static const char *_caption_edit                = "Edit";
static const char *_caption_delete              = "Delete";
static const char *_caption_warning             = "Warning";
static const char *_caption_delete_request      = "Do you really want to delete?";

typedef struct {
	Evas_Object *navi;
	Elm_Object_Item *navi_item;

	int person_id;
	int contact_id;

	Evas_Object *genlist;
	Evas_Object *popup;
} contact_details_view_data;

static void _cdetails_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _cdetails_view_update_details_cb(const char *view_uri, void *user_data);
static void _cdetails_view_fill(contact_details_view_data *data);

static void _cdetails_view_add_fav_btn(contact_details_view_data *data);
static void _cdetails_view_fav_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info);

static char *_cdetails_view_item_text_get_cb(void *data, Evas_Object *obj, const char *part);

static Elm_Genlist_Item_Class *_cdetails_view_create_item_class();

static Evas_Object *_cdetails_view_create_toolbar(contact_details_view_data *data);
static void _cdetails_view_edit_item_cb(void *data, Evas_Object *obj, void *event_info);
static void _cdetails_view_delete_item_cb(void *data, Evas_Object *obj, void *event_info);
static void _cdetails_view_popup_cancel_cb(void *data, Evas_Object *obj, void *event_info);
static void _cdetails_view_popup_ok_cb(void *data, Evas_Object *obj, void *event_info);


void sc_cdetails_view_add(Evas_Object *parent, int person_id)
{
	RETM_IF(!parent, "parent is null");

	contact_details_view_data *data = calloc(1, sizeof(contact_details_view_data));
	RETM_IF(!data, "data not allocated");
	data->navi = parent;

	data->person_id = person_id;
	if (data->person_id <= 0) {
		_cdetails_view_destroy(data, NULL, NULL, NULL);
		return;
	}

	contacts_record_h person_record = NULL;
	char *person_name = NULL;
	if (sc_db_utils_is_success(contacts_db_get_record(_contacts_person._uri, data->person_id, &person_record))) {
		sc_db_utils_is_success(contacts_record_get_int(person_record, _contacts_person.display_contact_id, &data->contact_id));
		sc_db_utils_is_success(contacts_record_get_str(person_record, _contacts_person.display_name, &person_name));
		contacts_record_destroy(person_record, false);
	}
	if (data->contact_id <= 0) {
		_cdetails_view_destroy(data, NULL, NULL, NULL);
		return;
	}

	data->genlist = elm_genlist_add(data->navi);
	elm_genlist_mode_set(data->genlist, ELM_LIST_COMPRESS);

	evas_object_event_callback_add(data->genlist, EVAS_CALLBACK_FREE, _cdetails_view_destroy, data);
	char *temp_str = elm_entry_utf8_to_markup(person_name);
	data->navi_item = elm_naviframe_item_push(data->navi, temp_str, NULL, NULL, data->genlist, NULL);
	free(temp_str);
	free(person_name);

	sc_db_utils_is_success(contacts_db_add_changed_cb(_contacts_person._uri, _cdetails_view_update_details_cb, data));
	_cdetails_view_fill(data);

	Evas_Object *toolbar = _cdetails_view_create_toolbar(data);
	if (toolbar) {
		elm_object_item_part_content_set(data->navi_item, PART_TOOLBAR, toolbar);
	}

	if (data->navi_item) {
		_cdetails_view_add_fav_btn(data);
	}
}

static void _cdetails_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is null");
	contacts_db_remove_changed_cb(_contacts_person._uri, _cdetails_view_update_details_cb, data);
	free(data);
}

static void _cdetails_view_update_details_cb(const char *view_uri, void *user_data)
{
	RETM_IF(!user_data, "user_data is null");
	contact_details_view_data *data = user_data;
	char *person_name = NULL;
	if (data->person_id > 0) {
		contacts_record_h person_record = NULL;
		sc_db_utils_is_success(contacts_db_get_record(_contacts_person._uri, data->person_id, &person_record));

		if (person_record) {
			sc_db_utils_is_success(contacts_record_get_str(person_record, _contacts_person.display_name, &person_name));
			contacts_record_destroy(person_record, false);
		} else {
			sc_view_common_return_to_view(data->navi, VI_PERSON_LIST);
			return;
		}

	}
	char *temp_str = elm_entry_utf8_to_markup(person_name);
	elm_object_item_text_set(data->navi_item, temp_str);

	free(temp_str);
	free(person_name);
	elm_genlist_clear(data->genlist);

	_cdetails_view_fill(data);
}


static void _cdetails_view_add_fav_btn(contact_details_view_data *data)
{
	RETM_IF(!data, "data is null");

	Evas_Object *fav_btn = elm_button_add(data->navi);

	elm_object_text_set(fav_btn, sc_cdetails_is_favourite(data->person_id) ? _caption_green_fav : _caption_fav);
	elm_object_style_set(fav_btn, STYLE_BUTTON_NAVIFRAME);
	elm_object_item_part_content_set(data->navi_item, PART_NAVI_TITLE_RIGHT_BUTTON, fav_btn);

	evas_object_smart_callback_add(fav_btn, EVENT_CLICKED, _cdetails_view_fav_btn_clicked_cb, data);
}

static void _cdetails_view_fav_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	contact_details_view_data *view = data;
	sc_cdetails_toggle_favourites(view->person_id);
	elm_object_text_set(obj, sc_cdetails_is_favourite(view->person_id) ? _caption_green_fav : _caption_fav);
}

static void _cdetails_view_fill(contact_details_view_data *data)
{
	RETM_IF(!data, "data is null");
	Elm_Genlist_Item_Class *itc = _cdetails_view_create_item_class();

	evas_object_data_set(data->genlist, CT_DETAILS_VIEW_DATA, data);

	sc_common_contact_fields i = CF_PHONE_NUMBER;
	for (; i < CF_MAX; ++i) {
		Elm_Object_Item *item = elm_genlist_item_append(data->genlist, itc, (void *)i, NULL,
				ELM_GENLIST_ITEM_NONE, NULL, NULL);
		elm_genlist_item_select_mode_set(item, ELM_OBJECT_SELECT_MODE_NONE);
	}

	elm_genlist_item_class_free(itc);
}

static Elm_Genlist_Item_Class *_cdetails_view_create_item_class()
{
	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();
	itc->item_style = STYLE_ITEM_DETAILS;
	itc->func.text_get = _cdetails_view_item_text_get_cb;
	return itc;
}

static char *_cdetails_view_item_text_get_cb(void *data, Evas_Object *obj, const char *part)
{
	if (!strcmp(part, PART_TEXT)) {
		const char *name = sc_common_get_contact_field_name((sc_common_contact_fields)data);
		if (name) {
			return strdup(name);
		}
	} else if (!strcmp(part, PART_TEXT_MULTILINE)) {
		contact_details_view_data *cv_data = evas_object_data_get(obj, CT_DETAILS_VIEW_DATA);
		if (cv_data) {
			char *text = NULL;
			if (sc_cdetails_get_contact_val((sc_common_contact_fields)data, cv_data->contact_id, &text)) {
				char *markup_str = elm_entry_utf8_to_markup(text);
				free(text);
				return markup_str;
			}
		}
	}

	return NULL;
}

static Evas_Object *_cdetails_view_create_toolbar(contact_details_view_data *data)
{
	RETVM_IF(NULL == data, NULL, "data is null");

	Evas_Object *toolbar = ui_utils_toolbar_add(data->genlist);
	RETVM_IF(NULL == toolbar, NULL, "toolbar is null");
	elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_NONE);
	elm_toolbar_item_append(toolbar, NULL, _caption_edit, _cdetails_view_edit_item_cb, data);
	elm_toolbar_item_append(toolbar, NULL, _caption_delete, _cdetails_view_delete_item_cb, data);
	return toolbar;
}

static void _cdetails_view_edit_item_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	contact_details_view_data *cd_data = data;
	sc_ec_view_add(cd_data->navi, cd_data->contact_id);
}

static void _cdetails_view_delete_item_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	contact_details_view_data *cd_data = data;

	cd_data->popup = elm_popup_add(cd_data->navi);
	RETM_IF(NULL == cd_data->popup, "popup is null");
	elm_object_part_text_set(cd_data->popup, PART_POPUP_TITLE, _caption_warning);
	elm_object_text_set(cd_data->popup, _caption_delete_request);
	eext_object_event_callback_add(cd_data->popup, EEXT_CALLBACK_BACK, eext_popup_back_cb, NULL);

	ui_utils_popup_button_add(cd_data->popup, _caption_cancel, _cdetails_view_popup_cancel_cb, cd_data);
	ui_utils_popup_button_add(cd_data->popup, _caption_ok, _cdetails_view_popup_ok_cb, cd_data);
	evas_object_show(cd_data->popup);
}

static void _cdetails_view_popup_cancel_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	contact_details_view_data *cd_data = data;
	evas_object_del(cd_data->popup);
	cd_data->popup = NULL;
}

static void _cdetails_view_popup_ok_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	RETM_IF(NULL == obj, "obj is null");
	contact_details_view_data *cd_data = data;
	if (cd_data->contact_id > 0) {
		sc_db_utils_is_success(contacts_db_delete_record(_contacts_contact._uri, cd_data->contact_id));
	}
	evas_object_del(cd_data->popup);
	cd_data->popup = NULL;
	elm_naviframe_item_pop(cd_data->navi);
}
