/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "view/create-group-view.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "model/create-group.h"
#include "model/common.h"
#include "utils/logger.h"
#include "utils/ui-utils.h"
#include "view/view-common.h"

#include <contacts.h>
#include <Elementary.h>

#define PART_ELM_ICON_ENTRY         "elm.icon.entry"

static const char *_caption_category_name       = "Category name";
static const char *_caption_categories          = "Categories";
static const char *_caption_error = "Error";
static const char *_caption_group_already_exists = "Group already exists";

typedef struct {
	Evas_Object *navi;
	Elm_Object_Item *navi_item;

	Evas_Object *layout;
	Evas_Object *popup;
	Evas_Object *entry;
	Elm_Object_Item *save_button;
} create_group_view_data;

static void _gcreate_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _gcreate_view_create_layout(create_group_view_data *view);
static Evas_Object *_gcreate_view_create_editfield(Evas_Object *parent, create_group_view_data *view);
static void _gcreate_view_entry_changed(void *data, Evas_Object *obj, void *event_info);

static void _gcreate_view_add_toolbar(create_group_view_data *view);
static void _gcreate_view_cancel_cb(void *data, Evas_Object *obj, void *event_info);
static void _gcreate_view_create_group_cb(void *data, Evas_Object *obj, void *event_info);
static void _gcreate_view_popup_ok_cb(void *data, Evas_Object *obj, void *event_info);

void sc_gcreate_view_add(Evas_Object *parent)
{
	RETM_IF(!parent, "parent is NULL");

	create_group_view_data *view = calloc(1, sizeof(create_group_view_data));
	RETM_IF(!view, "calloc failed");
	view->navi = parent;

	_gcreate_view_create_layout(view);
	if (!view->layout) {
		_gcreate_view_destroy(view, NULL, NULL, NULL);
	}

	view->navi_item = elm_naviframe_item_push(view->navi, _caption_categories, NULL, NULL, view->layout, NULL);

	_gcreate_view_add_toolbar(view);
}

static void _gcreate_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	free(data);
}

static void _gcreate_view_create_layout(create_group_view_data *view)
{
	RETM_IF(!view, "view is NULL");
	RETM_IF(!view->navi, "navi is NULL");

	view->layout = ui_utils_layout_add(view->navi, _gcreate_view_destroy, view);

	Evas_Object *box = elm_box_add(view->layout);

	Evas_Object *editfield = _gcreate_view_create_editfield(box, view);
	evas_object_size_hint_weight_set(editfield, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(editfield, EVAS_HINT_FILL, 0);

	elm_box_pack_end(box, editfield);
	evas_object_show(editfield);

	elm_object_part_content_set(view->layout, PART_CONTENT, box);
}

static Evas_Object *_gcreate_view_create_editfield(Evas_Object *parent, create_group_view_data *view)
{
	Evas_Object *editfield = ui_utils_editfield_add(parent, _caption_category_name, false);
	if (editfield) {
		view->entry = elm_object_part_content_get(editfield, PART_CONTENT);
		evas_object_smart_callback_add(view->entry, EVENT_CURSOR_CHANGED, _gcreate_view_entry_changed, view);
	}

	return editfield;
}

static void _gcreate_view_entry_changed(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	create_group_view_data *view = data;
	RETM_IF(!view->entry, "entry is NULL");

	char *name = sc_common_trim_spaces_in_string(elm_object_text_get(view->entry));
	if (name && *name) {
		elm_object_item_disabled_set(view->save_button, EINA_FALSE);
		free(name);
	} else {
		elm_object_item_disabled_set(view->save_button, EINA_TRUE);
	}
}

static void _gcreate_view_add_toolbar(create_group_view_data *view)
{
	RETM_IF(!view, "data is NULL");
	RETM_IF(!view->navi, "parent is NULL");
	RETM_IF(!view->navi_item, "navi item is NULL");

	Evas_Object *toolbar = ui_utils_toolbar_add(view->navi);
	if (toolbar) {
		elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_NONE);
		view->save_button = elm_toolbar_item_append(toolbar, NULL, _caption_save, _gcreate_view_create_group_cb, view);
		elm_object_item_disabled_set(view->save_button, EINA_TRUE);
		elm_toolbar_item_append(toolbar, NULL, _caption_cancel, _gcreate_view_cancel_cb, view);
		elm_object_item_part_content_set(view->navi_item, PART_TOOLBAR, toolbar);
	}
}

static void _gcreate_view_cancel_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	create_group_view_data *view = data;
	RETM_IF(!view->navi, "navi is NULL");
	elm_naviframe_item_pop(view->navi);
}

static void _gcreate_view_create_group_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	create_group_view_data *view = data;
	RETM_IF(!view->entry, "entry is NULL");
	const char *name = elm_object_text_get(view->entry);

	if (!sc_gcreate_is_group_exist(name)) {
		if (!sc_gcreate_db_create_group(name)) {
			ERR("sc_gcreate_db_create_group failed");
		}
		RETM_IF(!view->navi, "navi is NULL");
		elm_naviframe_item_pop(view->navi);
	} else {
		view->popup = elm_popup_add(view->navi);
		RETM_IF(!view->popup, "popup is null");
		elm_object_part_text_set(view->popup, PART_POPUP_TITLE, _caption_error);
		elm_object_text_set(view->popup, _caption_group_already_exists);
		ui_utils_popup_button_add(view->popup, _caption_ok, _gcreate_view_popup_ok_cb, view);
		eext_object_event_callback_add(view->popup, EEXT_CALLBACK_BACK, _gcreate_view_popup_ok_cb, data);
		evas_object_show(view->popup);
	}
}

static void _gcreate_view_popup_ok_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	create_group_view_data *view = data;
	evas_object_del(view->popup);
	view->popup = NULL;
}
