/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "view/edit-contact-view.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "model/contacts-utils/contacts-record-utils.h"
#include "model/common.h"
#include "model/edit-contact.h"
#include "utils/logger.h"
#include "utils/ui-utils.h"
#include "view/view-common.h"

#include <Elementary.h>
#include <Evas.h>
#include <time.h>

#define STYLE_ITEM_EDIT                 "full"
#define STYLE_ITEM_BIRTHDAY             "full"

#define CONTACT_VIEW_DATA           "contact_view_data"
#define DATETIME_POPUP              "date_time_popup"

static const char *_caption_create_contact          = "Create contact";
static const char *_caption_edit_contact            = "Edit contact";
static const char *_caption_info                    = "Info";
static const char *_caption_enter_fname_or_lname    = "Enter the first name or the last name";

typedef struct {
	Evas_Object *navi;
	Elm_Object_Item *navi_item;
	Evas_Object *layout;
	int contact_id;
	sc_ec_contact_edit *data;

	Evas_Object *genlist;
	Elm_Object_Item *birthday_item;

	Evas_Object *datetime;
} contact_edit_view_data;


static void _ec_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _ec_view_fill(contact_edit_view_data *data);

static Elm_Genlist_Item_Class *_ec_view_create_edit_item_class();
static Elm_Genlist_Item_Class *_ec_view_create_birthday_item_class();

static Evas_Object *_ec_view_item_content_get_cb(void *data, Evas_Object *obj, const char *part);
static Evas_Object *_ec_view_birthday_item_content_get_cb(void *data, Evas_Object *obj, const char *part);

static void _ec_view_create_datetime_popup_cb(void *data, Evas_Object *obj, void *event_info);
static Evas_Object *_ec_view_create_datetime(Evas_Object *parent, contact_edit_view_data *view);

static void _ec_view_entry_keyboard_layout_set(Evas_Object *entry, sc_common_contact_fields field_type);
static void _ec_view_entry_unfocused_cb(void *data, Evas_Object *obj, void *event_info);
static void _ec_view_entry_filter_cb(void *data, Evas_Object *entry, char **text);

static void _ec_view_add_save_btn(contact_edit_view_data *data);
static void _ec_view_save_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info);
static void _ec_view_datetime_changed_cb(void *data, Evas_Object *obj, void *event_info);

static void _ec_view_del_object_cb(void *data, Evas_Object *obj, void *event_info);

static void _ec_view_info_popup(Evas_Object *parent);

static void _ec_view_changed_cb(const char *view_uri, void *user_data);

void sc_ec_view_add(Evas_Object *parent, int contact_id)
{
	RETM_IF(NULL == parent, "parent is null");

	contact_edit_view_data *data = calloc(1, sizeof(contact_edit_view_data));
	RETM_IF(NULL == data, "data not allocated");
	data->navi = parent;

	if (contact_id > 0) {
		data->contact_id = contact_id;
		data->data = sc_ec_create_edit_contact(data->contact_id);
	} else {
		data->contact_id = 0;
		data->data = sc_ec_create_new_contact();
	}
	if (!data->data) {
		ERR("contact_edit_data is not created");
		_ec_view_destroy(data, NULL, NULL, NULL);
		return;
	}

	data->layout = ui_utils_layout_add(data->navi, _ec_view_destroy, data);
	RETM_IF(NULL == data->layout, "layout is null");
	contacts_db_add_changed_cb(_contacts_contact._uri, _ec_view_changed_cb, data);

	data->genlist = elm_genlist_add(data->layout);
	elm_object_part_content_set(data->layout, PART_CONTENT, data->genlist);

	_ec_view_fill(data);
	data->navi_item = elm_naviframe_item_push(data->navi,
			contact_id > 0 ? _caption_edit_contact : _caption_create_contact,
			NULL, NULL, data->layout, STYLE_NAVI_ITEM_TABBAR);

	_ec_view_add_save_btn(data);
}

static void _ec_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	contact_edit_view_data *view = data;
	sc_ec_destruct_edit_contact(view->data);
	contacts_db_remove_changed_cb(_contacts_contact._uri, _ec_view_changed_cb, data);
	free(data);
}

static void _ec_view_changed_cb(const char *view_uri, void *user_data)
{
	RETM_IF(!user_data, "data is NULL");
	contact_edit_view_data *view = user_data;
	contacts_record_h record = NULL;
	sc_db_utils_is_success(contacts_db_get_record(_contacts_contact._uri, view->contact_id, &record));
	if (!record) {
		sc_view_common_return_to_view(view->navi, VI_PERSON_LIST);
	}
	contacts_record_destroy(record, false);
}

static Elm_Genlist_Item_Class *_ec_view_create_edit_item_class()
{
	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();
	itc->item_style = STYLE_ITEM_EDIT;
	itc->func.content_get = _ec_view_item_content_get_cb;
	return itc;
}

static Elm_Genlist_Item_Class *_ec_view_create_birthday_item_class()
{
	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();
	itc->item_style = STYLE_ITEM_BIRTHDAY;
	itc->func.content_get = _ec_view_birthday_item_content_get_cb;
	return itc;
}

static void _ec_view_fill(contact_edit_view_data *data)
{
	RETM_IF(NULL == data, "data is null");
	Elm_Genlist_Item_Class *itc_birthday = _ec_view_create_birthday_item_class();
	Elm_Genlist_Item_Class *itc_entry = _ec_view_create_edit_item_class();

	evas_object_data_set(data->genlist, CONTACT_VIEW_DATA, data);

	sc_common_contact_fields i = CF_FIRST_NAME;
	for (; i < CF_MAX; ++i) {
		if (CF_BIRTHDAY == i) {
			data->birthday_item = elm_genlist_item_append(data->genlist, itc_birthday, (void *)i, NULL,
					ELM_GENLIST_ITEM_NONE, NULL, NULL);
		} else {
			elm_genlist_item_append(data->genlist, itc_entry, (void *)i, NULL,
					ELM_GENLIST_ITEM_NONE, NULL, NULL);
		}
	}

	elm_genlist_item_class_free(itc_birthday);
	elm_genlist_item_class_free(itc_entry);
}

static void _ec_view_entry_keyboard_layout_set(Evas_Object *entry, sc_common_contact_fields field_type)
{
	RETM_IF(field_type >= CF_MAX || field_type < CF_FIRST_NAME, "incorrect field type");
	switch (field_type) {
		case CF_PHONE_NUMBER:
		{
			elm_entry_input_panel_layout_set(entry, ELM_INPUT_PANEL_LAYOUT_PHONENUMBER);
			elm_entry_markup_filter_append(entry, _ec_view_entry_filter_cb, NULL);
			break;
		}
		case CF_EMAIL:
		{
			elm_entry_input_panel_layout_set(entry, ELM_INPUT_PANEL_LAYOUT_EMAIL);
			break;
		}
		case CF_URL:
		{
			elm_entry_input_panel_layout_set(entry, ELM_INPUT_PANEL_LAYOUT_URL);
			break;
		}
		case CF_FIRST_NAME:
		case CF_LAST_NAME:
		case CF_BIRTHDAY:
		case CF_COMPANY:
		case CF_NOTE:
		case CF_MAX:
		{
			/* Not needed */
			break;
		}
	}
}

static Evas_Object *_ec_view_item_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	RETVM_IF(NULL == obj, NULL, "genlist is null");
	sc_common_contact_fields field_type = (sc_common_contact_fields)data;
	RETVM_IF(field_type >= CF_MAX || field_type < CF_FIRST_NAME, NULL, "incorrect field type");

	Evas_Object *edit_layout = NULL;
	contact_edit_view_data *cv_data = evas_object_data_get(obj, CONTACT_VIEW_DATA);
	if (cv_data) {
		edit_layout = ui_utils_editfield_add(cv_data->genlist,
				sc_common_get_contact_field_name(field_type), CF_NOTE == field_type);
		Evas_Object *entry = elm_object_part_content_get(edit_layout, PART_CONTENT);

		if (entry && cv_data->data) {
			_ec_view_entry_keyboard_layout_set(entry, field_type);

			elm_entry_scrollable_set(entry, EINA_TRUE);
			char *text = NULL;
			if (sc_ec_get_contact_field_str_val(field_type, cv_data->data->contact_record, &text)) {
				char *markup_str = elm_entry_utf8_to_markup(text);
				elm_entry_entry_set(entry, markup_str);
				free(markup_str);
			}
			evas_object_smart_callback_add(entry, EVENT_UNFOCUSED, _ec_view_entry_unfocused_cb, (void *)field_type);
			evas_object_data_set(entry, CONTACT_VIEW_DATA, cv_data);
		}
	}
	return edit_layout;
}

static Evas_Object *_ec_view_birthday_item_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	contact_edit_view_data *cv_data = evas_object_data_get(obj, CONTACT_VIEW_DATA);

	Evas_Object *button = elm_button_add(obj);
	struct tm date = sc_ec_get_date(cv_data->data->contact_record);
	char *str_date = sc_common_date_to_str(&date);
	elm_object_text_set(button, str_date);
	free(str_date);
	evas_object_smart_callback_add(button, EVENT_CLICKED, _ec_view_create_datetime_popup_cb, cv_data);

	return button;
}

static void _ec_view_create_datetime_popup_cb(void *data, Evas_Object *obj, void *event_info)
{
	contact_edit_view_data *cv_data = data;

	Evas_Object *popup = ui_utils_popup_add(cv_data->navi, "Birthday", "Cancel");
	cv_data->datetime = _ec_view_create_datetime(popup, cv_data);
	elm_object_content_set(popup, cv_data->datetime);
	ui_utils_popup_button_add(popup, "Ok", _ec_view_datetime_changed_cb, popup);

	evas_object_data_set(popup, CONTACT_VIEW_DATA, cv_data);

	evas_object_show(popup);
}

static Evas_Object *_ec_view_create_datetime(Evas_Object *parent, contact_edit_view_data *view)
{
	RETVM_IF(NULL == view, NULL, "cv_data is null");
	RETVM_IF(NULL == view->data, NULL, "cv_data->data is null");

	Evas_Object *datetime = elm_datetime_add(parent);
	RETVM_IF(NULL == datetime, NULL, "datetime is not created");
	elm_datetime_format_set(datetime, "%%x");

	struct tm date = sc_ec_get_date(view->data->contact_record);
	elm_datetime_value_set(datetime, &date);

	return datetime;
}

static void _ec_view_entry_unfocused_cb(void *data, Evas_Object *obj, void *event_info)
{
	sc_common_contact_fields field_type = (sc_common_contact_fields)data;
	RETM_IF(field_type >= CF_MAX || field_type < CF_FIRST_NAME, "incorrect field type");
	contact_edit_view_data *cv_data = evas_object_data_get(obj, CONTACT_VIEW_DATA);
	RETM_IF(NULL == cv_data, "contact edit view data is null");
	RETM_IF(NULL == cv_data->data, "contact model data is null");
	RETM_IF(NULL == obj, "entry is null");

	char *utf8_str = elm_entry_markup_to_utf8(elm_entry_entry_get(obj));
	char *temp_str = sc_common_trim_spaces_in_string(utf8_str);
	sc_ec_set_contact_field_str_val(field_type, cv_data->data->contact_record, temp_str);
	free(temp_str);
	free(utf8_str);
}

static void _ec_view_entry_filter_cb(void *data, Evas_Object *entry, char **text)
{
	RETM_IF(!text, "text is NULL");
	const char *pattern = "p+*#";

	char *it = *text;
	while (*it) {
		if (!(isdigit(*it) || strchr(pattern, *it))) {
			free(*text);
			*text = NULL;
			return;
		}
		++it;
	}
}

static void _ec_view_add_save_btn(contact_edit_view_data *data)
{
	RETM_IF(NULL == data, "data is null");
	RETM_IF(NULL == data->navi, "navi is null");
	RETM_IF(NULL == data->navi_item, "navi item is null");

	Evas_Object *toolbar = ui_utils_toolbar_add(data->navi);
	elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_NONE);
	elm_toolbar_item_append(toolbar, NULL, _caption_save, _ec_view_save_btn_clicked_cb, data);
	elm_object_item_part_content_set(data->navi_item, PART_TOOLBAR, toolbar);
}

static void _ec_view_save_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	contact_edit_view_data *cv_data = data;
	RETM_IF(NULL == cv_data->data, "model data is null");
	RETM_IF(NULL == cv_data->data, "contact model data is null");

	if (sc_ec_is_exist_first_or_last_name(cv_data->data->contact_record)) {
		if (sc_ec_save_contact(cv_data->data) == 0) {
			ERR("contact is not saved");
		}
		RETM_IF(NULL == cv_data->navi, "navi is null");
		elm_naviframe_item_pop(cv_data->navi);
	} else {
		_ec_view_info_popup(cv_data->navi);
	}
}

static void _ec_view_datetime_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *popup = data;
	RETM_IF(!popup, "popup is NULL");
	contact_edit_view_data *view = evas_object_data_get(popup, CONTACT_VIEW_DATA);
	RETM_IF(!view, "view is null");
	RETM_IF(!view->data, "view->data is null");
	RETM_IF(!view->datetime, "datetime is NULL");
	struct tm date;
	if (elm_datetime_value_get(view->datetime, &date)) {
		if (sc_ec_set_contact_field_int_val(CF_BIRTHDAY, view->data->contact_record, sc_common_date_to_int(&date))) {
			sc_record_utils_set_child_int(view->data->contact_record, _contacts_contact.event, _contacts_event._uri, _contacts_event.type, CONTACTS_EVENT_TYPE_BIRTH);
		} else {
			ERR("Error setting birthday");
		}
	} else {
		ERR("Date not received");
	}
	evas_object_del(popup);
	elm_genlist_item_update(view->birthday_item);
}

static void _ec_view_del_object_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	Evas_Object *obj_to_del = data;
	evas_object_del(obj_to_del);
	if (event_info) {
		Elm_Object_Item *item = event_info;
		elm_genlist_item_selected_set(item, EINA_FALSE);
	}
}

static void _ec_view_info_popup(Evas_Object *parent)
{
	Evas_Object *popup = elm_popup_add(parent);
	RETM_IF(NULL == popup, "popup is not created");
	elm_object_part_text_set(popup, PART_POPUP_TITLE, _caption_info);
	elm_object_text_set(popup, _caption_enter_fname_or_lname);
	ui_utils_popup_button_add(popup, _caption_ok, _ec_view_del_object_cb, popup);
	evas_object_show(popup);
	eext_object_event_callback_add(popup, EEXT_CALLBACK_BACK, eext_popup_back_cb, NULL);
}
