/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "view/favourites-view.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "model/contacts-utils/contacts-list-utils.h"
#include "utils/logger.h"
#include "view/contact-details-view.h"

#include <Elementary.h>


#define DATA_LIST               "data_list"

static const char *_caption_favourites = "Favourites";

typedef struct {
	Evas_Object *navi;
	Elm_Object_Item *navi_item;
	Evas_Object *list;

	contacts_list_h fav_list;
} favourites_view_data;

static void _favourites_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _favourites_view_fill(favourites_view_data *data);
static void _favourites_view_sel_cb(void *data, Evas_Object *obj, void *event_info);

static void _favourites_view_update_list_cb(const char *view_uri, void *user_data);

void sc_favourites_view_add(Evas_Object *parent)
{
	RETM_IF(!parent, "parent is null");
	favourites_view_data *data = calloc(1, sizeof(favourites_view_data));
	RETM_IF(!data, "calloc failed");
	data->navi = parent;

	data->list = elm_list_add(data->navi);
	if (!data->list) {
		ERR("list is not created");
		free(data);
		return;
	}

	_favourites_view_fill(data);

	evas_object_event_callback_add(data->list, EVAS_CALLBACK_FREE, _favourites_view_destroy, data);
	sc_db_utils_is_success(contacts_db_add_changed_cb(_contacts_person._uri, _favourites_view_update_list_cb, data));
	data->navi_item = elm_naviframe_item_push(data->navi, _caption_favourites, NULL, NULL, data->list, NULL);
}

static void _favourites_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	RETM_IF(NULL == data, "data is null");
	favourites_view_data *fav_data = data;
	contacts_list_destroy(fav_data->fav_list, true);
	contacts_db_remove_changed_cb(_contacts_person._uri, _favourites_view_update_list_cb, data);
	free(fav_data);
}

static void _favourites_view_fill(favourites_view_data *data)
{
	RETM_IF(!data, "data is null");
	contacts_record_h record = NULL;

	data->fav_list = sc_list_utils_get_records_list_bool_filter(_contacts_person._uri, _contacts_person.is_favorite, true);
	RETM_IF(!data->fav_list, "Favourite list is not retrieved");

	CONTACTS_LIST_FOREACH(data->fav_list, record) {
		char *name = NULL;
		if (sc_db_utils_is_success(contacts_record_get_str_p(record, _contacts_person.display_name, &name))) {
			char *temp_str = elm_entry_utf8_to_markup(name);
			elm_list_item_append(data->list, temp_str, NULL, NULL, _favourites_view_sel_cb, record);
			free(temp_str);
			evas_object_data_set(data->list, DATA_LIST, data);
		}
	}
	elm_list_go(data->list);
}

static void _favourites_view_sel_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is null");
	RETM_IF(!obj, "obj is null");
	RETM_IF(!event_info, "event_info is null");
	contacts_record_h record = data;
	favourites_view_data *fav_data = evas_object_data_get(obj, DATA_LIST);
	RETM_IF(!fav_data, "favorite data is null");

	elm_list_item_selected_set(event_info, EINA_FALSE);

	int person_id = 0;
	sc_db_utils_is_success(contacts_record_get_int(record, _contacts_person.id, &person_id));

	sc_cdetails_view_add(fav_data->navi, person_id);
}

static void _favourites_view_update_list_cb(const char *view_uri, void *user_data)
{
	RETM_IF(!user_data, "user_data is null");
	favourites_view_data *data = user_data;
	data->fav_list = sc_list_utils_get_records_list_bool_filter(_contacts_person._uri, _contacts_person.is_favorite, true);
	if (data->fav_list) {
		elm_list_clear(data->list);
		_favourites_view_fill(data);
	}
}
