/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "view/group-details-view.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "model/contacts-utils/contacts-list-utils.h"
#include "utils/logger.h"
#include "utils/ui-utils.h"
#include "view/contact-details-view.h"
#include "view/group-edit-member-view.h"
#include "view/view-common.h"

#include <contacts.h>
#include <Elementary.h>

static const char *_caption_delete                  = "Delete";
static const char *_caption_add_member              = "Add Member";
static const char *_caption_remove_member           = "Remove Member";
static const char *_caption_caution                 = "Caution";
static const char *_caption_remove_category_request = "Remove this category?";
static const char *_group_emergency_name            = "Emergency";

static const char *_group_data = "group data";

typedef struct {
	Evas_Object *navi;
	Elm_Object_Item *navi_item;

	int group_id;

	Evas_Object *list;
	Elm_Object_Item  *remove_item;
} group_details_view_data;

static void _gdetails_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _gdetails_view_group_update_details_cb(const char *view_uri, void *user_data);
static void _gdetails_view_group_relation_update_details_cb(const char *view_uri, void *user_data);
static void _gdetails_view_fill(group_details_view_data *data);
static void _gdetails_view_sel_cb(void *data, Evas_Object *obj, void *item);
static Evas_Object *_gdetails_view_create_toolbar(group_details_view_data *data);
static void _gdetails_view_add_member_item_cb(void *data, Evas_Object *obj, void *event_info);
static void _gdetails_view_remove_member_item_cb(void *data, Evas_Object *obj, void *event_info);

static void _gdetails_view_add_delete_btn(group_details_view_data *data);
static void _gdetails_view_delete_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info);
static void _gdetails_view_delete_ok_clicked_cb(void *data, Evas_Object *obj, void *event_info);
static void _gdetails_view_delete_cancel_clicked_cb(void *data, Evas_Object *obj, void *event_info);


void sc_group_details_view_add(Evas_Object *parent, int group_id)
{
	RETM_IF(!parent, "parent is null");
	group_details_view_data *data = calloc(1, sizeof(group_details_view_data));
	RETM_IF(!data, "data not allocated");
	data->navi = parent;

	data->group_id = group_id;
	if (data->group_id <= 0) {
		_gdetails_view_destroy(data, NULL, NULL, NULL);
		return;
	}

	data->list = elm_list_add(data->navi);
	if (!data->list) {
		_gdetails_view_destroy(data, NULL, NULL, NULL);
		return;
	}
	evas_object_event_callback_add(data->list, EVAS_CALLBACK_FREE, _gdetails_view_destroy, data);
	sc_db_utils_is_success(contacts_db_add_changed_cb(_contacts_group._uri, _gdetails_view_group_update_details_cb, data));
	sc_db_utils_is_success(contacts_db_add_changed_cb(_contacts_group_relation._uri, _gdetails_view_group_relation_update_details_cb, data));
	sc_db_utils_is_success(contacts_db_add_changed_cb(_contacts_person._uri, _gdetails_view_group_relation_update_details_cb, data));
	_gdetails_view_fill(data);

	Evas_Object *toolbar = _gdetails_view_create_toolbar(data);

	contacts_record_h group_record = NULL;
	if (sc_db_utils_is_success(contacts_db_get_record(_contacts_group._uri, data->group_id, &group_record))) {
		char *name = NULL;
		sc_db_utils_is_success(contacts_record_get_str_p(group_record, _contacts_group.name, &name));
		data->navi_item = elm_naviframe_item_push(data->navi, name, NULL, NULL, data->list, NULL);
		contacts_record_destroy(group_record, false);
	} else {
		ERR("Error getting group record");
		evas_object_del(data->list);
	}

	if (toolbar) {
		elm_object_item_part_content_set(data->navi_item, PART_TOOLBAR, toolbar);
	}

	if (data->navi_item) {
		_gdetails_view_add_delete_btn(data);
	}
}

static void _gdetails_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is null");
	group_details_view_data *gd_data = data;
	contacts_db_remove_changed_cb(_contacts_group._uri, _gdetails_view_group_update_details_cb, gd_data);
	contacts_db_remove_changed_cb(_contacts_group_relation._uri, _gdetails_view_group_relation_update_details_cb, gd_data);
	contacts_db_remove_changed_cb(_contacts_person._uri, _gdetails_view_group_relation_update_details_cb, gd_data);
	free(gd_data);
}

static void _gdetails_view_group_update_details_cb(const char *view_uri, void *user_data)
{
	RETM_IF(!user_data, "user_data is null");
	group_details_view_data *data = user_data;

	contacts_record_h record = NULL;
	sc_db_utils_is_success(contacts_db_get_record(_contacts_group._uri, data->group_id, &record));
	if (!record) {
		sc_view_common_return_to_view(data->navi, VI_CATEGORIES_LIST);
		return;
	}
	char *name = NULL;
	sc_db_utils_is_success(contacts_record_get_str_p(record, _contacts_group.name, &name));
	elm_object_item_text_set(data->navi_item, name);
	contacts_record_destroy(record, true);
}

static void _gdetails_view_group_relation_update_details_cb(const char *view_uri, void *user_data)
{
	RETM_IF(!user_data, "user_data is null");
	group_details_view_data *data = user_data;
	elm_list_clear(data->list);
	_gdetails_view_fill(data);
	int person_list_count = 0;
	contacts_list_h person_list = sc_list_utils_get_records_list_int_filter(_contacts_group_relation._uri,
				_contacts_group_relation.group_id, data->group_id);
	contacts_list_get_count(person_list, &person_list_count);
	bool is_disabled = (!person_list_count);
	elm_object_item_disabled_set(data->remove_item, is_disabled);
	contacts_list_destroy(person_list, true);
}

static void _gdetails_view_fill(group_details_view_data *data)
{
	RETM_IF(NULL == data, "data is null");
	contacts_record_h record = NULL;
	contacts_list_h members = sc_list_utils_get_records_list_int_filter(_contacts_group_relation._uri,
			_contacts_group_relation.group_id, data->group_id);
	CONTACTS_LIST_FOREACH(members, record) {
		int contact_id = 0;
		sc_db_utils_is_success(contacts_record_get_int(record, _contacts_group_relation.contact_id, &contact_id));
		if (contact_id > 0) {
			sc_db_utils_is_success(contacts_db_get_record(_contacts_contact._uri, contact_id, &record));
			if (record) {
				char *name = NULL;
				sc_db_utils_is_success(contacts_record_get_str_p(record, _contacts_contact.display_name, &name));
				if (name) {
					int person_id = 0;
					sc_db_utils_is_success(contacts_record_get_int(record, _contacts_contact.person_id, &person_id));
					evas_object_data_set(data->list, _group_data, data);
					char *temp_str = elm_entry_utf8_to_markup(name);
					elm_list_item_append(data->list, temp_str, NULL, NULL, _gdetails_view_sel_cb, (void *)person_id);
					free(temp_str);
				}
				sc_db_utils_is_success(contacts_record_destroy(record, true));
			}
		}
	}
	sc_db_utils_is_success(contacts_list_destroy(members, true));
	elm_list_go(data->list);
}

static void _gdetails_view_sel_cb(void *data, Evas_Object *obj, void *item)
{
	RETM_IF(!item, "item is null");
	RETM_IF(!obj, "obj is null");
	RETM_IF(!data, "data is null");
	elm_list_item_selected_set(item, EINA_FALSE);
	group_details_view_data *gd_data = evas_object_data_get(obj, _group_data);
	RETM_IF(!gd_data, "group data is null");
	sc_cdetails_view_add(gd_data->navi, (int) data);
}

static Evas_Object *_gdetails_view_create_toolbar(group_details_view_data *data)
{
	RETVM_IF(NULL == data, NULL, "data is null");

	Evas_Object *toolbar = ui_utils_toolbar_add(data->list);
	RETVM_IF(NULL == toolbar, NULL, "toolbar is null");
	elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_NONE);
	elm_toolbar_item_append(toolbar, NULL, _caption_add_member, _gdetails_view_add_member_item_cb, data);
	data->remove_item = elm_toolbar_item_append(toolbar, NULL, _caption_remove_member, _gdetails_view_remove_member_item_cb, data);
	int person_list_count = 0;
	contacts_list_h person_list = sc_list_utils_get_records_list_int_filter(_contacts_group_relation._uri,
				_contacts_group_relation.group_id, data->group_id);
	contacts_list_get_count(person_list, &person_list_count);
	if (!person_list_count) {
		elm_object_item_disabled_set(data->remove_item, EINA_TRUE);
	}
	contacts_list_destroy(person_list, true);
	return toolbar;
}

static void _gdetails_view_add_member_item_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is null");
	group_details_view_data *gd_data = data;

	sc_group_edit_member_view_add(gd_data->navi, gd_data->group_id, EDIT_MEMBER_TYPE_ADD);
}

static void _gdetails_view_remove_member_item_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is null");
	group_details_view_data *gd_data = data;

	sc_group_edit_member_view_add(gd_data->navi, gd_data->group_id, EDIT_MEMBER_TYPE_REMOVE);
}

static void _gdetails_view_add_delete_btn(group_details_view_data *data)
{
	RETM_IF(!data, "data is null");
	RETM_IF(data->group_id <= 0, "group_id is invalid");

	Evas_Object *delete_btn = elm_button_add(data->navi);
	RETM_IF(NULL == delete_btn, "delete_btn is null");
	elm_object_text_set(delete_btn, _caption_delete);
	elm_object_style_set(delete_btn, STYLE_BUTTON_NAVIFRAME);
	elm_object_item_part_content_set(data->navi_item, PART_NAVI_TITLE_RIGHT_BUTTON, delete_btn);

	evas_object_smart_callback_add(delete_btn, EVENT_CLICKED, _gdetails_view_delete_btn_clicked_cb, data);

	contacts_record_h group_record = NULL;
	if (sc_db_utils_is_success(contacts_db_get_record(_contacts_group._uri, data->group_id, &group_record))) {
		char *name = NULL;
		sc_db_utils_is_success(contacts_record_get_str_p(group_record, _contacts_group.name, &name));
		bool is_disabled = (!strcmp(name, _group_emergency_name));
		elm_object_disabled_set(delete_btn, is_disabled);
		contacts_record_destroy(group_record, true);
	}
}

static void _gdetails_view_delete_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	group_details_view_data *view = data;

	Evas_Object *popup = elm_popup_add(view->navi);
	RETM_IF(!popup, "popup is not created");
	elm_object_part_text_set(popup, PART_POPUP_TITLE, _caption_caution);
	elm_object_text_set(popup, _caption_remove_category_request);
	Evas_Object *ok_btn = ui_utils_popup_button_add(popup, _caption_ok, _gdetails_view_delete_ok_clicked_cb, view);
	evas_object_data_set(ok_btn, "popup", popup);
	ui_utils_popup_button_add(popup, _caption_cancel, _gdetails_view_delete_cancel_clicked_cb, popup);
	eext_object_event_callback_add(popup, EEXT_CALLBACK_BACK, _gdetails_view_delete_cancel_clicked_cb, popup);
	evas_object_show(popup);
}

static void _gdetails_view_delete_ok_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	group_details_view_data *view = data;
	sc_db_utils_is_success(contacts_db_delete_record(_contacts_group._uri, view->group_id));
	Evas_Object *popup = evas_object_data_get(obj, "popup");
	if (popup) {
		evas_object_del(popup);
	}
	elm_naviframe_item_pop(view->navi);
}

static void _gdetails_view_delete_cancel_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is NULL");
	evas_object_del(data);
}
