/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "view/group-edit-member-view.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "model/contacts-utils/contacts-list-utils.h"
#include "model/group-edit-member.h"
#include "utils/logger.h"
#include "utils/ui-utils.h"
#include "view/view-common.h"

#include <Elementary.h>
#include <contacts.h>

#define PART_MAIN_TEXT               "elm.text.main.left"
#define PART_ICON                    "elm.icon.1"

static const char *_add_member              = "Add Member";
static const char *_remove_member           = "Remove Member";
static const char *_error                   = "Error";
static const char *_select_some_member      = "Select some member.";
static const char *_there_is_no_member      = "There is no member.";

typedef struct {
	Evas_Object *navi;
	Elm_Object_Item *navi_item;

	int group_id;
	Eina_List *check_list;
	enum sc_gedit_member_type type;

	Evas_Object *popup;
	Evas_Object *genlist;
} group_edit_member_view_data;

static const char *_gedit_member_view_label_get(enum sc_gedit_member_type type);
static void _gedit_member_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _gedit_member_view_fill(group_edit_member_view_data *data);
static void _gedit_member_view_sel_cb(void *data, Evas_Object *obj, void *item);
static Elm_Genlist_Item_Class *_gedit_member_view_create_item_class();
static char *_gedit_member_view_item_text_get_cb(void *data, Evas_Object *obj, const char *part);
static Evas_Object *_gedit_member_view_item_content_get_cb(void *data, Evas_Object *obj, const char *part);
static void _gedit_member_view_edit_btn(group_edit_member_view_data *data);
static void _gedit_member_view_edit_cb(void *data, Evas_Object *obj, void *event_info);
static void _gedit_view_popup(group_edit_member_view_data *data);
static void _gedit_view_del_object_cb(void *data, Evas_Object *obj, void *event_info);

void sc_group_edit_member_view_add(Evas_Object *parent, int group_id, enum sc_gedit_member_type type)
{
	RETM_IF(!parent, "parent is null");
	RETM_IF(group_id <= 0, "group_id is invalid");

	group_edit_member_view_data *data = calloc(1, sizeof(group_edit_member_view_data));
	RETM_IF(!data, "data not allocated");
	data->navi = parent;
	data->group_id = group_id;
	data->type = type;

	data->genlist = elm_genlist_add(data->navi);
	if (!data->genlist) {
		_gedit_member_view_destroy(data, NULL, NULL, NULL);
		return;
	}
	evas_object_event_callback_add(data->genlist, EVAS_CALLBACK_FREE, _gedit_member_view_destroy, data);

	_gedit_member_view_fill(data);
	data->navi_item = elm_naviframe_item_push(data->navi, _gedit_member_view_label_get(data->type), NULL, NULL, data->genlist, NULL);

	_gedit_member_view_edit_btn(data);
}

static const char *_gedit_member_view_label_get(enum sc_gedit_member_type type)
{
	if (type == EDIT_MEMBER_TYPE_ADD) {
		return _add_member;
	} else if (type == EDIT_MEMBER_TYPE_REMOVE) {
		return _remove_member;
	}
	return NULL;
}

static void _gedit_member_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is null");
	group_edit_member_view_data *gd_data = data;
	eina_list_free(gd_data->check_list);
	free(gd_data);
}

static void _gedit_member_view_fill(group_edit_member_view_data *data)
{
	Elm_Genlist_Item_Class *itc = _gedit_member_view_create_item_class();
	RETM_IF(!itc, "itc is null");

	contacts_list_h list = NULL;
	if (data->type == EDIT_MEMBER_TYPE_ADD) {
		list = sc_gedit_member_add_contact_list_get(data->group_id);
	} else if (data->type == EDIT_MEMBER_TYPE_REMOVE) {
		list = sc_gedit_member_remove_contact_list_get(data->group_id);
	}

	data->check_list = sc_gedit_member_check_data_list_get(data->group_id, list);
	Eina_List *l = NULL;
	void *check_data = NULL;
	EINA_LIST_FOREACH(data->check_list, l, check_data) {
		if (check_data) {
			elm_genlist_item_append(data->genlist, itc, check_data, NULL,
					ELM_GENLIST_ITEM_NONE, _gedit_member_view_sel_cb, check_data);
		}
	}
	elm_genlist_item_class_free(itc);
	contacts_list_destroy(list, true);
}

static void _gedit_member_view_sel_cb(void *data, Evas_Object *obj, void *item)
{
	RETM_IF(!data, "data is null");
	RETM_IF(!item, "item is null");
	elm_genlist_item_selected_set(item, EINA_FALSE);
	group_edit_member_check_data *check_data = data;
	Evas_Object *check = elm_object_item_part_content_get(item, PART_ICON);
	elm_check_state_set(check, !check_data->is_checked);
}

static Elm_Genlist_Item_Class *_gedit_member_view_create_item_class()
{
	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();
	RETVM_IF(!itc, NULL, "itc is null");
	itc->item_style = STYLE_ITEM_CHECKBOX;
	itc->func.text_get = _gedit_member_view_item_text_get_cb;
	itc->func.content_get = _gedit_member_view_item_content_get_cb;
	return itc;
}

static char *_gedit_member_view_item_text_get_cb(void *data, Evas_Object *obj, const char *part)
{
	RETVM_IF(!data, NULL, "data is null");
	RETVM_IF(strcmp(PART_MAIN_TEXT, part) != 0, NULL, "inappropriate part = %s", part);

	group_edit_member_check_data *check_data = data;
	if (check_data->name) {
		return elm_entry_utf8_to_markup(check_data->name);
	}
	return NULL;
}

static Evas_Object *_gedit_member_view_item_content_get_cb(void *data, Evas_Object *obj, const char *part)
{
	RETVM_IF(!data, NULL, "data is null");
	RETVM_IF(strcmp(PART_ICON, part) != 0, NULL, "inappropriate part = %s", part);

	group_edit_member_check_data *check_data = data;
	Evas_Object *check = elm_check_add(obj);
	RETVM_IF(!check, NULL, "check is null");
	elm_check_state_set(check, check_data->is_checked);
	evas_object_propagate_events_set(check, EINA_FALSE);
	elm_check_state_pointer_set(check, &check_data->is_checked);
	return check;
}

static void _gedit_member_view_edit_btn(group_edit_member_view_data *data)
{
	RETM_IF(!data, "data is null");
	RETM_IF(!data->navi, "navi is null");
	RETM_IF(!data->navi_item, "navi item is null");

	Evas_Object *toolbar = ui_utils_toolbar_add(data->navi);
	elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_NONE);
	elm_toolbar_item_append(toolbar, NULL, _gedit_member_view_label_get(data->type), _gedit_member_view_edit_cb, data);
	elm_object_item_part_content_set(data->navi_item, PART_TOOLBAR, toolbar);
}

static void _gedit_member_view_edit_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is null");
	group_edit_member_view_data *member_data = data;
	bool is_checked = false;

	if (member_data->check_list) {
		Eina_List *l = NULL;
		void *ch_data = NULL;
		EINA_LIST_FOREACH(member_data->check_list, l, ch_data) {
			group_edit_member_check_data *check_data = ch_data;
			if (check_data) {
				if (check_data->is_checked) {
					is_checked = true;
					INF("check_data->contact_id = %d", check_data->contact_id);
					if (member_data->type == EDIT_MEMBER_TYPE_ADD) {
						contacts_group_add_contact(member_data->group_id, check_data->contact_id);
					} else if (member_data->type == EDIT_MEMBER_TYPE_REMOVE) {
						contacts_group_remove_contact(member_data->group_id, check_data->contact_id);
					}
				}
			}
		}
		if (!is_checked) {
			if (!member_data->popup) {
				_gedit_view_popup(member_data);
			}
		} else {
			elm_naviframe_item_pop(member_data->navi);
		}
	} else if (!member_data->popup) {
		_gedit_view_popup(member_data);
	}
}

static void _gedit_view_popup(group_edit_member_view_data *data)
{
	RETM_IF(!data, "data is null");
	data->popup = elm_popup_add(data->navi);
	RETM_IF(!data->popup, "popup is not created");

	const char *content_text = NULL;
	if (data->check_list) {
		content_text = _select_some_member;
	} else {
		content_text = _there_is_no_member;
	}
	elm_object_part_text_set(data->popup, PART_POPUP_TITLE, _error);
	elm_object_text_set(data->popup, content_text);
	ui_utils_popup_button_add(data->popup, _caption_ok, _gedit_view_del_object_cb, data);
	evas_object_show(data->popup);
	eext_object_event_callback_add(data->popup, EEXT_CALLBACK_BACK, _gedit_view_del_object_cb, data);
}

static void _gedit_view_del_object_cb(void *data, Evas_Object *obj, void *event_info)
{
	RETM_IF(!data, "data is null");
	group_edit_member_view_data *member_data = data;
	evas_object_del(member_data->popup);
	member_data->popup = NULL;
}
