/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "ecore_idler"
#include "main.h"

struct context {  // helper struct to give some context to the callbacks
    int count;
    Ecore_Idle_Enterer *enterer;
    Ecore_Idler *idler;
    Ecore_Idle_Exiter *exiter;
    Ecore_Event_Handler *handler;
    Ecore_Timer *timer;
};

static int _event_type = 0;  // a new type of event will be defined and stored here
static struct context ctx = {0};

static Eina_Bool
_enterer_cb(void *data EINA_UNUSED) // the idle enterer callback
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "IDLE ENTERER: Ecore entering in idle state.");

    return ECORE_CALLBACK_RENEW;  // same as EINA_TRUE
}

static Eina_Bool
_exiter_cb(void *data EINA_UNUSED)  //the idle exiter callback
{
    dlog_print(DLOG_DEBUG, LOG_TAG, "IDLE EXITER: Ecore exiting idle state.");

    return ECORE_CALLBACK_RENEW;  // same as EINA_TURE
}

static Eina_Bool
_idler_cb(void *data)  // the idler callback - ran while the mainloop is idle
{
    struct context *ctx = data;
    dlog_print(DLOG_DEBUG, LOG_TAG, "IDLER: executing idler callback while in idle state. (count : %d)", ctx->count);

    ctx->count++;

    /* each 10 times that the callback gets called, generate an event that
     * will wake up the main loop, triggering idle enterers, exiters, etc. */
    if ((ctx->count % 10) == 0) ecore_event_add(_event_type, NULL, NULL, NULL);

    return ECORE_CALLBACK_RENEW; // same as EINA_TURE
}

static Eina_Bool
_event_handler_cb(void *data, int type EINA_UNUSED, void *event EINA_UNUSED)  // event callback
{
    struct context *ctx = data;

    dlog_print(DLOG_DEBUG, LOG_TAG, "EVENT: processng callback for the event received.");

    if (ctx->count > 20) {
        ecore_idle_enterer_del(ctx->enterer);
        ecore_idle_exiter_del(ctx->exiter);
        ecore_idler_del(ctx->idler);

        ctx->enterer = NULL;
        ctx->exiter = NULL;
        ctx->idler = NULL;

        if (ctx->timer) {
            ecore_timer_del(ctx->timer);
            ctx->timer = NULL;
        }
        ecore_main_loop_quit();
    }
    return ECORE_CALLBACK_DONE;  // same as EINA_FALSE
}

static Eina_Bool
_timer_cb(void *data)
{
    struct context *ctx = data;
    dlog_print(DLOG_DEBUG, LOG_TAG, "TIMER: timer callback called.");

    if (ctx->timer)
        ctx->timer = NULL;

    return ECORE_CALLBACK_CANCEL; // same as EINA_FALSE
}

static Eina_Bool
naviframe_pop_cb(void *data, Elm_Object_Item *it)
{
	struct context *ctx = data;

	_event_type = 0;
	ctx->count = 0;

	ecore_idle_enterer_del(ctx->enterer);
	ecore_idler_del(ctx->idler);
	ecore_idle_exiter_del(ctx->exiter);
	ecore_event_handler_del(ctx->handler);
	ecore_timer_del(ctx->timer);

	return EINA_TRUE;
}

void
ecore_idler_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;

	Elm_Object_Item *nf_it;

	/* Base Layout */
	ad->layout = elm_layout_add(ad->win);
	elm_layout_theme_set(ad->layout, "layout", "application", "default");
	evas_object_size_hint_weight_set(ad->layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_show(ad->layout);

	/* Label*/
	Evas_Object *label = elm_label_add(ad->layout);
	elm_object_text_set(label, "Ecore Idler");
	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(ad->layout, "elm.swallow.content", label);
	evas_object_show(label);

	/* Idler Sample */
	_event_type = ecore_event_type_new();

	ecore_event_add(_event_type, NULL, NULL, NULL);

	ctx.enterer = ecore_idle_enterer_add(_enterer_cb, &ctx);
	ctx.exiter = ecore_idle_exiter_add(_exiter_cb, &ctx);
	ctx.idler = ecore_idler_add(_idler_cb, &ctx);
	ctx.handler = ecore_event_handler_add(_event_type, _event_handler_cb, &ctx);
	ctx.timer = ecore_timer_add(0.0005, _timer_cb, &ctx);

	ecore_main_loop_begin();

	/* Push base layout into naviframe */
	nf_it = elm_naviframe_item_push(ad->nf, NULL, NULL, NULL, ad->layout, "empty");
	elm_naviframe_item_pop_cb_set(nf_it, naviframe_pop_cb, &ctx);
}
