/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "ecore_thread_4"
#include "main.h"

typedef struct user_data
{
	Evas_Object *btn;
	Evas_Coord x, y;
} user_data;

static Ecore_Thread *ethread;
static user_data *ud;

static void
thread_run_cb(void *data, Ecore_Thread *thread)
{
	double t = 0.0;
	Evas_Coord x, y;
	Evas_Object *btn = data;

	ud = calloc(1, sizeof(user_data));
	ud->btn = btn;

	while (1)
	{
		x = 150 + (150 * sin(t));
		y = 150 + (150 * cos(t));

		ud->x = x;
		ud->y = y;

		//Now we have recorded the time point we pass it as feedback back to the
		//mainloop. It will free data when done.
		ecore_thread_feedback(thread, ud);

		usleep(1000);
		t += 0.001;

		//In case someone has asked us to cancel the thread, then cancel this
		//loop co-operatively (cancelling is co-operative)
		if (ecore_thread_check(thread)) break;
	}
}

static void
thread_feedback_cb(void *data, Ecore_Thread *thread, void *msg)
{
	//This function is in critical section.
	user_data *ud = msg;
	evas_object_move(ud->btn, ud->x, ud->y);
}

static void
thread_end_cb(void *data, Ecore_Thread *thread)
{
	dlog_print(DLOG_DEBUG, LOG_TAG, "thread end!");
}

static void
thread_cancel_cb(void *data, Ecore_Thread *thread)
{
	dlog_print(DLOG_DEBUG, LOG_TAG, "thread cancel!");
}

static Eina_Bool
naviframe_pop_cb(void *data, Elm_Object_Item *it)
{
	Eina_Bool thread_cancel;
	int thread_active = ecore_thread_active_get();

	thread_cancel = ecore_thread_cancel(ethread);

	if (!thread_cancel)
		while (thread_active)
		{
			thread_active = ecore_thread_active_get();
		}

	free(ud);

	return EINA_TRUE;
}

void
ecore_thread_4_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;

	Evas_Object *btn;
	Elm_Object_Item *nf_it;

	/* Base Layout */
	ad->layout = elm_layout_add(ad->win);
	elm_layout_theme_set(ad->layout, "layout", "application", "default");
	evas_object_size_hint_weight_set(ad->layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_show(ad->layout);

	//Create a Button
	btn = elm_button_add(ad->layout);
	elm_object_text_set(btn, "Ecore Thread Main Loop");
	evas_object_resize(btn, ELM_SCALE_SIZE(400), ELM_SCALE_SIZE(100));
	evas_object_show(btn);

	//Create a thread
	ethread = ecore_thread_feedback_run(thread_run_cb, thread_feedback_cb,
			thread_end_cb, thread_cancel_cb, btn,
			EINA_FALSE);

	/* Push base layout into naviframe */
	nf_it = elm_naviframe_item_push(ad->nf, NULL, NULL, NULL, ad->layout, "empty");
	elm_naviframe_item_pop_cb_set(nf_it, naviframe_pop_cb, ad);
}
