/*
 * Samsung API
 * Copyright (c) 2009-2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */



#include <Ecore.h>
#include <tizen.h>
#include <time.h>

#include "log.h"
#include "eom_sample.h"
#include "main.h"

extern tizen_error_e _eom_mirror_cb(void *data)
{
	appdata_s *ad = NULL;
	viewdata_s *vd = NULL;

	retv_if(!data, EOM_ERROR_INVALID_PARAMETER);

	_D("Pressed mirror button");

	ad = data;
	vd = ad->vd;

	if (ad->support_external_display == EINA_FALSE)
		return EOM_ERROR_NONE;

	if (ad->mode == EOM_OUTPUT_MODE_PRESENTATION)
	{
		elm_object_text_set(vd->text3, "<font_size=40><color=#ffffff>Mode : mirror</color></font_size>");
		ad->mode = EOM_OUTPUT_MODE_MIRROR;
	}

	if (ad->connected == EINA_FALSE)
	{
		elm_object_text_set(vd->text2, "<font_size=40><color=#ffffff>Connect HDMI cable.</color></font_size>");
		return EOM_ERROR_NONE;
	}

	if (ad->external_win)
	{
		vd->box_external = NULL;
		evas_object_del (ad->external_win);
		ad->external_win = NULL;
	}

	return EOM_ERROR_NONE;
}

void _set_external_window (void *data)
{
	Evas_Object *box, *bg;
	appdata_s *ad = NULL;
	viewdata_s *vd = NULL;

	ret_if(!data);

	ad = (appdata_s*)data;
	vd = ad->vd;

//	elm_win_aux_hint_add(ad->external_win, "wm.policy.win.user.geometry", "1");

	box = elm_box_add (ad->external_win);
	evas_object_size_hint_weight_set(box, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(ad->external_win, box);
	evas_object_show (box);

	bg = elm_bg_add(ad->external_win);
	evas_object_size_hint_weight_set(bg, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(bg, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_bg_color_set(bg, 0, 0, 0);
	evas_object_show(bg);

	evas_object_show (ad->external_win);

	vd->box_external = box;
}

static tizen_error_e _eom_presentation_cb(void *data)
{
	appdata_s *ad = NULL;
	viewdata_s *vd = NULL;

	retv_if(!data, EOM_ERROR_INVALID_PARAMETER);

	_D("Pressed presentation button");

	ad = data;
	vd = ad->vd;

	if (ad->support_external_display == EINA_FALSE)
		return EOM_ERROR_NONE;

	if (ad->mode == EOM_OUTPUT_MODE_MIRROR)
	{
		elm_object_text_set(vd->text3, "<font_size=40><color=#ffffff>Mode : presentation</color></font_size>");
		ad->mode = EOM_OUTPUT_MODE_PRESENTATION;
	}

	if (ad->connected == EINA_FALSE)
	{
		elm_object_text_set(vd->text2, "<font_size=40><color=#ffffff>Connect HDMI cable.</color></font_size>");
		return EOM_ERROR_NONE;
	}

	if (!ad->external_win)
	{
		ad->external_win = elm_win_add(NULL, "external", ELM_WIN_BASIC);
		retv_if(!ad->external_win, EOM_ERROR_OUT_OF_MEMORY);
		if (eom_set_output_window(ad->output_id, ad->external_win) != EOM_ERROR_NONE)
		{
			vd->box_external = NULL;
			evas_object_del (ad->external_win);
			ad->external_win = NULL;
			_E("create external window fail\n");
		}
		else
		{
			_set_external_window(data);
			_D("create external window\n");
		}
	}

	return EOM_ERROR_NONE;
}

extern void eom_button_clicked(void *data, Evas_Object *obj, void *event_info)
{
	const char *str = NULL;
	appdata_s *ad = NULL;

	ret_if(!data);
	ret_if(!obj);

	ad = data;
	str = elm_object_text_get(obj);

	tizen_error_e result = TIZEN_ERROR_UNKNOWN;

	if (str && !strcmp(str, "MIRROR")) {
		result = _eom_mirror_cb(ad);
		retm_if(EOM_ERROR_NONE != result, "Failed to eom mirror: %d", result);
	} else {
		result = _eom_presentation_cb(ad);
		retm_if(EOM_ERROR_NONE != result, "Failed to eom presentation: %d", result);
	}
}

static void sample_notify_cb_output_add (eom_output_id output_id, void *user_data)
{
	appdata_s *ad = NULL;
	viewdata_s *vd = NULL;

	ret_if(!user_data);

	ad = (appdata_s*)user_data;
	vd = ad->vd;

	if (ad->output_id != output_id)
	{
		_D("OUTPUT ADDED. SKIP. my output ID is %d\n", ad->output_id);
		return;
	}
	_D("output(%d) connected\n", output_id);
	elm_object_text_set(vd->text2, "<font_size=40><color=#ffffff>Connected.</color></font_size>");

	ad->connected = EINA_TRUE;

	if (ad->mode == EOM_OUTPUT_MODE_PRESENTATION)
	{
		if (!ad->external_win)
		{
			ad->external_win = elm_win_add(NULL, "external", ELM_WIN_BASIC);
			ret_if(!ad->external_win);
			if (eom_set_output_window(ad->output_id, ad->external_win) != EOM_ERROR_NONE)
			{
				vd->box_external = NULL;
				evas_object_del (ad->external_win);
				ad->external_win = NULL;
				_E("create external window fail\n");
			}
			else
			{
				_set_external_window(user_data);
				_D("create external window\n");
			}
		}
	}    
}


static void sample_notify_cb_output_remove (eom_output_id output_id, void *user_data)
{
	appdata_s *ad = NULL;
	viewdata_s *vd = NULL;

	ret_if(!user_data);

	ad = (appdata_s*)user_data;
	vd = ad->vd;

	if (ad->output_id != output_id)
	{
		_D("OUTPUT REMOVED. SKIP. my output ID is %d\n", ad->output_id);
		return;
	}
	_D("output(%d) disconnected\n", output_id);
	elm_object_text_set(vd->text2, "<font_size=40><color=#ffffff>Disconnected.</color></font_size>");

	ad->connected = EINA_FALSE;

	if (ad->external_win)
	{
		vd->box_external = NULL;
		evas_object_del (ad->external_win);
		ad->external_win = NULL;
		_D("destory external window\n");
	}
}

static void sample_notify_cb_mode_changed (eom_output_id output_id, void *user_data)
{
	appdata_s *ad = NULL;
	eom_output_mode_e mode = EOM_OUTPUT_MODE_NONE;

	ret_if(!user_data);

	ad = (appdata_s*)user_data;

	if (ad->output_id != output_id)
	{
		_D("MODE CHANGED. SKIP. my output ID is %d\n", ad->output_id);
		return;
	}

	eom_get_output_mode(output_id, &mode);
	_D("output(%d) mode changed(%d)\n", output_id, mode);
}

void _show_image (void *user_data)
{
	Evas_Object *box, *image;
	appdata_s *ad = NULL;
	viewdata_s *vd = NULL;

	ret_if(!user_data);

	ad = (appdata_s*)user_data;
	vd = ad->vd;

	if (!vd->box_external)
	{
		_E("no vd->box_external.\n");
		return;
	}

	box = vd->box_external;

	image = elm_image_add(ad->external_win);
	elm_image_file_set(image, "/opt/usr/apps/org.tizen.eom/res/images/eom.jpg", NULL);
	evas_object_size_hint_weight_set(image, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(image, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_box_pack_end(box, image);
	evas_object_show(image);

	evas_object_data_set(ad->external_win, "image_external", image);

	evas_object_show(ad->external_win);
}

static void sample_notify_cb_attribute_changed (eom_output_id output_id, void *user_data)
{
	appdata_s *ad = NULL;
	viewdata_s *vd = NULL;
	eom_output_attribute_e attribute = EOM_OUTPUT_ATTRIBUTE_NONE;
	eom_output_attribute_state_e state = EOM_OUTPUT_ATTRIBUTE_STATE_NONE;

	ret_if(!user_data);

	ad = (appdata_s*)user_data;
	vd = ad->vd;

	if (ad->output_id != output_id)
	{
		_D("ATTR CHANGED. SKIP. my output ID is %d\n", ad->output_id);
		return;
	}

	eom_get_output_attribute(output_id, &attribute);
	eom_get_output_attribute_state(output_id, &state);

	_D("output(%d) attribute changed(%d, %d)\n", output_id, attribute, state);
	if (state == EOM_OUTPUT_ATTRIBUTE_STATE_ACTIVE)
	{
		_D("active : show image\n");
		_show_image(user_data);
	}
	else if (state == EOM_OUTPUT_ATTRIBUTE_STATE_INACTIVE)
	{
		_D("inactive : stop putimage & destroy external window\n");

		if (ad->external_win)
		{
			vd->box_external = NULL;
			evas_object_del(ad->external_win);
			ad->external_win = NULL;
		}
	}
    else if (state == EOM_OUTPUT_ATTRIBUTE_STATE_LOST)
    {
        _D("lost : stop putimage & destroy external window & remove callbacks\n");
		if (ad->external_win)
		{
			vd->box_external = NULL;
			evas_object_del(ad->external_win);
			ad->external_win = NULL;
		}

		eom_set_unset_cb(ad, EINA_FALSE);
    }
}

extern void eom_set_unset_cb(void *data, Eina_Bool set)
{
	appdata_s *ad = NULL;

	ret_if(!data);

	ad = data;

	if (set == EINA_TRUE)
	{
		eom_set_output_added_cb(sample_notify_cb_output_add, ad);
        eom_set_output_removed_cb(sample_notify_cb_output_remove, ad);
        eom_set_mode_changed_cb(sample_notify_cb_mode_changed, ad);
        eom_set_attribute_changed_cb(sample_notify_cb_attribute_changed, ad);
	}
	else
	{
		eom_unset_output_added_cb(sample_notify_cb_output_add);
        eom_unset_output_removed_cb(sample_notify_cb_output_remove);
        eom_unset_mode_changed_cb(sample_notify_cb_mode_changed);
        eom_unset_attribute_changed_cb(sample_notify_cb_attribute_changed);
	}

}

/* End of file */
