#include "$(appName).h"
#include "controller.h"
#include "view/view.h"

bool __controller_register_custom_event(const char *event_name, custom_ev_info_s **ev_info);

static void __controller_event_do_publish_cb(const char *event_name)
{
	controller_log(DLOG_INFO, "Event publishing: '%s'.", event_name);

	model_publish_event(event_name);
}

static bool __controller_event_get_system_info_cb(int index, void **ev_info)
{
	*ev_info = NULL;

	if (index >= model_get_system_events_count())
		return false;

	system_ev_info_s *ev_info_tmp = NULL;
	if (!model_get_system_event_info(index, &ev_info_tmp))
		return false;

	*ev_info = (system_ev_info_s *)ev_info_tmp;

	return true;
}

static bool __controller_event_set_custom_info_cb(const char *event_name, void **ev_info)
{
	*ev_info = NULL;

	custom_ev_info_s *ev_info_tmp = NULL;
	if (!__controller_register_custom_event(event_name, &ev_info_tmp))
		return false;

	*ev_info = (custom_ev_info_s *)ev_info_tmp;

	return true;
}

static void __system_event_cb(const char *event_name, bundle *event_data, void *user_data)
{
	system_ev_info_s *ev_info = (system_ev_info_s *)user_data;
	char *status_1 = NULL;
	char *status_2 = NULL;
	char *status_3 = NULL;

	controller_log(DLOG_INFO, "System event '%s' occured.", event_name);

	switch (ev_info->type) {
	case ET_BATTERY_CHARGER_STATUS:
		if (!model_get_bundle_str(event_data, EVENT_KEY_BATTERY_CHARGER_STATUS, &status_1))
			return;
		break;
	case ET_BATTERY_LEVEL_STATUS:
		if (!model_get_bundle_str(event_data, EVENT_KEY_BATTERY_LEVEL_STATUS, &status_1))
			return;
		break;
	case ET_USB_STATUS:
		if (!model_get_bundle_str(event_data, EVENT_KEY_USB_STATUS, &status_1))
			return;
		break;
	case ET_EARJACK_STATUS:
		if (!model_get_bundle_str(event_data, EVENT_KEY_EARJACK_STATUS, &status_1))
			return;
		break;
	case ET_DISPLAY_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_DISPLAY_STATE, &status_1))
			return;
		break;
	case ET_BOOT_COMPLETED:
		status_1 = "boot completed";
		break;
	case ET_SYSTEM_SHUTDOWN:
		status_1 = "system shutdown";
		break;
	case ET_LOW_MEMORY:
		if (!model_get_bundle_str(event_data, EVENT_KEY_LOW_MEMORY, &status_1))
			return;
		break;
	case ET_WIFI_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_WIFI_STATE, &status_1))
			return;
		break;
	case ET_BT_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_BT_STATE, &status_1) ||
			!model_get_bundle_str(event_data, EVENT_KEY_BT_LE_STATE, &status_2) ||
			!model_get_bundle_str(event_data, EVENT_KEY_BT_TRANSFERING_STATE, &status_3))
			return;
		break;
	case ET_LOCATION_ENABLE_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_LOCATION_ENABLE_STATE, &status_1))
			return;
		break;
	case ET_GPS_ENABLE_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_GPS_ENABLE_STATE, &status_1))
			return;
		break;
	case ET_NPS_ENABLE_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_NPS_ENABLE_STATE, &status_1))
			return;
		break;
	case ET_INCOMMING_MSG:
		if (!model_get_bundle_str(event_data, EVENT_KEY_MSG_TYPE, &status_1) ||
			!model_get_bundle_str(event_data, EVENT_KEY_MSG_ID, &status_2))
			return;
		break;
	case ET_TIME_CHANGED:
		status_1 = "time changed";
		break;
	case ET_TIME_ZONE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_TIME_ZONE, &status_1))
			return;
		break;
	case ET_HOUR_FORMAT:
		if (!model_get_bundle_str(event_data, EVENT_KEY_HOUR_FORMAT, &status_1))
			return;
		break;
	case ET_LANGUAGE_SET:
		if (!model_get_bundle_str(event_data, EVENT_KEY_LANGUAGE_SET, &status_1))
			return;
		break;
	case ET_REGION_FORMAT:
		if (!model_get_bundle_str(event_data, EVENT_KEY_REGION_FORMAT, &status_1))
			return;
		break;
	case ET_SILENT_MODE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_SILENT_MODE, &status_1))
			return;
		break;
	case ET_VIBRATION_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_VIBRATION_STATE, &status_1))
			return;
		break;
	case ET_SCREEN_AUTOROTATE_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_SCREEN_AUTOROTATE_STATE, &status_1))
			return;
		break;
	case ET_MOBILE_DATA_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_MOBILE_DATA_STATE, &status_1))
			return;
		break;
	case ET_DATA_ROAMING_STATE:
		if (!model_get_bundle_str(event_data, EVENT_KEY_DATA_ROAMING_STATE, &status_1))
			return;
		break;
	case ET_FONT_SET:
		if (!model_get_bundle_str(event_data, EVENT_KEY_FONT_SET, &status_1))
			return;
		break;
	default:
		return;
	}

	model_assign_event_status(&ev_info->status_1, status_1);
	model_assign_event_status(&ev_info->status_2, status_2);
	model_assign_event_status(&ev_info->status_3, status_3);

	view_update_system_events();
}

static void __custom_event_cb(const char *event_name, bundle *event_data, void *user_data)
{
	custom_ev_info_s *ev_info = (custom_ev_info_s *)user_data;
	char *status = NULL;

	if (!model_get_bundle_str(event_data, CUSTOM_EVENT_KEY_STATUS, &status))
		return;

	model_assign_event_status(&ev_info->status_1, status);

	view_display_custom_event(ev_info->name, ev_info->status_1);
	view_update_custom_events();
}

static void __add_system_event_handlers(void)
{
	int i;
	for (i = 0; i < model_get_system_events_count(); i++) {
		system_ev_info_s *ev_info = NULL;

		if (!model_get_system_event_info(i, &ev_info))
			continue;

		model_add_system_event_handler(ev_info, __system_event_cb, (void *)ev_info);
	}
}

bool __controller_register_custom_event(const char *event_name, custom_ev_info_s **ev_info)
{
	*ev_info = NULL;
	bool name_exists = false;

	if (!model_check_event_exists(event_name, &name_exists))
		return false;

	if (name_exists) {
		controller_log(DLOG_WARN, "Custom event '%s' already registered.", event_name);
		return false;
	}

	if (!model_create_custom_event_info(event_name, ev_info))
		return false;

	if (!model_add_custom_event_handler(*ev_info, __custom_event_cb, (void *)(*ev_info)))
		return false;

	controller_log(DLOG_INFO, "Custom event registered: '%s'.", (*ev_info)->name);

	return true;
}

bool controller_init(viewdata_s *vd)
{
	vd->callbacks.do_publish_cb = __controller_event_do_publish_cb;
	vd->callbacks.get_system_info_cb = __controller_event_get_system_info_cb;
	vd->callbacks.set_custom_info_cb = __controller_event_set_custom_info_cb;

	if (!view_create_base_gui(vd))
		return false;

	__add_system_event_handlers();

	return true;
}

void controller_finit(viewdata_s *vd)
{
	view_destroy_base_gui(vd);
	model_finit();
}

void controller_log(log_priority pri, const char *format, ...)
{
	va_list vl;

	va_start(vl, format);
	dlog_vprint(pri, LOG_TAG, format, vl);
	va_end(vl);
}
