#include <app_manager.h>
#include "$(appName).h"
#include "model.h"
#include "controller.h"

#define __SYSTEM_EVENT_COUNT_MAX 25
#define __CUSTOM_EVENT_COUNT_MAX 25
#define __STATUS_MSG_BUFF 256

static int __custom_ev_count = 0;
static custom_ev_info_s __custom_ev[__CUSTOM_EVENT_COUNT_MAX] = {{0,},};
static system_ev_info_s __system_ev[__SYSTEM_EVENT_COUNT_MAX] = {
	{ .type = ET_BATTERY_CHARGER_STATUS, .name = SYSTEM_EVENT_BATTERY_CHARGER_STATUS, .desc = "Battery charger status", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_BATTERY_LEVEL_STATUS, .name = SYSTEM_EVENT_BATTERY_LEVEL_STATUS, .desc = "Battery level status", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_USB_STATUS, .name = SYSTEM_EVENT_USB_STATUS, .desc = "USB status", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_EARJACK_STATUS, .name = SYSTEM_EVENT_EARJACK_STATUS, .desc = "Earjack status", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_DISPLAY_STATE, .name = SYSTEM_EVENT_DISPLAY_STATE, .desc = "Display state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_BOOT_COMPLETED, .name = SYSTEM_EVENT_BOOT_COMPLETED, .desc = "Boot completed", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_SYSTEM_SHUTDOWN, .name = SYSTEM_EVENT_SYSTEM_SHUTDOWN, .desc = "System shutdown", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_LOW_MEMORY, .name = SYSTEM_EVENT_LOW_MEMORY, .desc = "Low memory", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_WIFI_STATE, .name = SYSTEM_EVENT_WIFI_STATE, .desc = "Wi-Fi state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_BT_STATE, .name = SYSTEM_EVENT_BT_STATE, .desc = "Bluetooth state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_LOCATION_ENABLE_STATE, .name = SYSTEM_EVENT_LOCATION_ENABLE_STATE, .desc = "Location state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_GPS_ENABLE_STATE, .name = SYSTEM_EVENT_GPS_ENABLE_STATE, .desc = "GPS state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_NPS_ENABLE_STATE, .name = SYSTEM_EVENT_NPS_ENABLE_STATE, .desc = "NPS state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_INCOMMING_MSG, .name = SYSTEM_EVENT_INCOMMING_MSG, .desc = "Incoming message", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_TIME_CHANGED, .name = SYSTEM_EVENT_TIME_CHANGED, .desc = "Time changed", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_TIME_ZONE, .name = SYSTEM_EVENT_TIME_ZONE, .desc = "Time zone", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_HOUR_FORMAT, .name = SYSTEM_EVENT_HOUR_FORMAT, .desc = "Hour format", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_LANGUAGE_SET, .name = SYSTEM_EVENT_LANGUAGE_SET, .desc = "Language set", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_REGION_FORMAT, .name = SYSTEM_EVENT_REGION_FORMAT, .desc = "Region format", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_SILENT_MODE, .name = SYSTEM_EVENT_SILENT_MODE, .desc = "Silent mode", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_VIBRATION_STATE, .name = SYSTEM_EVENT_VIBRATION_STATE, .desc = "Vibration state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_SCREEN_AUTOROTATE_STATE, .name = SYSTEM_EVENT_SCREEN_AUTOROTATE_STATE, .desc = "Screen auto-rotate state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_MOBILE_DATA_STATE, .name = SYSTEM_EVENT_MOBILE_DATA_STATE, .desc = "Mobile data state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_DATA_ROAMING_STATE, .name = SYSTEM_EVENT_DATA_ROAMING_STATE, .desc = "Data roaming state", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
	{ .type = ET_FONT_SET, .name = SYSTEM_EVENT_FONT_SET, .desc = "Font set", .status_1 = NULL,.status_2 = NULL,.status_3 = NULL, },
};

static void __model_format_custom_event_name(const char *custom_name, char **ev_name)
{
	int ev_name_len = strlen("event") + strlen(APP_ID) + strlen(custom_name) + 3;
	*ev_name = (char *)calloc(ev_name_len, sizeof(char));
	snprintf(*ev_name, ev_name_len, "event.%s.%s", APP_ID, custom_name);

	int i;
	for (i = 0; i < ev_name_len; i++) {
		(*ev_name)[i] = tolower((*ev_name)[i]);

		if ((*ev_name)[i] == 32)
			(*ev_name)[i] = '_';
	}
}

static void __model_free_system_event_info(system_ev_info_s *ev_info)
{
	if (!ev_info) {
		controller_log(DLOG_ERROR, "Wrong argument provided.");
		return;
	}

	free(ev_info->status_1);
	free(ev_info->status_2);
	free(ev_info->status_3);
}

static void __model_free_custom_event_info(custom_ev_info_s *ev_info)
{
	if (!ev_info) {
		controller_log(DLOG_ERROR, "Wrong argument provided.");
		return;
	}

	free(ev_info->name);
	free(ev_info->desc);
	free(ev_info->status_1);
	free(ev_info->status_2);
	free(ev_info->status_3);
}

bool __model_remove_event_handler(event_handler_h event_handler)
{
	int ret = event_remove_event_handler(event_handler);
	if (ret != EVENT_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function event_remove_event_handler() failed with error = %d.", ret);
		return false;
	}

	return true;
}

static void __model_release_system_events(void)
{
	int i;
	for (i = 0; i < __SYSTEM_EVENT_COUNT_MAX; i++) {
		system_ev_info_s *ev_info = &__system_ev[i];

		__model_remove_event_handler(ev_info->event_h);
		__model_free_system_event_info(ev_info);
	}
}

static void __model_release_custom_events(void)
{
	int i;
	for (i = 0; i < __custom_ev_count; i++) {
		custom_ev_info_s *ev_info = &__custom_ev[i];

		__model_remove_event_handler(ev_info->event_h);
		__model_free_custom_event_info(ev_info);
	}

	__custom_ev_count = 0;
}

void model_finit(void)
{
	__model_release_system_events();
	__model_release_custom_events();
}

int model_get_system_events_count(void)
{
	return __SYSTEM_EVENT_COUNT_MAX;
}

bool model_get_system_event_info(int index, system_ev_info_s **ev_info)
{
	*ev_info = NULL;

	if (index < 0 || index >= __SYSTEM_EVENT_COUNT_MAX) {
		controller_log(DLOG_ERROR, "Wrong argument provided.");
		return false;
	}

	*ev_info = &__system_ev[index];

	return (*ev_info != NULL);
}

bool model_create_custom_event_info(const char *event_name, custom_ev_info_s **ev_info)
{
	*ev_info = NULL;

	if (!event_name) {
		controller_log(DLOG_ERROR, "Wrong argument provided.");
		return false;
	}

	if (__custom_ev_count == __CUSTOM_EVENT_COUNT_MAX) {
		controller_log(DLOG_ERROR, "Maximum number of custom events is already registered.");
		return false;
	}

	__custom_ev_count++;

	*ev_info = &__custom_ev[__custom_ev_count - 1];

	memset(*ev_info, 0, sizeof(custom_ev_info_s));

	__model_format_custom_event_name(event_name, &(*ev_info)->name);
	(*ev_info)->desc = strndup(event_name, strlen(event_name));

	return true;
}

bool model_publish_event(const char *event_name)
{
	static unsigned int custom_ev_counter = 0;

	if (!event_name) {
		controller_log(DLOG_ERROR, "Wrong argument provided.");
		return false;
	}

	bundle *bundle_ev = bundle_create();
	if (!bundle_ev) {
		controller_log(DLOG_ERROR, "Function bundle_create() failed.");
		return false;
	}

	custom_ev_counter++;
	char status[__STATUS_MSG_BUFF] = {0,};
	snprintf(status, __STATUS_MSG_BUFF, "status value %u", custom_ev_counter);

	int ret = bundle_add_str(bundle_ev, CUSTOM_EVENT_KEY_STATUS, status);
	if (ret != EVENT_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function bundle_add_str() failed with error %d.", ret);
		return false;
	}

	ret = event_publish_app_event(event_name, bundle_ev);
	bundle_free(bundle_ev);

	if (ret != EVENT_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function event_publish_app_event() failed with error %d.", ret);
		return false;
	}

	return true;
}

bool model_add_system_event_handler(system_ev_info_s *ev_info, event_cb callback, void *user_data)
{
	if (!ev_info || !callback) {
		controller_log(DLOG_ERROR, "Wrong argument provided.");
		return false;
	}

	int ret = event_add_event_handler(ev_info->name, callback, user_data, &ev_info->event_h);
	if (ret != EVENT_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function event_add_event_handler() failed with error = %d.", ret);
		return false;
	}

	return true;
}

bool model_add_custom_event_handler(custom_ev_info_s *ev_info, event_cb callback, void *user_data)
{
	if (!ev_info || !callback) {
		controller_log(DLOG_ERROR, "Wrong argument provided.");
		return false;
	}

	int ret = event_add_event_handler(ev_info->name, callback, user_data, &ev_info->event_h);
	if (ret != EVENT_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function event_add_event_handler() failed with error = %d.", ret);
		return false;
	}

	return true;
}

bool model_assign_event_status(char **ev_status, const char *new_status)
{
	if (!new_status)
		return false;

	if (*ev_status)
		free(*ev_status);

	*ev_status = strdup(new_status);

	return true;
}

bool model_check_event_exists(const char *event_name, bool *exists)
{
	*exists = false;

	if (!event_name) {
		controller_log(DLOG_ERROR, "Wrong argument provided.");
		return false;
	}

	char *formatted_name = NULL;

	__model_format_custom_event_name(event_name, &formatted_name);

	int i;
	for (i = 0; i < __custom_ev_count; i++) {
		if (strncmp(formatted_name, __custom_ev[i].name, strlen(formatted_name)) == 0 &&
			strlen(formatted_name) == strlen(__custom_ev[i].name)) {
			*exists = true;
			free(formatted_name);
			return true;
		}
	}

	free(formatted_name);

	return true;
}

bool model_get_bundle_str(bundle *bundle_obj, const char *key, char **str)
{
	*str = NULL;

	int ret = bundle_get_str(bundle_obj, key, str);
	if (ret != BUNDLE_ERROR_NONE) {
		controller_log(DLOG_ERROR, "Function bundle_get_str() failed with error = %d.", ret);
		return false;
	}

	return true;
}
