/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <app.h>
#include <Elementary_GL_Helpers.h>
#include <efl_extension.h>
#include "$(appName)_teapot.h"
#include "$(appName)_utils.h"
#include "$(appName).h"

ELEMENTARY_GLVIEW_GLOBAL_DEFINE();

const float EPSILON = 0.5f;
const int PAUSE_TIME = 100;

/* Vertext Shader Source */
static const char vertex_shader[] =
		"attribute vec3 a_position;\n"
		"attribute vec3 a_normal;\n"
		"uniform mat4 u_mvpMatrix;\n"
		"uniform vec4 u_light_dir;\n"
		"uniform float u_time_stamp;\n"
		"varying float intensity;\n"
		"\n"
		"void main()\n"
		"{\n"
		"    intensity = dot(u_light_dir, vec4(a_normal,1.0));\n"
		"    gl_Position = u_mvpMatrix * vec4(a_position.x + 0.25*sin(3.0*(a_position.z+u_time_stamp)),a_position.y,a_position.z,1.0);\n"
		"}";

/* Fragment Shader Source */
static const char fragment_shader[] =
		"#ifdef GL_ES\n"
		"precision mediump float;\n"
		"#endif\n"
		"varying float intensity;\n"
		"\n"
		"void main (void)\n"
		"{\n"
		"    vec4 color;\n"
		"    color = vec4(0.8*intensity,0.4*intensity,0.4*intensity,1.0);\n"
		"    gl_FragColor = color;\n"
		"}";

static void win_back_cb(void *data, Evas_Object *obj, void *event_info) {
	appdata_s *ad = data;
	/* Let window go to hidden state. */
	elm_win_lower(ad->win);
}

static void init_shaders(Evas_Object *obj) {
	appdata_s *ad = evas_object_data_get(obj, "ad");
	const char *p;

	p = vertex_shader;
	ad->vtx_shader = glCreateShader(GL_VERTEX_SHADER);
	glShaderSource(ad->vtx_shader, 1, &p, NULL);
	glCompileShader(ad->vtx_shader);

	p = fragment_shader;
	ad->fgmt_shader = glCreateShader(GL_FRAGMENT_SHADER);
	glShaderSource(ad->fgmt_shader, 1, &p, NULL);
	glCompileShader(ad->fgmt_shader);

	ad->program = glCreateProgram();
	glAttachShader(ad->program, ad->vtx_shader);
	glAttachShader(ad->program, ad->fgmt_shader);

	glLinkProgram(ad->program);

	ad->idx_light_dir = glGetUniformLocation(ad->program, "u_light_dir");
	ad->idx_mvp = glGetUniformLocation(ad->program, "u_mvpMatrix");
	ad->idx_time_stamp = glGetUniformLocation(ad->program, "u_time_stamp");

	ad->idx_vposition = glGetAttribLocation(ad->program, "a_position");
	ad->idx_vnormal = glGetAttribLocation(ad->program, "a_normal");

	glUseProgram(ad->program);
}

static void resize_gl(Evas_Object *obj)
{
	appdata_s *ad = evas_object_data_get(obj, "ad");

	elm_glview_size_get(obj, &ad->glview_w, &ad->glview_h);
}

static void draw_gl(Evas_Object *obj)
{
	appdata_s *ad = evas_object_data_get(obj, "ad");
	float model[16], view[16];
	float aspect;
	int rotation;

	if (!ad)
		return;

	ad->time_stamp += ad->stride_time_stamp;

	if (ad->time_stamp > 2 * M_PI)
		ad->time_stamp = 0.0f;

	if (ad->stop_count < 0) {
		ad->angle += 1.0f;

		if (ad->angle >= 360.0f)
			ad->angle -= 360.0f;

		float remain = ad->angle - 90.0f * (float) ((int) (ad->angle / 90.0f));

		if (remain > -EPSILON && remain < EPSILON)
			ad->stop_count = 0;
	} else {
		ad->stop_count++;

		if (ad->stop_count == PAUSE_TIME)
			ad->stop_count = -1;
	}

	init_matrix(model);
	init_matrix(view);

	rotation = elm_glview_rotation_get(obj);

	if (rotation % 180)
		aspect = (float) ad->glview_h / (float) ad->glview_w;
	else
		aspect = (float) ad->glview_w / (float) ad->glview_h;

	view_set_perspective(view, 60.0f, aspect, 1.0f, 20.0f);

	if (rotation != ad->view_rotation)
		rotate_xyz(view, (rotation - ad->view_rotation), 0.0f, 0.0f, -1.0f);

	translate_xyz(model, 0.0f, 0.0f, -3.5f);
	rotate_xyz(model, ad->angle, 1.0f, 0.0f, 0.0f);

	multiply_matrix(ad->mvp, view, model);

	glViewport(0, 0, ad->glview_w, ad->glview_h);

	glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glUniform4fv(ad->idx_light_dir, 1, ad->light_dir);
	glUniformMatrix4fv(ad->idx_mvp, 1, GL_FALSE, ad->mvp);
	glUniform1f(ad->idx_time_stamp, ad->time_stamp);

	glBindBuffer(GL_ARRAY_BUFFER, ad->idx_vbo);
	glVertexAttribPointer(ad->idx_vposition, 3, GL_FLOAT, GL_FALSE,
			sizeof(float) * 6, 0);
	glEnableVertexAttribArray(ad->idx_vposition);
	glVertexAttribPointer(ad->idx_vnormal, 3, GL_FLOAT, GL_FALSE,
			sizeof(float) * 6, (void*) (sizeof(float) * 3));
	glEnableVertexAttribArray(ad->idx_vnormal);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, ad->idx_ibo);
	glDrawElements(GL_TRIANGLES, MAX_F_COUNT * 3, GL_UNSIGNED_SHORT, 0);

	glFlush();

	glDisableVertexAttribArray(ad->idx_vposition);
	glDisableVertexAttribArray(ad->idx_vnormal);
	glBindBuffer(GL_ARRAY_BUFFER, 0);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

	display_fps();
}

static void init_gl(Evas_Object *obj)
{
	appdata_s *ad = evas_object_data_get(obj, "ad");

	/* Set rotation variables */
	ad->idx_vbo = 0;
	ad->idx_ibo = 0;
	ad->idx_vposition = 0;
	ad->idx_vnormal = 0;
	ad->idx_light_dir = 0;
	ad->idx_mvp = 0;
	ad->idx_time_stamp = 0;

	ad->angle = 0.0f;
	ad->stop_count = -1;
	ad->time_stamp = 1.0f;
	ad->stride_time_stamp = 0.02f * M_PI;

	if (!ad->initialized) {
		init_shaders(obj);

		ad->light_dir[0] = 1.0 / sqrt(3);
		ad->light_dir[1] = 1.0 / sqrt(3);
		ad->light_dir[2] = -1.0 / sqrt(3);
		ad->light_dir[3] = 1.0;

		glEnable(GL_DEPTH_TEST);

		glGenBuffers(1, &ad->idx_vbo);
		glBindBuffer(GL_ARRAY_BUFFER, ad->idx_vbo);
		glBufferData(GL_ARRAY_BUFFER, 6 * sizeof(float) * MAX_V_COUNT,
				TEAPOT_VERTICES, GL_STATIC_DRAW);

		glGenBuffers(1, &ad->idx_ibo);
		glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, ad->idx_ibo);
		glBufferData(GL_ELEMENT_ARRAY_BUFFER,
				3 * sizeof(unsigned short) * MAX_F_COUNT, TEAPOT_INDICES,
				GL_STATIC_DRAW);

		ad->initialized = EINA_TRUE;
	}
}

static void del_gl(Evas_Object *obj)
{
	appdata_s *ad = evas_object_data_get(obj, "ad");

	glDeleteBuffers(1, &ad->idx_vbo);
	glDeleteBuffers(1, &ad->idx_ibo);
	glDeleteShader(ad->vtx_shader);
	glDeleteShader(ad->fgmt_shader);
	glDeleteProgram(ad->program);

	evas_object_data_del((Evas_Object*) obj, "ad");
}

static void del_anim(void *data, Evas *evas, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	ecore_animator_del(ad->ani);
}

static Eina_Bool anim(void *data)
{
	elm_glview_changed_set(data);
	return EINA_TRUE;
}


static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static void create_indicator(appdata_s *ad)
{
	elm_win_conformant_set(ad->win, EINA_TRUE);

	elm_win_indicator_mode_set(ad->win, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(ad->win, ELM_WIN_INDICATOR_TRANSPARENT);

	ad->conform = elm_conformant_add(ad->win);
	evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(ad->win, ad->conform);
	evas_object_show(ad->conform);
}

static Evas_Object* add_win(const char *name)
{
	Evas_Object *win;


    // To use the Direct Rendering mode of GLView, set the same option values (depth, stencil, and MSAA)
    // to a rendering engine and a GLView object.
	elm_config_accel_preference_set("opengl:depth");
	win = elm_win_util_standard_add(name, "OpenGL example: Tea pot");

	if (!win)
		return NULL;

	if (elm_win_wm_rotation_supported_get(win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(win, rots, 4);
	}

	evas_object_show(win);

	return win;
}

static bool app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
	 * Initialize UI resources and application's data
	 * If this function returns true, the main loop of application starts
	 * If this function returns false, the application is terminated. */

	Evas_Object *gl;
	appdata_s *ad = data;

	if (!data)
		return false;

	/* Create the window */
	ad->win = add_win(ad->name);

	if (!ad->win)
		return false;

	create_indicator(ad);
	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);
	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);

	/* Create and initialize GLView */
	gl = elm_glview_add(ad->conform);
	ELEMENTARY_GLVIEW_GLOBAL_USE(gl);
	evas_object_size_hint_align_set(gl, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_size_hint_weight_set(gl, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	/* Request a surface with a depth buffer
	 *
	 * To use the Direct Rendering mode, set the same option values (depth, stencil, and MSAA)
	 * to a rendering engine and a GLView object.
	 * You can set the option values to a rendering engine using the elm_config_accel_preference_set() function and
	 * to a GLView object using the elm_glview_mode_set() function.
	 * If the GLView object option values are bigger or higher than the rendering engine's,
	 * the Direct Rendering mode is disabled.
	 */
	elm_glview_mode_set(gl, ELM_GLVIEW_DEPTH | ELM_GLVIEW_DIRECT | ELM_GLVIEW_CLIENT_SIDE_ROTATION);

	/* The resize policy tells GLView what to do with the surface when it
	 * resizes. ELM_GLVIEW_RESIZE_POLICY_RECREATE will tell it to
	 * destroy the current surface and recreate it to the new size.
	 */
	elm_glview_resize_policy_set(gl, ELM_GLVIEW_RESIZE_POLICY_RECREATE);

	/* The render policy sets how GLView should render GL code.
	 * ELM_GLVIEW_RENDER_POLICY_ON_DEMAND will have the GL callback
	 * called only when the object is visible.
	 * ELM_GLVIEW_RENDER_POLICY_ALWAYS would cause the callback to be
	 * called even if the object were hidden.
	 */
	elm_glview_render_policy_set(gl, ELM_GLVIEW_RENDER_POLICY_ON_DEMAND);

	/* The initialize callback function gets registered here */
	elm_glview_init_func_set(gl, init_gl);

	/* The delete callback function gets registered here */
	elm_glview_del_func_set(gl, del_gl);

	/* The resize callback function gets registered here */
	elm_glview_resize_func_set(gl, resize_gl);

	/* The render callback function gets registered here */
	elm_glview_render_func_set(gl, draw_gl);

	/* Add the GLView to the conformant and show it */
	elm_object_content_set(ad->conform, gl);
	evas_object_show(gl);

	elm_object_focus_set(gl, EINA_TRUE);

	/* This adds an animator so that the app will regularly
	 * trigger updates of the GLView using elm_glview_changed_set().
	 *
	 * NOTE: If you delete GL, this animator will keep running trying to access
	 * GL so this animator needs to be deleted with ecore_animator_del().
	 */
	ad->ani = ecore_animator_add(anim, gl);
	evas_object_data_set(gl, "ad", ad);
	evas_object_event_callback_add(gl, EVAS_CALLBACK_DEL, del_anim, gl);

	evas_object_show(ad->win);

	/* Return true: the main loop will now start running */
	return true;
}

static void app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void app_pause(void *data)
{
	appdata_s *ad = data;
	ecore_animator_freeze(ad->ani);
}

static void app_resume(void *data)
{
	appdata_s *ad = data;
	ecore_animator_thaw(ad->ani);
}

static void app_terminate(void *data)
{
	/* Release all resources. */
}

int main(int argc, char *argv[])
{
	int ret = 0;
	appdata_s ad = { NULL, };
	ui_app_lifecycle_callback_s event_callback = {NULL,};

	ad.name = "$(appName)";

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "The application failed to start, and returned %d", ret);
	}

	return ret;
}
