/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "model/notification-control.h"
#include <badge.h>
#include "model/constants.h"
#include "utils/logger.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <notification.h>
#include <Eina.h>

enum ongoing_notification {
	ONGOING_NOTIFICATION_PERCENT,
	ONGOING_NOTIFICATION_BYTE,
	ONGOING_NOTIFICATION_TEXT,
	ONGOING_NOTIFICATION_BY_APP_CONTROL = 4,
	ONGOING_NOTIFICATION_BY_APP_ID = 6,
};

static notification_data *ongoing_notification_data_get(enum ongoing_notification type);
static void notification_content_text_set(notification_h notification, const char *content_text, int press_count);

static notification_h create_ongoing_notification(const char *image_path, const char *title_text)
{
	int size_of_buffer = 0;
	char *full_icon_path = NULL;
	notification_h ongoing_notification = notification_create(NOTIFICATION_TYPE_ONGOING);
	RETVM_IF(!ongoing_notification, NULL, "notify is NULL");

	if (image_path) {
		size_of_buffer = strlen(image_path) + strlen(get_shared_res_path()) + 1;
		full_icon_path = calloc(sizeof(char), size_of_buffer);
		if (full_icon_path) {
			snprintf(full_icon_path, size_of_buffer, "%s%s", get_shared_res_path(), image_path);
			notification_set_image(ongoing_notification, NOTIFICATION_IMAGE_TYPE_ICON, full_icon_path);
			free(full_icon_path);
		}
	}

	notification_set_property(ongoing_notification, NOTIFICATION_PROP_DISABLE_TICKERNOTI);
	notification_set_text(ongoing_notification, NOTIFICATION_TEXT_TYPE_TITLE, title_text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);

	return ongoing_notification;
}

static void notification_content_text_set(notification_h notification, const char *content_text, int press_count)
{
	char text[TEXT_SHORT_LEN] = {'\0'};
	snprintf(text, sizeof(text), "%s%d", content_text, press_count);
	notification_set_text(notification, NOTIFICATION_TEXT_TYPE_CONTENT, text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
}

static bool delete_ongoing_notification(enum ongoing_notification type)
{
	notification_data *data = ongoing_notification_data_get(type);
	int err = NOTIFICATION_ERROR_NONE;
	if (data) {
		if (data->notification) {
			err = notification_delete(data->notification);
			data->notification = NULL;
		}
		data->press_count = 0;
		data->progress_value = 0;
	}

	return (err == NOTIFICATION_ERROR_NONE);
}

void delete_all_ongoing_notifications()
{
	delete_ongoing_notification(ONGOING_NOTIFICATION_PERCENT);
	delete_ongoing_notification(ONGOING_NOTIFICATION_BYTE);
	delete_ongoing_notification(ONGOING_NOTIFICATION_TEXT);
	delete_ongoing_notification(ONGOING_NOTIFICATION_BY_APP_CONTROL);
	delete_ongoing_notification(ONGOING_NOTIFICATION_BY_APP_ID);
}

static void ongoing_notification_cb(notification_data *notify_data)
{
	RETM_IF(!notify_data, "notify_data is NULL");
	bool to_post = false;

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = create_ongoing_notification(ICON2_PATH, notify_data->name);
	}
	RETM_IF(!notify_data->notification, "ongoing_notification is NULL");

	notify_data->progress_value += percentage_increment;
	if (notify_data->progress_value > percentage_max) {
		notify_data->progress_value = 0;
	}
	notification_set_progress(notify_data->notification, notify_data->progress_value);
	notification_content_text_set(notify_data->notification, percentage_content_text, notify_data->press_count);

	int value = round(percentage_multiplier * notify_data->progress_value);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br> <br>%s<br>%s%d<br> <br>%s<br>%s = %d<br> <br>%s<br>%d<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
					notify_message_text, percentage_content_text, notify_data->press_count, launch_argument_text, app_message_text, value,
					ongoing_activity_progress_text, value);

	char launch_argument[TEXT_SHORT_LEN] = {'\0'};
	snprintf(launch_argument, TEXT_SHORT_LEN, "%s = %d", app_message_text, value);

	launch_arguments_set(notify_data->notification, launch_argument, PACKAGE, NULL, NULL, NULL, NULL, NULL);

	if (to_post) {
		notification_post(notify_data->notification);
	} else {
		notification_update(notify_data->notification);
	}
}

static void ongoing_notification_byte_cb(notification_data *notify_data)
{
	RETM_IF(!notify_data, "notify_data is NULL");
	bool to_post = false;

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = create_ongoing_notification(ICON2_PATH, notify_data->name);
	}
	RETM_IF(!notify_data->notification, "ongoing_notification_byte is NULL");

	if (notify_data->progress_value > byte_max) {
		notify_data->progress_value = 0;
	}
	notification_set_size(notify_data->notification, notify_data->progress_value);
	notification_content_text_set(notify_data->notification, byte_content_text, notify_data->press_count);

	char launch_argument[TEXT_SHORT_LEN] = {'\0'};
	snprintf(launch_argument, TEXT_SHORT_LEN, "%s = %d", app_message_text, (int)round(notify_data->progress_value));

	launch_arguments_set(notify_data->notification, launch_argument, PACKAGE, NULL, NULL, NULL, NULL, NULL);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s%d<br> <br>%s<br>%s = %d<br><br>%s<br>%d<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, percentage_content_text, notify_data->press_count, launch_argument_text, app_message_text, (int)round(notify_data->progress_value),
			ongoing_activity_progress_text, (int)round(notify_data->progress_value));

	notify_data->progress_value += byte_increment;
	if (to_post) {
		notification_post(notify_data->notification);
	} else {
		notification_update(notify_data->notification);
	}
}

static void ongoing_notification_text_cb(notification_data *notify_data)
{
	RETM_IF(!notify_data, "notify_data is NULL");
	bool to_post = false;

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = create_ongoing_notification(ICON2_PATH, notify_data->name);
	}
	RETM_IF(!notify_data->notification, "ongoing_notification_byte is NULL");

	notification_set_layout(notify_data->notification, NOTIFICATION_LY_ONGOING_EVENT);
	notification_content_text_set(notify_data->notification, text_content_text, notify_data->press_count);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s%d<br><br>%s<br>%s<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, text_content_text, notify_data->press_count, launch_argument_text, app_message_text);

	launch_arguments_set(notify_data->notification, app_message_text, PACKAGE, NULL, NULL, NULL, NULL, NULL);

	if (to_post) {
		notification_post(notify_data->notification);
	} else {
		notification_update(notify_data->notification);
	}
}

static void remove_ongoing_notification_cb(notification_data *notify_data)
{
	RETM_IF(!notify_data, "notify_data is NULL");

	bool is_success = true;
	is_success = delete_ongoing_notification(ONGOING_NOTIFICATION_PERCENT);
	is_success = delete_ongoing_notification(ONGOING_NOTIFICATION_BYTE) && is_success;
	is_success = delete_ongoing_notification(ONGOING_NOTIFICATION_TEXT) && is_success;

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>", result_message,
			(is_success) ? result_message_success_text : result_message_failure_text, remove_ongoing_notification_text);
}

static void ongoing_notification_by_app_control_cb(notification_data *notify_data)
{
	RETM_IF(!notify_data, "notify_data is NULL");
	bool to_post = false;

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = create_ongoing_notification(ICON1_PATH, notify_data->name);
	}
	RETM_IF(!notify_data->notification, "ongoing_notification is NULL");

	notify_data->progress_value += percentage_increment;
	if (notify_data->progress_value > percentage_max) {
		notify_data->progress_value = 0;
	}
	notification_set_progress(notify_data->notification, notify_data->progress_value);
	notification_content_text_set(notify_data->notification, percentage_content_text, notify_data->press_count);

	unsigned int count = 0;
	if (!increase_badge(&count)) {
		ERR("increase_badge failed");
	}

	int value = round(percentage_multiplier * notify_data->progress_value);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br> <br>%s<br>%s%d<br> <br>%s<br>%s = %d<br> <br>%s<br>%d<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, percentage_content_text, notify_data->press_count, launch_argument_text, ongoing_by_app_control_text, value,
			ongoing_activity_progress_text, value);

	launch_arguments_set(notify_data->notification, ongoing_by_app_control_text, PACKAGE, uri_text, mime_text, operation_id_text, NULL, NULL);

	if (to_post) {
		notification_post(notify_data->notification);
	} else {
		notification_update(notify_data->notification);
	}
}

static void remove_notification_by_app_control_cb(notification_data *notify_data)
{
	RETM_IF(!notify_data, "notify_data is NULL");

	bool result = delete_ongoing_notification(ONGOING_NOTIFICATION_BY_APP_CONTROL);
	badge_remove(PACKAGE);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>", result_message,
			(result) ? result_message_success_text : result_message_failure_text, remove_ongoing_notification_by_app_control_text);
}

static void ongoing_notification_by_app_id_cb(notification_data *notify_data)
{
	RETM_IF(!notify_data, "notify_data is NULL");
	bool to_post = false;

	notify_data->press_count++;
	if (!notify_data->notification) {
		to_post = true;
		notify_data->notification = create_ongoing_notification(ICON1_PATH, notify_data->name);
	}
	RETM_IF(!notify_data->notification, "ongoing_notification_byte is NULL");

	if (notify_data->progress_value > byte_max) {
		notify_data->progress_value = 0;
	}
	notification_set_size(notify_data->notification, notify_data->progress_value);
	notification_content_text_set(notify_data->notification, byte_content_text,
			notify_data->press_count);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br> <br>%s<br>%s%d<br> <br>%s<br>%s = %d<br> <br>%s<br>%d<br>", result_message,
			(notify_data->notification) ? result_message_success_text : result_message_failure_text,
			notify_message_text, byte_content_text, notify_data->press_count, launch_argument_text, app_message_text, (int)round(notify_data->progress_value),
			ongoing_activity_progress_text, (int)round(notify_data->progress_value));

	launch_arguments_set(notify_data->notification, app_message_text, SETTING_PKGNAME, NULL, NULL, NULL, NULL, NULL);

	notify_data->progress_value += byte_increment;
	if (to_post) {
		notification_post(notify_data->notification);
	} else {
		notification_update(notify_data->notification);
	}
}

static void remove_ongoing_notification_by_app_id_cb(notification_data *notify_data)
{
	RETM_IF(!notify_data, "notify_data is NULL");

	bool result = delete_ongoing_notification(ONGOING_NOTIFICATION_BY_APP_ID);

	snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>", result_message,
			(result) ? result_message_success_text : result_message_failure_text, remove_ongoing_notification_by_app_id_text);
}

static notification_data *ongoing_notification_data_get(enum ongoing_notification type)
{
	int size = 0;
	notification_data *data = NULL;
	notification_data *list = (notification_data *)ongoing_notification_list_get(&size);
	if (type < size) {
		data = &list[type];
	}
	return data;
}

const notification_data *ongoing_notification_list_get(int *size)
{
	static notification_data ongoing_notification[] = {
			{ "OngoingNotification(%)", NULL,  0, 0, ongoing_notification_cb, },
			{ "OngoingNotification(byte)", NULL,  0, 0, ongoing_notification_byte_cb, },
			{ "OngoingNotification(text)", NULL,  0, 0, ongoing_notification_text_cb, },
			{ "RemoveOngoingNotification", NULL,  0, 0, remove_ongoing_notification_cb, },
			{ "OngoingNotificationByAppControl", NULL,  0, 0, ongoing_notification_by_app_control_cb, },
			{ "RemoveNotificationByAppControl", NULL,  0, 0, remove_notification_by_app_control_cb, },
			{ "OngoingNotificationByAppId", NULL,  0, 0, ongoing_notification_by_app_id_cb, },
			{ "RemoveNotificationByAppId", NULL,  0, 0, remove_ongoing_notification_by_app_id_cb, }
	};
	*size = sizeof(ongoing_notification) / sizeof(ongoing_notification[0]);
	return ongoing_notification;
}
