#include "$(appName).h"
#include "view.h"
#include "conf.h"

typedef enum {
	COLOR_SELECTOR_MAIN = 0,
	COLOR_SELECTOR_FILL,
} color_selector_t;

typedef struct {
	mode_type_t type;
	Eina_List *parts;
} object_t;

static struct {
	Evas_Object *win;
	Evas_Object *layout;
	Evas_Object *draw_area;
	object_t *current_object;
	object_t *selected;
	Evas_Object *selection_frame;

	Evas *evas;

	int r1;
	int g1;
	int b1;
	int a1;

	int r2;
	int g2;
	int b2;
	int a2;

	Evas_Coord_Point start;
	Evas_Coord_Point prev;
	Evas_Coord_Point curr;

	mode_type_t mode;
	mode_type_t prev_mode;
	bool mouse_pressed;

	Eina_List *objects;
	int win_layer;

} s_info = {
	.win = NULL,
	.draw_area = NULL,
	.evas = NULL,
	.current_object = NULL,
	.selected = NULL,
	.selection_frame = NULL,
	.r1 = 0,
	.g1 = 0,
	.b1 = 0,
	.a1 = 255,

	.r2 = 255,
	.g2 = 255,
	.b2 = 255,
	.a2 = 255,

	.start = {0, 0},
	.prev = {0, 0},
	.curr = {0, 0},

	.mode = SELECT,
	.mouse_pressed = false,
	.objects = NULL,
	.win_layer = 0,
};


static void __get_app_resource(const char *edj_file_in, char *edj_path_out, int edj_path_max);
static Evas_Object * __get_first_part_of_current_object(void);
static Evas_Object * __get_first_part_of_selected_object(void);
static Eina_Bool __is_cursor_in_selection_frame(void);
static void __unselect_item(void);
static void __set_color_selector_panel_visibility(appdata_s *ad);
static void __toolbar_colorselector_show_cb(void *data, Evas_Object *obj, void *event_info);
static void __toolbar_selected_item_del_cb(void *data, Evas_Object *obj, void *event_info);
static void __key_press_cb(void *data, Evas_Object *obj, void *event_info);
static void __delete_win_request_cb(void *data, Evas_Object *obj, void *event_info);
static Elm_Object_Item * __append_toolbar_item(Evas_Object *toolbar, icon_data_s *id, Evas_Smart_Cb func, void *data);
static void __set_toolbar_mode(mode_type_t mode);
static void __set_color(color_selector_t cs_num, int r, int g, int b, int a);
static void __clear_draw_area(void);
static object_t * __create_object(mode_type_t type);
static void __add_object_part(object_t *object, Evas_Object *part);
static void __get_object_geometry(object_t *selected_object, int *min_x, int *min_y, int *max_x, int *max_y);
static void __update_selection_frame(object_t *selected_object);
static void __set_selected_item(object_t *selected);
static void __create_line(void);
static void __update_line(void);
static Evas_Object * __add_rect_part(object_t *object, int r, int g, int b, int a, int x, int y);
static void __create_rect(void);
static void __update_rect(void);
static void __update_freehand(void);
static Evas_Object * __create_circle_part(object_t *object, int r, int g, int b, int a);
static void __update_circle_part(Evas_Object *part, int radius_x, int radius_y, int center_x, int center_y);
static void ___update_circle(void);
static void __move_object(void);
static void __tollbar_item_clicked_cb(void *data, Evas_Object *obj, void *event_info);
static void __toolbar_clear_clicked_cb(void *data, Evas_Object *obj, void *event_info);
static void __colorselector_color_set_cb(void *data, Evas_Object *obj, void *event_info);
static void __mouse_up_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void __mouse_down_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void __mouse_move_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static Evas_Object * __create_app_win(void);
static Evas_Object * __create_conformant(Evas_Object *win);
static Evas_Object * __create_layout(Evas_Object *win, Evas_Object *conform, appdata_s *ad);
static Eina_Bool __create_toolbar(Evas_Object *layout, appdata_s *ad);
static Evas_Object * __create_colorselector(Evas_Object *win, color_selector_t cs_num);
static void __box_resize_cb(void *data, Evas *e, Evas_Object *box, void *event_info);
static Evas_Object * __create_color_selector_panel(Evas_Object *win);
static Eina_Bool __create_draw_area(Evas_Object *layout);
static Eina_Bool __get_win_evas(Evas_Object *win);
static Eina_Bool __create_selection_frame(appdata_s *ad);
static void __win_rotation_cb(void *data, Evas_Object *obj, void *event_info);
static void __object_clicked_cb(void *data, Evas *e, Evas_Object *selected, void *event_info);
static void __box_position_set(Evas_Object *win, Evas_Object *box, int win_rotation);

Eina_Bool
create_base_gui(appdata_s *ad)
{
	ad->win = __create_app_win();
	if (!ad->win) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _app_win_create() failed");
		return EINA_FALSE;
	}

	ad->conform = __create_conformant(ad->win);
	if (!ad->conform) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _conformant_create() failed");
		evas_object_del(ad->win);
		return EINA_FALSE;
	}

	ad->layout = __create_layout(ad->win, ad->conform, ad);
	if (!ad->layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _layout_create() failed");
		evas_object_del(ad->win);
		return EINA_FALSE;
	}

	if (!__create_toolbar(ad->layout, ad)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _toolbar_create() failed");
		evas_object_del(ad->win);
		return EINA_FALSE;
	}

	ad->color_selector_panel = __create_color_selector_panel(ad->win);
	if (!ad->color_selector_panel) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _color_selector_create() failed");
		evas_object_del(ad->win);
		return EINA_FALSE;
	}

	if (!__get_win_evas(ad->win)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _draw_area_create() failed");
		evas_object_del(ad->win);
		return EINA_FALSE;
	}

	if (!__create_selection_frame(ad)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _selection_frame_create() failed");
		evas_object_del(ad->win);
		return EINA_FALSE;
	}

	if (!__create_draw_area(ad->layout)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _draw_area_create() failed");
		evas_object_del(ad->win);
		return EINA_FALSE;
	}

	return EINA_TRUE;
}

static void
__get_app_resource(const char *edj_file_in, char *edj_path_out, int edj_path_max)
{
	char *res_path = app_get_resource_path();
	if (res_path) {
		snprintf(edj_path_out, edj_path_max, "%s%s", res_path, edj_file_in);
		free(res_path);
	}
}

static Evas_Object *
__get_first_part_of_current_object(void)
{
	if (!s_info.current_object) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.current_object == NULL");
		return NULL;
	}

	return eina_list_data_get(s_info.current_object->parts);
}

static Evas_Object *
__get_first_part_of_selected_object(void)
{
	if (!s_info.selected) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.current_object == NULL");
		return NULL;
	}

	return eina_list_data_get(s_info.selected->parts);
}


static Eina_Bool
__is_cursor_in_selection_frame(void)
{
	int x = 0, y = 0, w = 0, h = 0;
	if (!s_info.selection_frame) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.selection_framet == NULL");
		return EINA_FALSE;
	}

	evas_object_geometry_get(s_info.selection_frame, &x, &y, &w, &h);
	dlog_print(DLOG_INFO, LOG_TAG, "[%d, %d, %d, %d] (%d, %d)", x, y, w, h, s_info.start.x, s_info.start.y);

	if (s_info.start.x >= x &&
		s_info.start.y >= y &&
		s_info.start.x <= x + w &&
		s_info.start.y <= y + h &&
		evas_object_visible_get(s_info.selection_frame)) {
			return EINA_TRUE;
	}

	return EINA_FALSE;
}


static void
__unselect_item(void)
{
	s_info.selected = NULL;
	evas_object_hide(s_info.selection_frame);

}

static void
__set_color_selector_panel_visibility(appdata_s *ad)
{
	Eina_Bool is_visible = EINA_FALSE;

	if (!ad->color_selector_panel) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ad->color_selector_panel == NULL");
		return;
	}

	is_visible = evas_object_visible_get(ad->color_selector_panel);

	if (!is_visible) {
		evas_object_show(ad->color_selector_panel);
	} else {
		evas_object_hide(ad->color_selector_panel);
	}
}

static void
__toolbar_colorselector_show_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;
	if (!ad) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ad == NULL");
		return;
	}

	if (!ad->icon_data[s_info.mode].item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ad->icon_data[s_info.mode].item == NULL");
		return;
	}

	elm_toolbar_item_selected_set(ad->icon_data[s_info.mode].item, EINA_TRUE);


	__set_color_selector_panel_visibility(ad);
}

static void
__toolbar_selected_item_del_cb(void *data, Evas_Object *obj, void *event_info)
{
	Eina_List *l = NULL;
	Evas_Object *part = NULL;
	appdata_s *ad = (appdata_s *)data;
	if (!ad) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ad == NULL");
		return;
	}

	if (!ad->icon_data[s_info.mode].item) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ad->icon_data[s_info.mode].item == NULL");
		return;
	}

	elm_toolbar_item_selected_set(ad->icon_data[s_info.mode].item, EINA_TRUE);

	if (!s_info.selected) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.current_object == NULL");
		return;
	}

	EINA_LIST_FOREACH(s_info.selected->parts, l, part)
	{
		if (!part) {
			dlog_print(DLOG_ERROR, LOG_TAG, "part == NULL");
			continue;
		}

		dlog_print(DLOG_INFO, LOG_TAG, "Removing part of type: %s", evas_object_type_get(part));
		evas_object_del(part);
	}

	s_info.objects = eina_list_remove(s_info.objects, s_info.selected);
	free(s_info.selected);
	__unselect_item();
}

static void
__key_press_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = (appdata_s *)data;

	if (!ad) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ad == NULL");
		return;
	}

	if (!ad->color_selector_panel) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ad->color_selector_panel == NULL", ad->color_selector_panel);
		return;
	}

	dlog_print(DLOG_INFO, LOG_TAG, "PRESSED");
	if (evas_object_visible_get(ad->color_selector_panel)) {
		__set_color_selector_panel_visibility(ad);
	} else {
		elm_win_lower(ad->win);
	}

	return;
}

static void
__win_rotation_cb(void *data, Evas_Object *obj, void *event_info)
{
	Evas_Object *layout = NULL;
	Evas_Object *box = NULL;
	appdata_s *ad = data;
	int rotation = elm_win_rotation_get(obj);

	if (!ad) {
		dlog_print(DLOG_ERROR, LOG_TAG, "layout == NULL");
		return;
	}

	layout = ad->layout;
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "layout == NULL");
		return;
	}

	box = ad->color_selector_panel;
	if (!box) {
		dlog_print(DLOG_ERROR, LOG_TAG, "box == NULL");
		return;
	}

	if (rotation == 90 || rotation == 270) {
		elm_layout_signal_emit(layout, SIGNAL_ROTATION_LANDSCAPE, PART_TOOLBAR);
	} else {
		elm_layout_signal_emit(layout, SIGNAL_ROTATION_PORTRAIT, PART_TOOLBAR);
	}

	__box_position_set(obj, box, rotation);

	return;
}

static void
__delete_win_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	ui_app_exit();
}

static Elm_Object_Item *
__append_toolbar_item(Evas_Object *toolbar, icon_data_s *id, Evas_Smart_Cb func, void *data)
{
	Elm_Object_Item *it = NULL;
	char path[PATH_MAX];

	if (!id) {
		dlog_print(DLOG_ERROR, LOG_TAG, "id == NULL");
		return NULL;
	}

	__get_app_resource(id->file_path, path, (int)PATH_MAX);
	it = elm_toolbar_item_append(toolbar, path, NULL, func, data);

	if (!it) {
		dlog_print(DLOG_ERROR, LOG_TAG, "it == NULL");
		return NULL;
	}

	elm_object_item_tooltip_text_set(it, id->tooltip);
	id->item = it;
	return it;
}

static void
__set_toolbar_mode(mode_type_t mode)
{
	s_info.mode = mode;
	dlog_print(DLOG_INFO, LOG_TAG, "Mode selected: %d", s_info.mode);
}

static void
__set_color(color_selector_t cs_num, int r, int g, int b, int a)
{
	Eina_List *l = NULL;
	Evas_Object *part = NULL;

	if (cs_num == COLOR_SELECTOR_MAIN) {
		s_info.r1 = r;
		s_info.g1 = g;
		s_info.b1 = b;
		s_info.a1 = a;
	} else {
		s_info.r2 = r;
		s_info.g2 = g;
		s_info.b2 = b;
		s_info.a2 = a;
	}

	if (!s_info.selected) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.selected == NULL");
		return;
	}

	if (s_info.selected->type == LINE || s_info.selected->type == FREEHAND) {
		EINA_LIST_FOREACH(s_info.selected->parts, l, part)
		{
			if (!part) {
				dlog_print(DLOG_ERROR, LOG_TAG, "part == NULL");
				continue;
			}
			evas_object_color_set(part, s_info.r1, s_info.g1, s_info.b1, s_info.a1);
		}
	} else {
		if (cs_num == COLOR_SELECTOR_MAIN) {
			part = eina_list_nth(s_info.selected->parts, 0);
		} else {
			part = eina_list_nth(s_info.selected->parts, 1);
		}

		evas_object_color_set(part, r, g, b, a);
	}
}

static void
__clear_draw_area(void)
{
	Eina_List *l = NULL;
	object_t *obj = NULL;
	Evas_Object *part = NULL;
	Eina_List *part_list = NULL;


	EINA_LIST_FOREACH(s_info.objects, l, obj)
	{
		if (!obj) {
			dlog_print(DLOG_ERROR, LOG_TAG, "obj == NULL");
			continue;
		}

		EINA_LIST_FOREACH(obj->parts, part_list, part)
		{
			if (!part) {
				dlog_print(DLOG_ERROR, LOG_TAG, "part == NULL");
				continue;
			}

			evas_object_del(part);
		}

		eina_list_free(obj->parts);
		free(obj);
	}

	s_info.objects = eina_list_free(s_info.objects);
	evas_object_hide(s_info.selection_frame);
}

static object_t *
__create_object(mode_type_t type)
{
	object_t *object = (object_t *)calloc(1, sizeof(object_t));
	if (!object) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create object");
		return NULL;
	}

	object->type = type;
	object->parts = NULL;

	dlog_print(DLOG_INFO, LOG_TAG, "Object of type %d created", type);
	return object;
}

static void
__add_object_part(object_t *object, Evas_Object *part)
{
	if (!object) {
		dlog_print(DLOG_ERROR, LOG_TAG, "object == NULL");
		return;
	}

	if (!part) {
		dlog_print(DLOG_ERROR, LOG_TAG, "part == NULL");
		return;
	}

	evas_object_data_set(part, OBJECT_DATA, object);
	object->parts = eina_list_append(object->parts, part);
	evas_object_event_callback_add(part, EVAS_CALLBACK_MOUSE_DOWN, __object_clicked_cb, NULL);
}


static void
__get_object_geometry(object_t *selected_object, int *min_x, int *min_y, int *max_x, int *max_y)
{
	int x = 0, y = 0, w = 0, h = 0;
	Eina_List *l = NULL;
	Evas_Object *part;

	if (!selected_object) {
		dlog_print(DLOG_ERROR, LOG_TAG, "selected_object == NULL");
		return;
	}

	if (!selected_object->parts) {
		dlog_print(DLOG_ERROR, LOG_TAG, "selected_object->parts == NULL");
		return;
	}

	part = eina_list_data_get(selected_object->parts);
	evas_object_geometry_get(part, min_x, min_y, max_x, max_y);
	*max_x = *min_x + *max_x;
	*max_y = *min_y + *max_y;

	EINA_LIST_FOREACH(selected_object->parts, l, part)
	{
		evas_object_geometry_get(part, &x, &y, &w, &h);

		dlog_print(DLOG_INFO, LOG_TAG, "*** part geometry: [%d %d %d %d] [%d, %d]", x, y, x + w, y + h, w, h);

		if (x < *min_x)
			*min_x = x;

		if (y < *min_y)
			*min_y = y;

		if (x + w > *max_x)
			*max_x = x + w;

		if (y + h > *max_y)
			*max_y = y + h;
	}

	dlog_print(DLOG_INFO, LOG_TAG, "*** object geometry: [%d %d %d %d]",
			*min_x, *min_y, *max_x, *max_y);
}

static void
__update_selection_frame(object_t *selected_object)
{
	int min_x = 0, min_y = 0, max_x = 0, max_y = 0;

	if (!selected_object) {
		dlog_print(DLOG_ERROR, LOG_TAG, "selected_object == NULL");
		return;
	}

	__get_object_geometry(selected_object, &min_x, &min_y, &max_x, &max_y);

	evas_object_move(s_info.selection_frame, min_x - 1, min_y - 1);
	evas_object_resize(s_info.selection_frame, max_x - min_x + 1, max_y - min_y + 1);
	evas_object_show(s_info.selection_frame);
	dlog_print(DLOG_INFO, LOG_TAG, "Frame: %p; [%d, %d] [%d, %d]",
			s_info.selection_frame,
			min_x - 1, min_y - 1,
			max_x - min_x + 1, max_y - min_y + 1);

}

static void
__set_selected_item(object_t *selected)
{
	if (!selected) {
		dlog_print(DLOG_ERROR, LOG_TAG, "selected == NULL");
		return;
	}

	s_info.selected = selected;
	__update_selection_frame(s_info.selected);
}

static void __object_clicked_cb(void *data, Evas *e, Evas_Object *selected, void *event_info)
{
	Eina_Bool frame_visibility = EINA_FALSE;

	if (!s_info.selection_frame) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.selection_frame == NULL");
		return;
	}

	frame_visibility = evas_object_visible_get(s_info.selection_frame);
	evas_object_hide(s_info.selection_frame);

	if (frame_visibility) {
		evas_object_show(s_info.selection_frame);
	}

	dlog_print(DLOG_INFO, LOG_TAG, "Selecting item");
	if (selected == s_info.selection_frame) {
		dlog_print(DLOG_INFO, LOG_TAG, "Trying to select the selection frame. Exiting...");
		return;
	}

	if (selected == s_info.win || selected == s_info.draw_area) {
		dlog_print(DLOG_INFO, LOG_TAG, "No item selected");

		if (!__is_cursor_in_selection_frame())
			__unselect_item();

		return;
	}
	__set_selected_item(evas_object_data_get(selected, OBJECT_DATA));
}

static void
__create_line(void)
{
	Evas_Object *line_new = NULL;
	line_new = evas_object_line_add(s_info.evas);

	if (!line_new) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.current_object == NULL");
		return;
	}

	s_info.current_object = __create_object(LINE);
	if (!s_info.current_object) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _object_create() failed");
		return;
	}

	evas_object_color_set(line_new,
			s_info.r1,
			s_info.g1,
			s_info.b1,
			s_info.a1);

	evas_object_line_xy_set(line_new,
			s_info.start.x,
			s_info.start.y,
			s_info.start.x,
			s_info.start.y);

	dlog_print(DLOG_INFO, LOG_TAG, "Creating line at: [%d, %d]; color: [%d, %d, %d, %d]",
			s_info.start.x, s_info.start.y,
			s_info.r1, s_info.g1, s_info.b1, s_info.a1);

	evas_object_repeat_events_set(line_new, EINA_TRUE);
	evas_object_show(line_new);
	evas_object_layer_set(line_new, s_info.win_layer + 1);

	__add_object_part(s_info.current_object, line_new);
	s_info.objects = eina_list_append(s_info.objects, s_info.current_object);
}

static void
__update_line(void)
{
	int x1 = 0, y1 = 0, x2 = 0, y2 = 0;

	if (!s_info.mouse_pressed) {
		dlog_print(DLOG_INFO, LOG_TAG, "s_info.mouse_pressed == NULL");
		return;
	}

	if (!s_info.current_object) {
		__create_line();
		return;
	}

	if (s_info.start.y <= s_info.curr.y) {
		x1 = s_info.start.x;
		x2 = s_info.curr.x;
		y1 = s_info.start.y;
		y2 = s_info.curr.y;
	} else {
		x2 = s_info.start.x;
		x1 = s_info.curr.x;
		y2 = s_info.start.y;
		y1 = s_info.curr.y;
	}

	evas_object_line_xy_set(__get_first_part_of_current_object(), x1, y1, x2, y2);
}

static Evas_Object *
__add_rect_part(object_t *object, int r, int g, int b, int a, int x, int y)
{
	Evas_Object *part = evas_object_rectangle_add(s_info.evas);
	if (!part) {
		dlog_print(DLOG_ERROR, LOG_TAG, "part == NULL");
		return NULL;
	}

	evas_object_color_set(part, r, g, b, a);
	evas_object_move(part, x, y);

	evas_object_repeat_events_set(part, EINA_TRUE);
	evas_object_show(part);
	evas_object_layer_set(part, s_info.win_layer + 1);
	__add_object_part(object, part);
	return part;
}


static void
__create_rect(void)
{
	s_info.current_object = __create_object(RECTANGLE);
	if (!s_info.current_object) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function _object_create() failed");
		return;
	}

	__add_rect_part(s_info.current_object,
			s_info.r1, s_info.g1, s_info.b1, s_info.a1,
			s_info.start.x, s_info.start.y);

	__add_rect_part(s_info.current_object,
			s_info.r2, s_info.g2, s_info.b2, s_info.a2,
			s_info.start.x, s_info.start.y);

	s_info.objects = eina_list_append(s_info.objects, s_info.current_object);
}

static void
__update_rect(void)
{
	int x1 = 0, y1 = 0, x2 = 0, y2 = 0;
	Evas_Object *part = NULL;

	if (!s_info.mouse_pressed) {
		dlog_print(DLOG_INFO, LOG_TAG, "s_info.mouse_pressed == NULL");
		return;
	}

	if (!s_info.current_object) {
		__create_rect();
		return;
	}

	if (s_info.start.x < s_info.curr.x) {
		x1 = s_info.start.x;
		x2 = s_info.curr.x - s_info.start.x;
	} else {
		x1 = s_info.curr.x;
		x2 = s_info.start.x - s_info.curr.x;
	}

	if (s_info.start.y < s_info.curr.y) {
		y1 = s_info.start.y;
		y2 = s_info.curr.y - s_info.start.y;
	} else {
		y1 = s_info.curr.y;
		y2 = s_info.start.y - s_info.curr.y;
	}

	part = eina_list_nth(s_info.current_object->parts, 0);
	evas_object_move(part, x1, y1);
	evas_object_resize(part, x2, y2);

	part = eina_list_nth(s_info.current_object->parts, 1);
	evas_object_move(part, x1 + OBJECT_BORDER, y1 + OBJECT_BORDER);
	evas_object_resize(part, x2 - (OBJECT_BORDER*2), y2 - (OBJECT_BORDER*2));
}

static void
__update_freehand(void)
{
	Evas_Object *part = NULL;
	int x1 = 0, y1 = 0, x2 = 0, y2 = 0;
	if (!s_info.mouse_pressed) {
		return;
	}

	if (!s_info.current_object) {
		s_info.current_object = __create_object(FREEHAND);
		s_info.objects = eina_list_append(s_info.objects, s_info.current_object);
	}

	part = evas_object_line_add(s_info.evas);
	if (!part) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.current_object == NULL");
		return;
	}

	__add_object_part(s_info.current_object, part);

	if (s_info.prev.y <= s_info.curr.y) {
		x1 = s_info.prev.x;
		x2 = s_info.curr.x;
		y1 = s_info.prev.y;
		y2 = s_info.curr.y;
	} else {
		x2 = s_info.prev.x;
		x1 = s_info.curr.x;
		y2 = s_info.prev.y;
		y1 = s_info.curr.y;
	}

	evas_object_color_set(part, s_info.r1, s_info.g1, s_info.b1, s_info.a1);
	evas_object_line_xy_set(part, x1, y1, x2, y2);

	dlog_print(DLOG_INFO, LOG_TAG, "Updating line at: [%d, %d] - [%d, %d]", x1, y1, x2, y2);

	evas_object_repeat_events_set(part, EINA_TRUE);
	evas_object_show(part);
	evas_object_layer_set(part, s_info.win_layer + 1);
}

static Evas_Object*
__create_circle_part(object_t *object, int r, int g, int b, int a)
{
	Evas_Object *part = NULL;

	if (!s_info.evas) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.evas == NULL");
		return NULL;
	}

	if (!object) {
		dlog_print(DLOG_ERROR, LOG_TAG, "object == NULL");
		return NULL;
	}

	part = evas_object_polygon_add(s_info.evas);
	if (!part) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function evas_object_polygon_add() failed");
		return NULL;
	}

	evas_object_color_set(part, r, g, b, a);

	__add_object_part(object, part);

	evas_object_repeat_events_set(part, EINA_TRUE);
	evas_object_show(part);
	evas_object_layer_set(part, s_info.win_layer + 1);
	return part;
}

static void
__update_circle_part(Evas_Object *part, int radius_x, int radius_y, int center_x, int center_y)
{
	int i;

	if (!part) {
		dlog_print(DLOG_ERROR, LOG_TAG, "part == NULL");
		return;
	}

	evas_object_polygon_points_clear(part);

	for (i = 0; i < 360; ++i)
		evas_object_polygon_point_add(part,
				sin(i * M_PI / 180) * radius_x + center_x,
				cos(i * M_PI / 180) * radius_y + center_y);
}

static void
___update_circle(void)
{
	Evas_Object *part = NULL;
	int radius_x = 0;
	int radius_y = 0;
	int center_x = 0;
	int center_y = 0;

	if (!s_info.mouse_pressed)
		return;

	if (!s_info.current_object) {
		s_info.current_object = __create_object(CIRCLE);
		__create_circle_part(s_info.current_object, s_info.r1, s_info.g1, s_info.b1, s_info.a1);
		__create_circle_part(s_info.current_object, s_info.r2, s_info.g2, s_info.b2, s_info.a2);
		s_info.objects = eina_list_append(s_info.objects, s_info.current_object);
	}

	radius_x = (s_info.curr.x - s_info.start.x) / 2;
	radius_y = (s_info.curr.y - s_info.start.y) / 2;

	center_x = s_info.start.x + radius_x;
	center_y = s_info.start.y + radius_y;

	radius_x = abs(radius_x);
	radius_y = abs(radius_y);

	part = eina_list_nth(s_info.current_object->parts, 0);
	__update_circle_part(part, radius_x, radius_y, center_x, center_y);

	part = eina_list_nth(s_info.current_object->parts, 1);
	__update_circle_part(part, radius_x - OBJECT_BORDER, radius_y - OBJECT_BORDER, center_x, center_y);
}

static void
__move_object(void)
{
	Evas_Object *part = NULL;
	int x = 0, y = 0;
	int offset_x = 0, offset_y = 0;
	Eina_List *l = NULL;
	int draw_area_y;

	if (!s_info.mouse_pressed)
		return;

	if (!s_info.selected) {
		dlog_print(DLOG_INFO, LOG_TAG, "selected == NULL. Exiting...");
		return;

	}

	evas_object_geometry_get(s_info.draw_area, NULL, &draw_area_y, NULL, NULL);

	offset_x = s_info.curr.x - s_info.prev.x;
	offset_y = s_info.curr.y - s_info.prev.y;

	evas_object_geometry_get(s_info.selection_frame, &x, &y, NULL, NULL);

	if (y + offset_y <= draw_area_y) {/* using frame position so we don't have to recalculate drawn objects geometry */
		dlog_print(DLOG_INFO, LOG_TAG, "Moving out of draw area");
		return;
	}

	evas_object_move(s_info.selection_frame, x+offset_x, y+offset_y);

	dlog_print(DLOG_INFO, LOG_TAG, "Moving item: offset: [%d, %d]", offset_x, offset_y);

	EINA_LIST_FOREACH(s_info.selected->parts, l, part)
	{
		if (!part) {
			dlog_print(DLOG_ERROR, LOG_TAG, "part == NULL");
			continue;
		}
		evas_object_geometry_get(part, &x, &y, NULL, NULL);
		evas_object_move(part, x+offset_x, y+offset_y);
	}

}


static void
__tollbar_item_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	mode_type_t mode = (mode_type_t)data;
	__set_toolbar_mode(mode);
}

static void
__toolbar_clear_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	__clear_draw_area();
	elm_toolbar_item_selected_set(ad->icon_data[s_info.mode].item, EINA_TRUE);
}

static void
__colorselector_color_set_cb(void *data, Evas_Object *obj, void *event_info)
{
	color_selector_t cs_num = (color_selector_t)data;
	int r = 0, g = 0, b = 0, a = 0;
	Elm_Object_Item *color_it = (Elm_Object_Item *) event_info;
	if (!color_it) {
		dlog_print(DLOG_ERROR, LOG_TAG, "color_it == NULL");
		return;
	}

	elm_colorselector_palette_item_color_get(color_it, &r, &g, &b, &a);
	evas_color_argb_premul(a, &r, &g, &b);
	dlog_print(DLOG_INFO, LOG_TAG, "NEW COLOR: %d %d %d %d", r, g, b, a);
	__set_color(cs_num, r, g, b, a);
}

static void
__mouse_up_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	Eina_List *l = NULL;
	Eina_List *ll = NULL;
	Evas_Object *part = NULL;
	object_t *object = NULL;

	s_info.mouse_pressed = false;
	__set_selected_item(s_info.current_object);
	s_info.current_object = NULL;
	dlog_print(DLOG_INFO, LOG_TAG, "MOUSE UP CB: %p; Objects count: %d; selecte_item: %p (type: %s)",
			s_info.current_object,
			eina_list_count(s_info.objects),
			s_info.selected,
			evas_object_type_get(__get_first_part_of_selected_object()));

	EINA_LIST_FOREACH(s_info.objects, l, object)
	{
		if (!object) {
			dlog_print(DLOG_ERROR, LOG_TAG, "o == NULL");
			continue;
		}
		dlog_print(DLOG_INFO, LOG_TAG, ">> Object: %d", object->type);

		EINA_LIST_FOREACH(object->parts, ll, part)
		{
			if (!part) {
				dlog_print(DLOG_ERROR, LOG_TAG, "part == NULL");
				continue;
			}
			dlog_print(DLOG_INFO, LOG_TAG, ">>>> %p (%s)", part, evas_object_type_get(part));
		}
	}
}

static void
__mouse_down_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	Evas_Event_Mouse_Down *eemd = (Evas_Event_Mouse_Down *)event_info;
	if (!eemd) {
		dlog_print(DLOG_ERROR, LOG_TAG, "eemd == NULL");
		return;
	}

	s_info.mouse_pressed = true;

	s_info.start.x = eemd->canvas.x;
	s_info.start.y = eemd->canvas.y;

	s_info.prev.x = eemd->canvas.x;
	s_info.prev.y = eemd->canvas.y;

	s_info.curr.x = eemd->canvas.x;
	s_info.curr.y = eemd->canvas.y;

	switch (s_info.mode) {
	case SELECT:
		break;
	case RECTANGLE:
	case LINE:
	case FREEHAND:
	case CIRCLE:
	case CLEAN:
	case COLOR_SELECTOR:
	case REMOVE:
		dlog_print(DLOG_INFO, LOG_TAG, "Mouse down - mode selected: %d", s_info.mode);
		__unselect_item();
		break;
	default:
		dlog_print(DLOG_ERROR, LOG_TAG, "Unknow mode");
		break;
	}
}

static void
__mouse_move_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
	int x = 0, y = 0, w = 0, h = 0;

	Evas_Event_Mouse_Move *eemm = (Evas_Event_Mouse_Move *)event_info;
	if (!eemm) {
		dlog_print(DLOG_ERROR, LOG_TAG, "eemm == NULL");
		return;
	}

	evas_object_geometry_get(obj, &x, &y, &w, &h);

	s_info.prev.x = s_info.curr.x;
	s_info.prev.y = s_info.curr.y;

	if (eemm->cur.canvas.x < x) {
		s_info.curr.x = x;
	} else if (eemm->cur.canvas.x > x + w) {
		s_info.curr.x = x + w;
	} else {
		s_info.curr.x = eemm->cur.canvas.x;
	}

	if (eemm->cur.canvas.y < y) {
		s_info.curr.y = y;
	} else if (eemm->cur.canvas.y > y + h) {
		s_info.curr.y = y + h;
	} else {
		s_info.curr.y = eemm->cur.canvas.y;
	}

	dlog_print(DLOG_INFO, LOG_TAG, "Mouse position: [%d, %d] -> [%d, %d]",
			s_info.prev.x, s_info.prev.y,
			s_info.curr.x, s_info.curr.y);

	switch (s_info.mode) {
	case FREEHAND:
		__update_freehand();
		break;
	case RECTANGLE:
		__update_rect();
		break;
	case CIRCLE:
		___update_circle();
		break;
	case LINE:
		__update_line();
		break;
	case SELECT:
		__move_object();
		break;
	case CLEAN:
	case COLOR_SELECTOR:
	case REMOVE:
		break;
	default:
		dlog_print(DLOG_ERROR, LOG_TAG, "Unknow mode");
		break;
	}
}


static Evas_Object*
__create_app_win(void)
{
	Evas_Object *win = NULL;

	win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	if (!win) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_win_util_standard_add() failed");
		return NULL;
	}

	elm_win_conformant_set(win, EINA_TRUE);
	elm_win_autodel_set(win, EINA_TRUE);

	if (elm_win_wm_rotation_supported_get(win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(win, (const int *) (&rots), 4);
	}

	elm_win_indicator_mode_set(win, ELM_WIN_INDICATOR_HIDE);
	elm_win_indicator_opacity_set(win, ELM_WIN_INDICATOR_OPAQUE);

	evas_object_smart_callback_add(win, "delete,request", __delete_win_request_cb, NULL);
	s_info.win_layer = evas_object_layer_get(win);
	evas_object_show(win);
	s_info.win = win;

	return win;
}

static Evas_Object *
__create_conformant(Evas_Object *win)
{
	Evas_Object *conform = NULL;
	if (!win) {
		dlog_print(DLOG_ERROR, LOG_TAG, "win == NULL");
		return NULL;
	}

	conform = elm_conformant_add(win);
	if (!conform) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_conformant_add() failed");
		return NULL;
	}

	evas_object_size_hint_weight_set(conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(win, conform);
	evas_object_show(conform);

	return conform;
}

static Evas_Object *
__create_layout(Evas_Object *win, Evas_Object *conform, appdata_s *ad)
{
	char path[PATH_MAX];
	Evas_Object *layout = NULL;

	if (!win) {
		dlog_print(DLOG_ERROR, LOG_TAG, "win == NULL");
		return NULL;
	}

	if (!conform) {
		dlog_print(DLOG_ERROR, LOG_TAG, "conform == NULL");
		return NULL;
	}

	__get_app_resource(EDJ_FILE, path, (int)PATH_MAX);

	layout = elm_layout_add(conform);
	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function elm_layout_add() failed");
		return NULL;
	}

	if (!elm_layout_file_set(layout, path, GRP_MAIN)) {
		dlog_print(DLOG_ERROR, LOG_TAG, "function elm_layout_file_set() failed");
		return NULL;
	}

	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	eext_object_event_callback_add(layout, EEXT_CALLBACK_BACK, __key_press_cb, ad);
	evas_object_smart_callback_add(win, "rotation,changed", __win_rotation_cb, ad);
	elm_object_content_set(conform, layout);
	s_info.layout = ad->layout;

	return layout;
}

static Eina_Bool
__create_toolbar(Evas_Object *layout, appdata_s *ad)
{
	Evas_Object *toolbar = NULL;
	int i = 0;

	if (!layout) {
		dlog_print(DLOG_ERROR, LOG_TAG, "layout == NULL");
		return EINA_FALSE;
	}

	toolbar = elm_toolbar_add(layout);
	if (!toolbar) {
		dlog_print(DLOG_ERROR, LOG_TAG, "function elm_layout_file_set() failed");
		return EINA_FALSE;
	}

	elm_toolbar_transverse_expanded_set(toolbar, EINA_TRUE);
	elm_toolbar_homogeneous_set(toolbar, EINA_TRUE);
	elm_toolbar_shrink_mode_set(toolbar, ELM_TOOLBAR_SHRINK_EXPAND);
	elm_toolbar_icon_size_set(toolbar, TOOLBAR_ICON_SIZE);

	__append_toolbar_item(toolbar, &ad->icon_data[0], __toolbar_clear_clicked_cb, (void *)ad);
	for (i = 1; i < IMAGE_POOL_SIZE - 2; ++i)
		__append_toolbar_item(toolbar, &ad->icon_data[i], __tollbar_item_clicked_cb, (void *)ad->icon_data[i].mode);

	__append_toolbar_item(toolbar, &ad->icon_data[IMAGE_POOL_SIZE - 2], __toolbar_colorselector_show_cb, (void *)ad);
	__append_toolbar_item(toolbar, &ad->icon_data[IMAGE_POOL_SIZE - 1], __toolbar_selected_item_del_cb, (void *)ad);

	elm_layout_content_set(layout, PART_TOOLBAR, toolbar);

	return EINA_TRUE;
}

static Evas_Object *
__create_colorselector(Evas_Object *win, color_selector_t cs_num)
{
	Evas_Object *cs = elm_colorselector_add(win);
	if (!cs) {
		dlog_print(DLOG_ERROR, LOG_TAG, "function elm_colorselector_add() failed");
		return NULL;
	}

	evas_object_show(cs);
	evas_object_layer_set(cs, EVAS_LAYER_MAX);
	evas_object_smart_callback_add(cs, COLOR_SELECTOR_COLOR_SELECTED, __colorselector_color_set_cb, (void *)cs_num);

	return cs;
}

static void
 __box_position_set(Evas_Object *win, Evas_Object *box, int win_rotation)
{
	int box_h;
	int win_h;
	int win_w;

	elm_win_screen_size_get(win, NULL, NULL, &win_w, &win_h);
	evas_object_geometry_get(box, NULL, NULL, NULL, &box_h);

	if (win_rotation == 90 || win_rotation == 270) {
		evas_object_move(box, 0,  win_w - box_h);
	} else {
		evas_object_move(box, 0, win_h - box_h);
	}
}

static void
__box_resize_cb(void *data, Evas *e, Evas_Object *box, void *event_info)
{
	int win_rotation = 0;
	Evas_Object* win = (Evas_Object*) data;
	if (!win) {
		dlog_print(DLOG_ERROR, LOG_TAG, "win == NULL");
		return;
	}

	win_rotation = elm_win_rotation_get(win);

	__box_position_set(win, box, win_rotation);
}

static Evas_Object *
__create_color_selector_panel(Evas_Object *win)
{
	Evas_Object *color_selector = NULL;
	Evas_Object *box = NULL;
	if (!win) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Wrong parameters provided: ad == %p; ad->win == %p", win, color_selector);
		return NULL;
	}

	box = elm_box_add(win);
	if (!box) {
		dlog_print(DLOG_ERROR, LOG_TAG, "box == NULL");
		return NULL;
	}

	elm_box_horizontal_set(box, false);

	color_selector = __create_colorselector(box, COLOR_SELECTOR_MAIN);
	elm_box_pack_end(box, color_selector);

	color_selector = __create_colorselector(box, COLOR_SELECTOR_FILL);
	elm_box_pack_end(box, color_selector);
	evas_object_event_callback_add(box, EVAS_CALLBACK_RESIZE, __box_resize_cb, win);

	evas_object_layer_set(box, EVAS_LAYER_MAX);
	evas_object_show(box);

	return box;
}

static Eina_Bool
__create_draw_area(Evas_Object *layout)
{
	if (!s_info.evas) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.evas == NULL");
		return EINA_FALSE;
	}

	s_info.draw_area = evas_object_rectangle_add(s_info.evas);
	if (!s_info.draw_area) {
		dlog_print(DLOG_ERROR, LOG_TAG, "function evas_object_rectangle_add() failed");
		return EINA_FALSE;
	}

	evas_object_size_hint_weight_set(s_info.draw_area, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(s_info.draw_area, EVAS_HINT_FILL, EVAS_HINT_FILL);

	evas_object_event_callback_add(s_info.draw_area, EVAS_CALLBACK_MOUSE_DOWN, __mouse_down_cb, NULL);
	evas_object_event_callback_add(s_info.draw_area, EVAS_CALLBACK_MOUSE_UP, __mouse_up_cb, NULL);
	evas_object_event_callback_add(s_info.draw_area, EVAS_CALLBACK_MOUSE_MOVE, __mouse_move_cb, NULL);

	elm_layout_content_set(layout, PART_DRAW_AREA, s_info.draw_area);

	return EINA_TRUE;
}

static Eina_Bool
__get_win_evas(Evas_Object *win)
{
	s_info.evas = evas_object_evas_get(win);
	if (!s_info.evas) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Function evas_object_evas_get() failed");
		return EINA_FALSE;
	}

	return EINA_TRUE;
}

static Eina_Bool
__create_selection_frame(appdata_s *ad)
{
	char path[PATH_MAX] = {0,};

	if (!s_info.evas) {
		dlog_print(DLOG_ERROR, LOG_TAG, "s_info.evas == NULL");
		return EINA_FALSE;
	}

	s_info.selection_frame = evas_object_image_filled_add(s_info.evas);
	if (!s_info.selection_frame) {
		dlog_print(DLOG_ERROR, LOG_TAG, "function evas_object_image_add() failed");
		return EINA_FALSE;
	}

	__get_app_resource(IMAGE_FRAME, path, (int)PATH_MAX);
	evas_object_image_file_set(s_info.selection_frame, path, NULL);
	evas_object_image_border_set(s_info.selection_frame, FRAME_BORDER, FRAME_BORDER, FRAME_BORDER, FRAME_BORDER);
	evas_object_layer_set(s_info.selection_frame, EVAS_LAYER_MAX - 1);
	evas_object_repeat_events_set(s_info.selection_frame, EINA_TRUE);

	return EINA_TRUE;
}
