#include <sensor.h>
#include "acceleration_sensor.h"
#include "$(appName).h"
#include "view.h"

struct acdata {
	sensor_listener_h acceleration_listener;
	double prev_acc_av;
	double init_acc_av;
	int steps_count;
} s_acc_data = {
	.acceleration_listener = NULL,
	.prev_acc_av = MAX_ACCEL_INIT_VALUE,
	.init_acc_av = MAX_ACCEL_INIT_VALUE,
	.steps_count = 0
};

static void
__accel_cb(sensor_h sensor, sensor_event_s *event, void *data)
{
	/* Get current average value of acceleration on three axes */
	double current_acc_av = (event->values[1] +
								event->values[2] +
								event->values[3]) / 3;

	/* If initial average acceleration value is not set, do it now */
	if (fabs(s_acc_data.init_acc_av - MAX_ACCEL_INIT_VALUE) < DOUBLE_COMPARIZON_THRESHOLD) {
		s_acc_data.init_acc_av = s_acc_data.prev_acc_av = current_acc_av;
		return;
	}

	/* Register a drop of acceleration average value */
	if ((s_acc_data.prev_acc_av > s_acc_data.init_acc_av &&
		s_acc_data.init_acc_av - current_acc_av > TRESHOLD))
		view_set_steps_count(++s_acc_data.steps_count);

	s_acc_data.prev_acc_av = current_acc_av;
}

int
acceleration_sensor_get_steps_count(void)
{
	return s_acc_data.steps_count;
}

bool
acceleration_sensor_init_handle(void)
{
	sensor_h sensor;
	bool supported = false;

	int ret = sensor_is_supported(SENSOR_ACCELEROMETER, &supported);
	if (ret != SENSOR_ERROR_NONE || !supported) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Accelerometer sensor not supported on current device");
		return false;
	}

	ret = sensor_get_default_sensor(SENSOR_ACCELEROMETER, &sensor);
	if (ret != SENSOR_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to get default accelerometer sensor");
		return false;
	}

	ret = sensor_create_listener(sensor, &s_acc_data.acceleration_listener);
	if (ret != SENSOR_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to create accelerometer sensor");
		return false;
	}

	ret = sensor_listener_set_event_cb(s_acc_data.acceleration_listener,
										200,
										__accel_cb,
										NULL);
	if (ret != SENSOR_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to set event callback for sensor listener");
		sensor_destroy_listener(s_acc_data.acceleration_listener);
		s_acc_data.acceleration_listener = NULL;
		return false;
	}

	ret = sensor_listener_set_option(s_acc_data.acceleration_listener,
										SENSOR_OPTION_ALWAYS_ON);
	if (ret != SENSOR_ERROR_NONE)
		dlog_print(DLOG_ERROR, LOG_TAG, "Failed to set sensor's always on option");

	return true;
}

void
acceleration_sensor_release_handle(void)
{
	if (s_acc_data.acceleration_listener) {
		int ret = sensor_destroy_listener(s_acc_data.acceleration_listener);
		if (ret != SENSOR_ERROR_NONE)
			dlog_print(DLOG_ERROR, LOG_TAG, "Failed to destroy accelerometer sensor listener");
		s_acc_data.acceleration_listener = NULL;
	}
}

void
acceleration_sensor_start(void)
{
	if (s_acc_data.acceleration_listener) {
		int ret = sensor_listener_start(s_acc_data.acceleration_listener);
		if (ret != SENSOR_ERROR_NONE)
			dlog_print(DLOG_ERROR, LOG_TAG, "Failed to start accelerometer sensor listener");
	}
}

void
acceleration_sensor_stop(void)
{
	if (s_acc_data.acceleration_listener) {
		int ret = sensor_listener_stop(s_acc_data.acceleration_listener);
		if (ret != SENSOR_ERROR_NONE) {
			dlog_print(DLOG_ERROR, LOG_TAG, "Failed to stop accelerometer sensor listener");
			return;
		}

		/* Reset init/prev acceleration data */
		s_acc_data.init_acc_av = s_acc_data.prev_acc_av = MAX_ACCEL_INIT_VALUE;
	}
}
